#ifndef URLAN_H
#define URLAN_H
/*
    Urlan
    Copyright (C) 2005-2006  Karl Robillard

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <stdint.h>


/*
   The following namespaces are used:

   UR_*  Constants
   UT_*  Data type constants
   U*    Structures and types
   ur_*  Functions, variables, macros
*/


#define UR_VERSION_STR  "0.0.1"
#define UR_VERSION      0x000001


/* Datatypes */

#define UT_UNSET        0
#define UT_DATATYPE     1
#define UT_NONE         2
#define UT_LOGIC        3
#define UT_WORD         4
#define UT_SETWORD      5
#define UT_GETWORD      6
#define UT_LITWORD      7
#define UT_SELECT       8
#define UT_SETSELECT    9
#define UT_OPCODE       10
#define UT_HASH_ID      11
#define UT_CHAR         12
#define UT_INT          13
#define UT_INT64        14      /* UT_BIGNUM? */
#define UT_DECIMAL      15
#define UT_COORD        16
#define UT_VEC3         17
#define UT_BINARY       18
#define UT_STRING       19
#define UT_BLOCK        20
#define UT_PAREN        21
#define UT_PATH         22
#define UT_SETPATH      23
#define UT_SLICE        24
#define UT_ARRAY        25
#define UT_BITSET       26
#define UT_LIST         27
#define UT_CONTEXT      28
#define UT_FUNCTION     29
#define UT_CALL         30
#define UT_DATE         31
#define UT_TIME         32
#define UT_ERROR        33
#define UT_CODE         34
#define UT_PORT         35
#define UT_STRUCT       36
#define UT_MATRIX       37

#define UT_BI_COUNT     38
#define UT_MAX          64
#define UT_TYPEMASK     99


/* Cell flags */

//#define UR_FLAG_PATH_DOT        0x01
#define UR_FLAG_INT_HEX         0x01
#define UR_FLAG_CTX_RECURSION   0x01
#define UR_FLAG_SEL_ATOM        0x01
#define UR_FLAG_FUNC_LOOP       0x01
#define UR_FLAG_EOL             0x80


/* String Encodings */

#define UR_ENC_ASCII    0
#define UR_ENC_UTF8     1
#define UR_ENC_UTF16    2

#define UR_ENC_COUNT    3


/* Eval/call return */

#define UR_EVAL_OK      0
#define UR_EVAL_HALT    1
#define UR_EVAL_QUIT    2
#define UR_EVAL_ERROR   3


/* Access Query */

#define UR_ALLOW_NONE   0
#define UR_ALLOW_READ   1
#define UR_ALLOW_ALL    2


/* Exception types */

#define UR_EX_DATATYPE  0
#define UR_EX_SCRIPT    1
#define UR_EX_SYNTAX    2
#define UR_EX_ACCESS    3
#define UR_EX_INTERNAL  4

#define UR_ERR_DATATYPE     ur_thread, UR_EX_DATATYPE
#define UR_ERR_SCRIPT       ur_thread, UR_EX_SCRIPT
#define UR_ERR_SYNTAX       ur_thread, UR_EX_SYNTAX
#define UR_ERR_ACCESS       ur_thread, UR_EX_ACCESS
#define UR_ERR_INTERNAL     ur_thread, UR_EX_INTERNAL


/* Evaluation Monitor Hook Message */

#define UR_EMH_STEP     1
#define UR_EMH_HALT     2


#define UR_UNBOUND          -1
#define UR_INVALID_HOLD     -1
#define UR_COPY_ALL         0x7fffffff


typedef int32_t     UIndex;
typedef int16_t     UAtom;


typedef struct
{
    uint8_t  type;
    uint8_t  flags;
    uint16_t _pad0;
}
UCellId;


typedef struct
{
    UCellId  id;
    uint16_t n;
    uint16_t bitCount;
    uint32_t mask0;     /* LIMIT: Maximum of 64 datatypes. */
    uint32_t mask1;
}
UCellDatatype;


/*
  UCellWord is used for word!, lit-word!, set-word!, get-word!,
  opcode! & selector!
*/
typedef struct
{
    uint8_t  type;
    uint8_t  flags;
    UAtom    atom;
    UIndex   wordBlk;   /* wordBlk at same position as in context. */
    UIndex   valBlk;    /* valBlk  at same position as in context. */
    int16_t  index;     /* index & sel are adjacent for _orderWords. */
    UAtom    sel;
}
UCellWord;


/*
  UCellContext is used for context! & port!
*/
typedef struct
{
    uint8_t  type;
    uint8_t  flags;
    uint16_t deviceId;
    UIndex   wordBlk;
    UIndex   valBlk;
    UIndex   protoValBlk;
}
UCellContext;


typedef struct
{
    uint8_t  type;
    uint8_t  flags;
    uint8_t  localArgs;     /* localArgs, localVars, fetch & sigN are unused */
    uint8_t  localVars;     /* by Thune.  Other dialects may use them. */
    int8_t   fetch;
    uint8_t  _pad;
    uint16_t sigN;
    void (*addr)(void*,void*);  /* On 8 byte boundary for 64-bit systems. */
}
UCellCall;


typedef struct
{
    uint8_t  type;
    uint8_t  flags;
    uint8_t  localArgs;
    uint8_t  localVars;
    UIndex   bodyN;
    UIndex   closureN;
    UIndex   sigN;
}
UCellFunction;


/* UCellNumber is used for int!, decimal!, char! & logic! */
typedef struct
{
    UCellId id;
    int32_t i;
    double  d;              /* On 8 byte boundary. */
}
UCellNumber;


typedef struct
{
    UCellId id;
    int32_t _pad1;
    int64_t i;              /* On 8 byte boundary. */
}
UCellInt64;


typedef struct
{
    UCellId     id;
    uint8_t     day;
    uint8_t     month;
    uint16_t    year;
    double      sec;
}
UCellTime;


/* UCellSeries is used for binary!, string!, block!, paren!, slice! & array! */
typedef struct
{
    uint8_t  type;
    uint8_t  flags;
    uint8_t  encoding;
    uint8_t  datatype;
    UIndex  n;      /* Series number */
    UIndex  it;     /* Element Iterator */
    UIndex  end;    /* Slice end */
}
UCellSeries;   // Series, set, sequence, array, chain, group, list


#define UR_COORD_MAX    6

typedef struct
{
    uint8_t  type;
    uint8_t  flags;
    uint16_t len;
    int16_t  elem[ UR_COORD_MAX ];
}
UCellCoord;


typedef struct
{
    UCellId     id;
    uint16_t    poolN;      /* LIMIT: Maximum of 65535 pair pools. */
    uint16_t    pairN;
    void*       ptr;
}
UCellPair;


/*
typedef struct
{
    UCellWord word;
    int16_t   nodes[8];
}
UPairPath;
*/


typedef struct
{
    uint8_t  type;
    uint8_t  flags;
    uint16_t linked;
    int32_t  prev;
    int32_t  next;
    int32_t  free;
}
UCellList;


typedef struct
{
    UCellId id;
    float xyz[3];
}
UCellVec3;


typedef struct
{
    uint8_t  type;
    uint8_t  flags;
    uint16_t exType;
    UIndex messageStr;
    UIndex traceBlk;
}
UCellError;


typedef struct
{
    uint8_t  type;
    uint8_t  flags;
    uint8_t  langId;
    uint8_t  _pad;
    UIndex   n;
}
UCellCode;


typedef union
{
    UCellId       id;
    UCellDatatype dt;
    UCellWord     word;
    UCellContext  ctx;
    UCellContext  port;
    UCellCall     call;
    UCellFunction func;
    UCellNumber   num;
    UCellInt64    int64;
    UCellTime     time;
    UCellSeries   series;
    UCellSeries   slice;
    UCellCoord    coord;
    UCellPair     slist;
    UCellList     list;
    UCellVec3     vec3;
    UCellError    err;
    UCellCode     code;
}
UCell;


typedef UCell   UContext;


typedef struct
{
    int32_t     avail;
    int32_t     used;
    union
    {
        void*       v;
        UCell*      cells;
        char*       c;
        uint8_t*    b;
        uint16_t*   u16;
        int32_t*    i;
        double*     d;
        float*      f;
    } ptr;
}
UArray;


typedef UArray      UBlock;     // Array of UCell
typedef UArray      UBinary;    // Array of uint8_t
typedef UArray      UString;    // Array of char


typedef struct
{
    UArray      arr;
    uint32_t    freeCount;
    UIndex      freeList;
    UIndex      sweepStart;
}
UGCArray;


typedef struct
{
    uint8_t     dataType;
    uint8_t     flags;
    uint16_t    user;
    UIndex      link;
    void*       ptr;
}
UBuffer;


typedef struct
{
    uint16_t    dataType;
    uint16_t    _pad;
    UIndex      which;
}
UHold;


typedef struct
{
    UBinary bsBin;
    UBinary bsBlock;
    UBinary bsBuf;
    UBinary blkChecked;
}
UCollector;


#ifdef UR_CONFIG_UDS
#define UR_DSTACK_SIZE       4
#define UR_CSTACK_SIZE       2
#else
#define UR_DSTACK_SIZE       256
#define UR_CSTACK_SIZE       128
#endif

#define UR_THREAD_RUNNING    1
#define UR_THREAD_READY      2
#define UR_THREAD_BLOCKED    3
#define UR_THREAD_TERM       4

typedef struct UrlanEnv  UrlanEnv;

typedef struct
{
    UrlanEnv* env;
    void*   nextThread; // UThread*
    int     state;
    int     callOp;
    UIndex  callTempBinN;
    UIndex  callTempBinHold;
    UCell*  tos;
    UCell*  eos;
    UCell*  toc;        // CEntry*
    UCell*  localFT;    // CEntry*
    UCell   dstack[ UR_DSTACK_SIZE ];
    UCell   cstack[ UR_CSTACK_SIZE ];
    // Can use separate exception stack?
}
UThread;


typedef struct UPortDevice  UPortDevice;

struct UPortDevice
{
    UAtom    name;
    uint16_t deviceId;

    int  (*open) ( UPortDevice*, UThread*, UCell* );
    void (*close)( UThread*, UCell* );
    void (*read) ( UThread*, UCell* );
    void (*write)( UThread*, UCell* );
    void (*seek) ( UThread*, UCell* );
};


typedef struct
{
    UAtom    name;
    uint16_t langId;

    void (*compile)( UThread*, UCell* );
    void (*run)( UThread*, UCell* );
    void (*codeGC)( UCollector*, UCell* );
}
ULanguage;


typedef  void (*UCall)(UThread*, UCell* tos);

//#define UR_DT_FLAG_SERIES  0x0001
#define UR_GC_PHASE_MARK   0 
#define UR_GC_PHASE_SWEEP  1 

typedef struct
{
    const char* name;
    int   id;
    int   flags;

    UCall make;
    UCall to;
    void (*toString)( const UCell*, UString*, int depth );
    void (*toText)( const UCell*, UString*, int depth );
    int  (*selectAtom)( UThread*, UCell* val, const UCell* sel, UCell* res );
    void (*recycle)( UrlanEnv*, int phase );
    void (*gcMark)( UCollector*, UCell* );
    void (*gcDestroyBuffer)( UBuffer* );
}
UDatatype;


#define UR_ENV_GC           0x01
#define UR_ENV_SECURE       0x02

struct UrlanEnv
{
    uint16_t    flags;
    uint16_t    dtCount;
    UArray      atoms;
    UGCArray    bin;
    UGCArray    blocks;
    UGCArray    buffers;
    UGCArray    holds;
    UThread*    threads;
    UDatatype*  customDT;

#ifndef UR_CONFIG_UDS
    UArray      devices;
#ifdef UR_CONFIG_DT_CODE
    UArray      languages;
#endif
#ifdef UR_CONFIG_EMH
    int (*monitor)(int,UCell*,UCell*);
#endif
#endif
};


typedef struct
{
    void (*addr)(UThread*, UCell*);
    const char* name;
}
UCallDef;


#ifdef __cplusplus
extern "C" {
#endif


extern UrlanEnv*    ur_env;
extern UContext     ur_global;
extern int          ur_encodingCharSize[UR_ENC_COUNT];

int     ur_startup( UrlanEnv*, UDatatype* custom, int customCount );
void    ur_shutdown( UrlanEnv* );
void    ur_enable( UrlanEnv*, int envFlags );
void    ur_disable( UrlanEnv*, int envFlags );
UThread* ur_threadMake( UrlanEnv* );
void     ur_threadFree( UThread* );
void     ur_threadReset( UThread* );
uint32_t ur_hash( const char* str, const char* end );
uint32_t ur_atomHash( UrlanEnv*, UAtom );
const char* ur_typeName( int type );
const char* ur_atomCStr( UAtom, int* plen );
const char* ur_cstring( UCell* );
void    ur_atomStr( UAtom atom, UString* str );
UAtom   ur_intern( UrlanEnv*, const char* str, int len );
int     ur_lookup( const UContext* ctx, UAtom );
int     ur_internWord( const UContext*, UAtom );
UBlock* ur_bind( UIndex blkN, UContext* );
void    ur_infuse( UThread*, UBlock*, const UContext* );
int     ur_eval( UThread*, UIndex blkN, UIndex si );
int     ur_evalCStr( UThread*, const char* cmd, int len );
UCell*  ur_wordCell( UThread*, const UCell* );
UCell*  ur_result( UThread*, int pop );
UCell*  ur_resolveArgPath( int, ... );
void    ur_recycle( UrlanEnv* );
void    ur_gcMarkBin( UCollector*, UIndex );
void    ur_gcMarkBlock( UCollector*, UIndex );
void    ur_gcMarkBuffer( UCollector*, UIndex );
UIndex  ur_tokenize( UThread*, const char* it, const char* end, void* );
void    ur_throwErr( UThread*, int exceptionType, const char* fmt, ... );
int     ur_userAllows( UThread*, const char* fmt, ... );

void    ur_registerPortDevice( UrlanEnv*, const char* name,
                               UPortDevice*, const char* );
#ifdef UR_CONFIG_DT_CODE
void    ur_registerLanguage( UrlanEnv*, const char* name, ULanguage* );
#endif

UCell*  ur_makePort( UPortDevice*, UCell* res );
UIndex  ur_makeBuffer( int dataType, int size, UBuffer** );
UIndex  ur_makeBlock( int size );
UIndex  ur_makeBinary( int size );
UIndex  ur_makeBinaryFrom( const UCell* );
UIndex  ur_makeString( const char* txt, int len );
void    ur_makeContext( UContext* ctx, int wordCount );
int     ur_makeSelector( UCell*, const char* spA, const char* end );
void    ur_makeCalls( UrlanEnv*, const UCallDef*, int count );
void    ur_clone( UCell*, UIndex part, int deep );
void    ur_copyCells( const UCell* src, const UCell* srcEnd, UCell* dest );

UCell*  ur_appendCell( UBlock*, int type );
UCell*  ur_appendWord( UBlock*, int type, const char* name, int len );

void ur_setUnbound( UCell*, UAtom atom );

void ur_toStr( const UCell*, UString* out, int depth );
void ur_toStrNatural( const UCell*, UString* out, int depth );
void ur_strCat( UString*, const char* cp, int len );

void ur_setDatatypeMask( UCell* cell, int type );
void ur_setDatatypeBit( UCell* cell, int type );
void ur_clrDatatypeBit( UCell* cell, int type );
int  ur_verifyDatatype( const UCell* cell, const UCell* type );

UIndex ur_sliceEnd( const UCell*, const UArray* );
UString* ur_stringSlice( const UCell*, char** cpA, char** cpB );
UBinary* ur_binarySlice( const UCell*, char** cpA, char** cpB );
UBlock*  ur_blockSlice( const UCell*, UCell** cpA, UCell** cpB );

int  ur_equal( const UCell*, const UCell* );
int  ur_same( const UCell*, const UCell* );
int  ur_seriesLen( const UCell* );
int  ur_pick( const UCell*, int n, UCell* res );
int  ur_poke( UCell*, int n, const UCell* val );

void ur_arrayInit( UArray*, int elemSize, int elemCount );
void ur_arrayFree( UArray* );
void ur_arrayReserve( UArray*, int elemSize, int elemCount );
void ur_arrayErase( UArray*, int elemSize, int index, int count );

UIndex ur_hold( UrlanEnv*, int type, UIndex which );
UIndex ur_holdIndex( UrlanEnv*, UIndex );
void   ur_release( UrlanEnv*, UIndex );

#ifdef __cplusplus
}
#endif


#define ur_bitIsSet(array,n)    (array[(n)>>3] & 1<<((n)&7))
#define ur_setBit(array,n)      (array[(n)>>3] |= 1<<((n)&7))
#define ur_clrBit(array,n)      (array[(n)>>3] &= ~(1<<((n)&7)))

#define ur_blockPtr(n)      (((UBlock*) ur_env->blocks.arr.ptr.v) + n)
#define ur_binPtr(n)        (((UArray*) ur_env->bin.arr.ptr.v) + n)
#define ur_bufPtr(n)        (((UBuffer*)ur_env->buffers.arr.ptr.v) + n)

#define ur_blockN(bp)       (bp - ((UBlock*) ur_env->blocks.arr.ptr.v))
#define ur_binN(bp)         (bp - ((UArray*) ur_env->bin.arr.ptr.v))
#define ur_bufN(bp)         (bp - ((UBuffer*)ur_env->buffers.arr.ptr.v))

#define ur_block(c)         ur_blockPtr( (c)->series.n )
#define ur_bin(c)           ur_binPtr( (c)->series.n )

#define ur_type(c)          (c)->id.type
#ifdef __BIG_ENDIAN__
#define ur_initType(c,t)    *((uint32_t*) (c)) = t << 24
#else
#define ur_initType(c,t)    *((uint32_t*) (c)) = t
#endif
#define ur_setFlags(c,m)    (c)->id.flags |= m
#define ur_is(c,t)          ((c)->id.type == (t))

#define ur_isAWord(c) \
    ((c->id.type) >= UT_WORD && (c->id.type) <= UT_LITWORD)

#define ur_isASeries(c) \
    ((c->id.type) >= UT_BINARY && (c->id.type) <= UT_ARRAY)

#define ur_isABlock(c) \
    ((c->id.type) >= UT_BLOCK && (c->id.type) <= UT_SETPATH)

#define ur_setSeries(c,sn,sit) \
    (c)->series.n  = sn; \
    (c)->series.it = sit

#define ur_setEncoding(c,en)    (c)->series.encoding = en
#define ur_encoding(c)          (c)->series.encoding
#define ur_encCharSize(c)       ur_encodingCharSize[(c)->series.encoding]

#define ur_atom(c)          (c)->word.atom
#define ur_datatype(c)      (c)->dt.n
#define ur_logic(c)         (c)->num.i
#define ur_opcode(c)        (c)->word.index
#define ur_char(c)          (c)->num.i
#define ur_int(c)           (c)->num.i
#define ur_int64(c)         (c)->int64.i
#define ur_decimal(c)       (c)->num.d
#define ur_seconds(c)       (c)->time.sec
#define ur_pair(c)          ((UCell*) (c)->slist.ptr)

#define ur_setNone(c)       ur_initType(c,UT_NONE)
#define ur_setTrue(c)       ur_initType(c,UT_LOGIC); ur_logic(c) = 1
#define ur_setFalse(c)      ur_initType(c,UT_LOGIC); ur_logic(c) = 0

/*
#define ur_pathElem(c,n)    (c)->path.elem[n]
#define ur_pathIsAtom(c,n)  ((c)->path.elemType & (1 << n))
#define ur_pathHasAtoms(c)  (c)->path.elemType
*/

#define ur_sliceDT(c)       (c)->slice.datatype

#define ur_arrayDT(c)       (c)->series.datatype
#define ur_arrayStride(c)   (c)->series.encoding

#define ur_sel(c)           (c)->word.sel
#define ur_selIsAtom(c)     ((c)->word.flags & UR_FLAG_SEL_ATOM)

#define ur_wordIsUnbound(pc)    (pc->word.index < 0)

/*
#define ur_wordCell(pc,blk,val) \
    blk = ur_blockPtr( pc->word.valBlk ); \
    val = blk->ptr.cells + pc->word.index
*/

#define ur_copyCell(dst,src)    *(dst) = src;

#define ur_termCStr(s) \
    if(s->used == s->avail) \
        ur_arrayReserve(s,sizeof(char),s->used+1); \
    s->ptr.c[s->used] = '\0'

#define UR_CALL(func)       static void func(UThread* ur_thread, UCell* tos)
#define UR_CALL_PUB(func)   void func(UThread* ur_thread, UCell* tos)
#define UR_CALL_UNUSED_TH   (void) ur_thread;
#define UR_CALL_UNUSED_TOS  (void) tos;
#define UR_INTERN(s,l)      ur_intern(ur_thread->env,s,l)

#define UR_ITER_BLOCK(ita,itb,blk,scell) \
    ita = blk->ptr.cells + scell->series.it; \
    itb = blk->ptr.cells + blk->used;

#define UR_ITER_BIN(ita,itb,bin,scell) \
    ita = bin->ptr.b + scell->series.it; \
    itb = bin->ptr.b + bin->used;

#define UR_ITER_STR(ita,itb,str,scell) \
    ita = str->ptr.c + scell->series.it; \
    itb = str->ptr.c + str->used;

#define ur_language(cell) \
    (((ULanguage**) ur_env->languages.ptr.v)[(cell)->code.langId])

//#define ur_error            ur_env->error


/* Data Stack */

#define UR_TOS              ur_thread->tos
#define UR_BOS              ur_thread->dstack
#define UR_S_PUSH(v)        *++UR_TOS = v
#define UR_S_GROW           ++UR_TOS
#define UR_S_GROWN(n)       UR_TOS += n
#define UR_S_DROP           --UR_TOS
#define UR_S_DROPN(n)       UR_TOS -= n
#define UR_S_SAFE_DROP  	if(UR_TOS != UR_BOS) UR_S_DROP
#define UR_S_DUP            UR_TOS[1] = *UR_TOS; ++UR_TOS
#define UR_S_NIP            --UR_TOS; *UR_TOS = UR_TOS[1]
#define ur_s_atBottom(sp)   (sp == ur_thread->dstack)
#define ur_s_notBottom(sp)  (sp != ur_thread->dstack)
#define ur_s_prev(v)        (v - 1)
#define ur_s_next(v)        (v + 1)
#define ur_s_backN(v,n)     (v - (n))
#define ur_s_aheadN(v,n)    (v + (n))


#endif  /*EOF*/
