// oCADis (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <gui/checkbox.h>
#include "tool_copy_state_2.h"
#include "common.h"
#include "toolbox_button.h"
#include "toolbox_point.h"
#include "toolbox_frame.h"
#include "toolbox_action.h"
#include "tableview.h"

using namespace os;

///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E
//
///////////////////////////////////////////////////////////////////////////////

class CopyState2 :: Private
{
public:
	Private(){};

	void _Layout( Rect cFrame )
	{
		m_Table->SetFrame( Rect( 0, 0, cFrame.Width(), cFrame.Height() ) );
	}

	enum CopyState2Message { MSG_POINT, MSG_CANCEL, MSG_MULTICOPIES };

	CheckBox* m_MultipleCopies;
	
	ToolboxPoint* m_NextPoint;
	ToolboxFrame* m_Settings;
	ToolboxAction* m_Action;

	TableView* m_Table;
	TableView* m_SettingsTable;

	Point m_BasePoint;
};

///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S
//
///////////////////////////////////////////////////////////////////////////////

CopyState2 :: CopyState2( Tool* theTool, ToolboxSideBar* toolbox ) : ToolState( theTool, toolbox )
{
	// Create the private class
	m = new Private();

	// Create the table
	m->m_Table = new TableView( Rect(), "", 1, 3, false );
	m->m_Table->SetRowSpacings( 4 );
	AddChild( m->m_Table );

	// Create table used in the Settings frame
	m->m_SettingsTable = new TableView( Rect(), "", 1, 1, false );

	// Create settings
	m->m_MultipleCopies = new CheckBox( Rect(), "", "Multiple copies", new Message( Private::MSG_MULTICOPIES ) );
	m->m_SettingsTable->Attach( m->m_MultipleCopies, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );
	
	// Create buttons and attach them to table
	m->m_NextPoint = new ToolboxPoint( Rect(), "Second point", "X: ", "Y: ", new Message( Private::MSG_POINT ), true );
	m->m_Table->Attach( m->m_NextPoint, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m->m_Settings = new ToolboxFrame( Rect( 0, 0, 100, 100 ), "", "Settings" );
	m->m_Settings->AddTableView( m->m_SettingsTable );
	m->m_Table->Attach( m->m_Settings, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m->m_Action = new ToolboxAction( Rect(), "" );
	m->m_Table->Attach( m->m_Action, 0, 1, 2, 3, ( TABLE_EXPAND | TABLE_FILL ), ( TABLE_EXPAND | TABLE_FILL ) );

	// Add action items
	m->m_Action->AddButton( "Cancel", (Image*) LoadImage( "circle.png" ), new Message( Private::MSG_CANCEL ) , "Cancel the tool and return to main toolbox " );

	State::SetToolbox( this );
}

CopyState2 :: ~CopyState2()
{
	delete m;
}

void CopyState2 :: Init( Canvas* canvas, Document* doc )
{
	m->m_BasePoint = GetTool()->GetPropertyAsPoint( "base_point" );
	m->m_NextPoint->SetOrigin( m->m_BasePoint );
}

void CopyState2 :: GotFocus( Canvas* canvas, Document* doc, Point coord )
{
	m->m_NextPoint->SetValue( coord );
}

void CopyState2 :: LostFocus( Canvas* canvas, Document* doc, Point coord )
{
	m->m_NextPoint->ClearValue();
}

void CopyState2 :: MouseMove( Canvas* canvas, Document* doc, Point coord )
{
	m->m_NextPoint->SetValue( coord );

	canvas->SetDefaultStyle();
	canvas->DrawLine( m->m_BasePoint, coord, true );

	Transform transform;
	transform.Translation( coord - m->m_BasePoint );

	// Draw objects
	canvas->DrawSelectedObjects( transform );
}

String CopyState2 :: GetStatusText()
{
	return String( "Select second point of displacement" );
}

void CopyState2 :: PointSelected( Canvas* canvas, Document* doc, Point coord )
{
	canvas->Undraw();

	Transform transform;
	transform.Translation( coord - m->m_BasePoint );

	// Create Undo object
	UndoAddObject* undo = new UndoAddObject( "Copy objects");

	for( int i = 0 ; i < doc->GetNoSelectedObjects() ; i++ )
	{
		Object* obj = doc->GetSelectedObject( i );

		if( !m->m_MultipleCopies->GetValue().AsBool() )
		{
			obj->SetSelected( false );
			obj->Draw( canvas );
		}

		Object* new_obj = obj->Duplicate();
		new_obj->SetSelected( false );
		new_obj->DoTransform( transform );
		new_obj->Draw( canvas );
		doc->AddObject( obj->GetLayer(), new_obj ); 	
		undo->AddObject( new_obj );
	}

	doc->AddUndoAction( undo );
	doc->SetModification();

	if( m->m_MultipleCopies->GetValue().AsBool() )
	{
		canvas->PrepareForXORMode();
		GetTool()->SetState( 2 );	
	}
	else
		SetMainToolbox();
}


void CopyState2 :: SetFrame( const Rect& cRect, bool bNotifyServer )
{
	Toolbox::SetFrame( cRect, bNotifyServer );
	m->_Layout( GetFrame() );
}

void CopyState2 :: AllAttached( void )
{
	Toolbox::AllAttached();

	m->m_NextPoint->SetTarget( this );
	m->m_Action->SetTarget( this );
}

void CopyState2 :: HandleMessage( Canvas* canvas, Document* doc, Message* pcMessage )
{
	switch( pcMessage->GetCode() )
	{	
	case Private::MSG_CANCEL:
		SetMainToolbox();
		break;
	case Private::MSG_POINT:
		{
			Point p;

			if( pcMessage->FindPoint( "value", &p ) == EOK )
				PointSelected( canvas, doc, p );
			break;
		}
	case Private::MSG_MULTICOPIES:
		break;
	}
}

