// oCADis (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <math.h>

#include "tool_circle_state_1.h"
#include "common.h"
#include "main.h"
#include "object_circle.h"

using namespace os;

///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E
//
///////////////////////////////////////////////////////////////////////////////
class CircleState1 :: _Private
{
public:
	_Private(){};

	void _Layout( Rect cFrame )
	{
		m_Table->SetFrame( Rect( 0, 0, cFrame.Width(), cFrame.Height() ) );
	}

	float _CalculateRadius( Point p2, Point p1 )
	{
		Point dist = p1 - p2;
		float d = dist.x * dist.x + dist.y * dist.y;

		return sqrt( d );
	}

	void _CreateObject( Tool* tool, Canvas* canvas, Document* doc, Point center, float radius )
	{
		canvas->Undraw();

		// Create object
		ObjectCircle* obj = new ObjectCircle( center, radius );
		doc->AddObject( obj );
		obj->Draw( canvas );

		// Create Undo object
		UndoAddObject* undo = new UndoAddObject( "Create Circle", obj );
		doc->AddUndoAction( undo );
		doc->SetModification();

		tool->SetState( 0 );
	}

	enum CircleState1Message { MSG_CLOSE, MSG_POINT, MSG_DIAMETER, MSG_CANCEL };

	ToolboxValue* m_RadiusValue;
	ToolboxAction* m_Action;
	ToolboxNavigation* m_Navigation;

	os::TableView* m_Table;
};

///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S
//
///////////////////////////////////////////////////////////////////////////////
CircleState1 :: CircleState1( ToolCircle* theTool, ToolboxSideBar* toolbox ) : ToolState( theTool, toolbox )
{
	// Create the private class
	m = new _Private();

	// Create the table
	m->m_Table = new TableView( Rect(), "", 1, 2, false );
	m->m_Table->SetRowSpacings( 4 );
	AddChild( m->m_Table );

	// Create buttons and attach them to table
	m->m_RadiusValue = new ToolboxValue( Rect(), "Radius", "", new Message( _Private::MSG_POINT ) );
	m->m_Table->Attach( m->m_RadiusValue, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m->m_Action = new ToolboxAction( Rect(), "" );
	m->m_Table->Attach( m->m_Action, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), ( TABLE_EXPAND | TABLE_FILL ) );

	// Add action items
	m->m_Action->AddButton( "Cancel", (Image*) LoadImage( "cancel.png" ), new Message( _Private::MSG_CANCEL ) , "Cancel the tool and return to main toolbox " );
	m->m_Action->AddButton( "Diameter", (Image*) LoadImage( "circle.png" ), new Message( _Private::MSG_DIAMETER ), "Describe a circle using diameter " );

	State::SetToolbox( this );
}

CircleState1 :: ~CircleState1()
{
	delete m;
}

void CircleState1 :: Init( Canvas* canvas, Document* doc )
{
	canvas->PrepareForXORMode();
}

void CircleState1 :: GotFocus( Canvas* canvas, Document* doc, Point coord )
{
	coord = doc->MakeSnap( coord );
	m->m_RadiusValue->SetValue( m->_CalculateRadius( GetTool()->GetPropertyAsPoint( "center_point" ), coord ) );
}

void CircleState1 :: LostFocus( Canvas* canvas, Document* doc, Point coord )
{
	m->m_RadiusValue->ClearValue();
}

void CircleState1 :: MouseMove( Canvas* canvas, Document* doc, Point coord )
{
	coord = doc->MakeSnap( coord );

	canvas->SetStyle( doc->GetCurrentStyle() );

	// Calculate radius
	float r = m->_CalculateRadius( GetTool()->GetPropertyAsPoint( "center_point" ), coord );
	m->m_RadiusValue->SetValue( r );
	canvas->DrawCircle( GetTool()->GetPropertyAsPoint( "center_point" ), r, true );
}

String CircleState1 :: GetStatusText()
{
	return String( "Select radius of the circle" );
}

void CircleState1 :: PointSelected( Canvas* canvas, Document* doc, Point coord )
{
	coord = doc->MakeSnap( coord );
	m->_CreateObject( GetTool(), canvas, doc, GetTool()->GetPropertyAsPoint( "center_point" ), m->_CalculateRadius( GetTool()->GetPropertyAsPoint( "center_point" ), coord ) );
}


void CircleState1 :: SetFrame( const Rect& cRect, bool bNotifyServer )
{
	Toolbox::SetFrame( cRect, bNotifyServer );
	m->_Layout( cRect );
}

void CircleState1 :: AllAttached( void )
{
	Toolbox::AllAttached();

	m->m_Action->SetTarget( this );
	m->m_RadiusValue->SetTarget( this );
}

void CircleState1 :: HandleMessage( Canvas* canvas, Document* doc, Message* pcMessage )
{
	switch( pcMessage->GetCode() )
	{	
		case _Private::MSG_CANCEL:
			SetMainToolbox();
			break;
		case _Private::MSG_DIAMETER:
		{
			GetTool()->SetState( 2 );
			break;
		}

		case _Private::MSG_POINT:
		{
			float p;

			if( pcMessage->FindFloat( "value", &p ) == EOK )
				m->_CreateObject( GetTool(), canvas, doc, GetTool()->GetPropertyAsPoint( "center_point" ), p );
			break;
		}
	}
}




