// oCADis (C)opyright 2007 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <gui/radiobutton.h>
#include "tool_align_state_1.h"
#include "common.h"
#include "transform.h"
#include "tool_align.h"
#include "toolbox_button.h"
#include "toolbox_point.h"
#include "toolbox_action.h"
#include "tableview.h"

using namespace os;

///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E
//
///////////////////////////////////////////////////////////////////////////////

class AlignState1 :: Private
{
public:
	Private(){};

	enum AlignState1Message { MSG_APPLY, MSG_CANCEL, MSG_VERT_NONE, MSG_VERT_TOP, MSG_VERT_MIDDLE, MSG_VERT_BOTTOM,
							  MSG_HORIZ_NONE, MSG_HORIZ_LEFT, MSG_HORIZ_CENTER, MSG_HORIZ_RIGHT,
							  MSG_BASE_BOUND, MSG_BASE_FIRST, MSG_BASE_LAST };

	TableView* m_Table;

	// Base frame
	ToolboxFrame* m_BaseFrame;
	TableView* m_BaseTable;
	RadioButton* m_BaseFirst;
	RadioButton* m_BaseLast;
	RadioButton* m_BaseBound;

	// Vertical frame
	ToolboxFrame* m_VertFrame;
	TableView* m_VertTable;
	RadioButton* m_VertNone;
	RadioButton* m_VertTop;
	RadioButton* m_VertMiddle;
	RadioButton* m_VertBottom;

	// Horizontal frame
	ToolboxFrame* m_HorizFrame;
	TableView* m_HorizTable;
	RadioButton* m_HorizNone;
	RadioButton* m_HorizLeft;
	RadioButton* m_HorizCenter;
	RadioButton* m_HorizRight;

	ToolboxAction* m_Action;

	void _Layout( Rect cFrame )
	{
		m_Table->SetFrame( Rect( 0, 0, cFrame.Width(), cFrame.Height() ) );
	}

	void _Apply( Canvas* canvas, Document* doc )
	{
		Rect bounds;

		if( m_BaseFirst->GetValue().AsBool() )
		{
			bounds = doc->GetSelectedObject( 0 )->GetBounds();
		}
		else if( m_BaseLast->GetValue().AsBool() )
		{
			bounds = doc->GetSelectedObject( doc->GetNoSelectedObjects() - 1 )->GetBounds();
		}
		else if( m_BaseBound->GetValue().AsBool() )
		{
			bounds = doc->GetSelectedBoundingBox();
		}

		UndoModifyObjects* undo = new UndoModifyObjects( "Align" );
	
		double horiz_translation = 0.0f;
		double vert_translation = 0.0f;

		for( int i = 0 ; i < (int)doc->GetNoSelectedObjects() ; i++ )
		{
			Object* obj = doc->GetSelectedObject( i );
			obj->Erase( canvas );
			obj->SetSelected( false );

			undo->AddOriginalObject( obj );

			Object* new_obj = obj->Duplicate();
			Rect obj_bounds = new_obj->GetBounds();

			// Check vertical
			if( m_VertTop->GetValue().AsBool() )
			{
				vert_translation = obj_bounds.bottom - bounds.bottom;
			}
			else if( m_VertMiddle->GetValue().AsBool() )
			{
				double b_cen = ( bounds.bottom - bounds.top ) / 2.0f + bounds.top;
				double o_cen = ( obj_bounds.bottom - obj_bounds.top ) / 2.0f + obj_bounds.top;
				vert_translation = o_cen - b_cen;
			}
			else if( m_VertBottom->GetValue().AsBool() )
			{
				vert_translation = obj_bounds.top - bounds.top;
			}

			// Check horizontal
			if( m_HorizLeft->GetValue().AsBool() )
			{
				horiz_translation = obj_bounds.left - bounds.left;
			}
			else if( m_HorizCenter->GetValue().AsBool() )
			{
				double b_cen = ( bounds.right - bounds.left ) / 2.0f + bounds.left;
				double o_cen = ( obj_bounds.right - obj_bounds.left ) / 2.0f + obj_bounds.left;
				horiz_translation = o_cen - b_cen;
			}
			else if( m_HorizRight->GetValue().AsBool() )
			{
				horiz_translation = obj_bounds.right - bounds.right;
			}
	
			Transform transform;
			transform.Translation( Point( - horiz_translation, - vert_translation ) );

			// Apply to object		
			new_obj->SetSelected( false );
			new_obj->DoTransform( transform );
			new_obj->Draw( canvas );

			undo->AddTransformedObject( new_obj );

			doc->ReplaceObject( obj, new_obj );
		}

		doc->AddUndoAction( undo );
		doc->SetModification();
		doc->DeselectAllObjects();
	}
};

///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S
//
///////////////////////////////////////////////////////////////////////////////

AlignState1 :: AlignState1( Tool* theTool, ToolboxSideBar* toolbox ) : ToolState( theTool, toolbox )
{
	// Create the private class
	m = new Private();

	// Create the table
	m->m_Table = new TableView( Rect(), "", 1, 4, false );
	m->m_Table->SetRowSpacings( 4 );
	AddChild( m->m_Table );

	// Create the base frame
	m->m_BaseFrame = new ToolboxFrame( Rect(0,0,100,100), "Base", "Base" );
	m->m_Table->Attach( m->m_BaseFrame, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create Base table
	m->m_BaseTable = new TableView( Rect(), "", 1, 4, false );
	m->m_BaseTable->SetRowSpacings( 4 );
	m->m_BaseFrame->AddTableView( m->m_BaseTable );

	// Create widgets in base table
	m->m_BaseBound = new RadioButton( Rect(), "The bounding box", "The bounding box", new Message( Private::MSG_BASE_BOUND ) );
	m->m_BaseTable->Attach( m->m_BaseBound, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );
	m->m_BaseFirst = new RadioButton( Rect(), "First selected object", "First selected object", new Message( Private::MSG_BASE_FIRST ) );
	m->m_BaseTable->Attach( m->m_BaseFirst, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );
	m->m_BaseLast = new RadioButton( Rect(), "Last selected object", "Last selected object", new Message( Private::MSG_BASE_LAST ) );
	m->m_BaseTable->Attach( m->m_BaseLast, 0, 1, 2, 3, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create the vertical frame
	m->m_VertFrame = new ToolboxFrame( Rect(0,0,100,100), "Vertical", "Vertical" );
	m->m_Table->Attach( m->m_VertFrame, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create Vert table
	m->m_VertTable = new TableView( Rect(), "", 1, 4, false );
	m->m_VertTable->SetRowSpacings( 4 );
	m->m_VertFrame->AddTableView( m->m_VertTable );

	// Create widgets in Vert table
	m->m_VertNone = new RadioButton( Rect(), "None", "None", new Message( Private::MSG_VERT_NONE ) );
	m->m_VertTable->Attach( m->m_VertNone, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );
	m->m_VertTop = new RadioButton( Rect(), "Top", "Top", new Message( Private::MSG_VERT_TOP ) );
	m->m_VertTable->Attach( m->m_VertTop, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );
	m->m_VertMiddle = new RadioButton( Rect(), "Middle", "Middle", new Message( Private::MSG_VERT_MIDDLE ) );
	m->m_VertTable->Attach( m->m_VertMiddle, 0, 1, 2, 3, ( TABLE_EXPAND | TABLE_FILL ), 0 );
	m->m_VertBottom = new RadioButton( Rect(), "Bottom", "Bottom", new Message( Private::MSG_VERT_BOTTOM ) );
	m->m_VertTable->Attach( m->m_VertBottom, 0, 1, 3, 4, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create the horizontal frame
	m->m_HorizFrame = new ToolboxFrame( Rect(0,0,100,100), "Horizontal", "Horizontal" );
	m->m_Table->Attach( m->m_HorizFrame, 0, 1, 2, 3, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create horiz table
	m->m_HorizTable = new TableView( Rect(), "", 1, 4, false );
	m->m_HorizTable->SetRowSpacings( 4 );
	m->m_HorizFrame->AddTableView( m->m_HorizTable );

	// Create widgets in Horiz table
	m->m_HorizNone = new RadioButton( Rect(), "None", "None", new Message( Private::MSG_HORIZ_NONE ) );
	m->m_HorizTable->Attach( m->m_HorizNone, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );
	m->m_HorizLeft = new RadioButton( Rect(), "Left", "Left", new Message( Private::MSG_HORIZ_LEFT ) );
	m->m_HorizTable->Attach( m->m_HorizLeft, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );
	m->m_HorizCenter = new RadioButton( Rect(), "Center", "Center", new Message( Private::MSG_HORIZ_CENTER ) );
	m->m_HorizTable->Attach( m->m_HorizCenter, 0, 1, 2, 3, ( TABLE_EXPAND | TABLE_FILL ), 0 );
	m->m_HorizRight = new RadioButton( Rect(), "Right", "Right", new Message( Private::MSG_HORIZ_RIGHT ) );
	m->m_HorizTable->Attach( m->m_HorizRight, 0, 1, 3, 4, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create buttons and attach them to table
	m->m_Action = new ToolboxAction( Rect(), "" );
	m->m_Table->Attach( m->m_Action, 0, 1, 3, 4, ( TABLE_EXPAND | TABLE_FILL ), ( TABLE_EXPAND | TABLE_FILL ) );

	// Add action items
	m->m_Action->AddButton( "Apply", (Image*) LoadImage( "apply.png" ), new Message( Private::MSG_APPLY ) , "Apply alignment to selected objects " );
	m->m_Action->AddButton( "Cancel", (Image*) LoadImage( "cancel.png" ), new Message( Private::MSG_CANCEL ) , "Cancel the tool and return to main toolbox " );

	State::SetToolbox( this );
}

AlignState1 :: ~AlignState1()
{
	delete m;
}

String AlignState1 :: GetStatusText()
{
	return String( "Select type of alignment" );
}

void AlignState1 :: SetFrame( const Rect& cRect, bool bNotifyServer )
{
	Toolbox::SetFrame( cRect, bNotifyServer );
	m->_Layout( GetFrame() );
}

void AlignState1 :: AllAttached( void )
{
	Toolbox::AllAttached();

	m->m_VertNone->SetValue( true );
	m->m_HorizNone->SetValue( true );
	m->m_BaseBound->SetValue( true );

	m->m_BaseFirst->SetTarget( this );
	m->m_BaseLast->SetTarget( this );
	m->m_BaseBound->SetTarget( this );

	m->m_VertNone->SetTarget( this );
	m->m_VertTop->SetTarget( this );
	m->m_VertMiddle->SetTarget( this );
	m->m_VertBottom->SetTarget( this );
	m->m_HorizNone->SetTarget( this );
	m->m_HorizLeft->SetTarget( this );
	m->m_HorizCenter->SetTarget( this );
	m->m_HorizRight->SetTarget( this );
	m->m_Action->SetTarget( this );
}

void AlignState1 :: HandleMessage( Canvas* canvas, Document* doc, Message* pcMessage )
{
	switch( pcMessage->GetCode() )
	{	
	case Private::MSG_BASE_BOUND:
		break;
	case Private::MSG_BASE_FIRST:
		break;
	case Private::MSG_BASE_LAST:
		break;
	case Private::MSG_APPLY:
		m->_Apply( canvas, doc );
		// Fall through
	case Private::MSG_CANCEL:
		SetMainToolbox();
		break;
	}
}



