// oCADis (C)opyright 2007 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <cassert>

#include "object_line.h"
#include "document.h"
#include "cadmath.h"

using namespace os;
using namespace std;

ObjectLine :: ObjectLine( Point p1, Point p2 ) : Object()
{
	m_P1 = p1;
	m_P2 = p2;
}

ObjectLine :: ~ObjectLine()
{
}

void ObjectLine :: Erase( Canvas* canvas )
{
	canvas->SetStyle( m_Style );
	canvas->SetColour( 1.0f, 1.0f, 1.0f );

	float i = canvas->WorldToScreen( m_Style.GetLineWidth() );
	i+= 3.0f;
	canvas->SetLineWidth( canvas->ScreenToWorld( i ) );

	canvas->DrawLine( m_P1, m_P2, false, true );
}

void ObjectLine :: Draw( Canvas* canvas, bool save_background, bool update_server )
{
	canvas->SetStyle( m_Style );

	if( m_Selected )
		canvas->SetColour( 0.0f, 1.0f, 0.0f );

	canvas->DrawLine( m_P1, m_P2, save_background, update_server );
}

void ObjectLine :: Draw( Canvas* canvas, Transform& transform, bool save_background, bool update_server )
{
	canvas->SetStyle( m_Style );

	if( m_Selected )
		canvas->SetColour( 0.0f, 1.0f, 0.0f );

	canvas->DrawLine( transform.Calculate( m_P1 ), transform.Calculate( m_P2 ), save_background, update_server );
}

void ObjectLine :: DoTransform( Transform& transform )
{
	m_P1 = transform.Calculate( m_P1 );
	m_P2 = transform.Calculate( m_P2 );
}

Rect ObjectLine :: GetBounds()
{
	Rect r = CadMath::SortRect( m_P1, m_P2 );
	
	// Consider the linewidth
	r.left -= m_Style.GetLineWidth() / 2.0f;
	r.top -= m_Style.GetLineWidth() / 2.0f;
	r.right += m_Style.GetLineWidth() / 2.0f;
	r.bottom += m_Style.GetLineWidth() / 2.0f;

	return r;
}

bool ObjectLine :: HitCheck( Point coord, double sens )
{
	double L;
  	double s,r;

	L = sqrt( ( m_P2.x - m_P1.x ) * ( m_P2.x - m_P1.x ) + ( m_P2.y - m_P1.y ) * ( m_P2.y - m_P1.y ) );
	r = ( ( m_P1.y - coord.y ) * ( m_P1.y - m_P2.y ) - ( m_P1.x - coord.x ) * ( m_P2.x - m_P1.x ) ) / ( L * L );

	if( ( r >= ( - 0.0000001f ) ) && ( r <= ( 1.0f + 0.0000001f ) ) )
	{
		s = ( ( m_P1.y - coord.y ) * ( m_P2.x - m_P1.x ) - ( m_P1.x - coord.x ) * ( m_P2.y - m_P1.y) ) / ( L * L );
		return( fabs( s * L ) < ( m_Style.GetLineWidth() + sens ) );  // <-- FIXME
	}
	
	return false;
}

Object* ObjectLine :: Duplicate()
{
	Object* obj = new ObjectLine( m_P1, m_P2 );

	obj->SetStyle( m_Style );
	obj->SetLayer( m_Layer );

	return obj;
}


bool ObjectLine :: SetProperty( int which, Variant& value )
{
	switch( which )
	{
		case 0:  // P1
			m_P1 = value.AsPoint();
			break;
		case 1:  // P2
			m_P2 = value.AsPoint();
			break;
		default:
			return false;
	}

	return true;
}

bool ObjectLine :: GetProperty( int which, Variant& value )
{
	switch( which )
	{
		case 0:  // P1
			value.SetPoint( m_P1 );
			break;
		case 1:  // P2
			value.SetPoint( m_P2 );
			break;
		default:
			return false;
	}

	return true;
}

bool ObjectLine :: GetSnapPoint( int type, Point p, vector< Point >& list )
{
	if( type & OSNAP_ENDPOINT )
	{
		list.push_back( m_P1 );
		list.push_back( m_P2 );
	}

	if( type & OSNAP_MIDPOINT )
	{
		list.push_back( Point( m_P1.x  + ( m_P2.x - m_P1.x ) / 2.0f, m_P1.y  + ( m_P2.y - m_P1.y ) / 2.0f ) );
	}

	if( type & OSNAP_NEAPOINT )
	{
		double L = sqrt( ( m_P2.x - m_P1.x ) * ( m_P2.x - m_P1.x) + ( m_P2.y - m_P1.y ) * ( m_P2.y - m_P1.y ) );
		double r = ( ( m_P1.y - p.y ) * ( m_P1.y - m_P2.y ) - ( m_P1.x - p.x ) * ( m_P2.x - m_P1.x ) ) / ( L * L );

		// MAke sure we stay on the line
		if( r < 0.0 ) r = 0.0;
		if( r > 1.0 ) r = 1.0;

		list.push_back( Point( m_P1.x + r * ( m_P2.x - m_P1.x ), m_P1.y + r * ( m_P2.y - m_P1.y ) ) );
	}

	return true;
}

String ObjectLine :: Save()
{
	String data;
	data.Format( "OL, %f, %f, %f, %f", m_P1.x, m_P1.y, m_P2.x, m_P2.y );
	data += SaveAttributes();
	return String( data );  
}

bool ObjectLine :: Load( Document* doc, Layer* current_layer, vector< String >& split )
{
	assert( doc != NULL );
	assert( current_layer != NULL );

	// Make sure we have the correct number of arguments
	Object* obj = new ObjectLine( Point( atof( split[1].c_str() ), atof( split[2].c_str() )), Point( atof( split[3].c_str() ), atof( split[4].c_str() ) ) );
	
	obj->SetLayer( current_layer );
	obj->GetStyle().SetStyle( 5, split );

	current_layer->AddObject( obj );

	return true;
}

