// oCADis (C)opyright 2007 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "windialogs/windia_alert.h"
#include "common.h"
#include "cadmath.h"
#include "main.h"

using namespace os;

WinDiaAlert :: 	WinDiaAlert( enum AlertType type, String title, String msg, ... ) : Window( Rect( 100, 50, 500, 350 ), "alert_window", "", WND_NOT_RESIZABLE | WND_NO_ZOOM_BUT | WND_NO_DEPTH_BUT )
{
	// Create the table
	m_theTable = new TableView( Rect(), "", 2, 2, false );
	m_theTable->SetRowSpacings( 5 );
	m_theTable->SetColumnSpacings( 10 );
	AddChild( m_theTable );

	// Create title
	m_Title = new StringView( Rect(), "", title );
	Font* font = m_Title->GetFont();
	font->SetSize( font->GetSize() + 2.0f );
	m_Title->SetFont( font );

	m_theTable->Attach( m_Title, 1, 2, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create message
	m_Message = new StringView( Rect(), "", msg );
	m_theTable->Attach( m_Message, 1, 2, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create the button table
	m_ButtonTable = new TableView( Rect(), "", 0, 0, false );
	m_ButtonTable->SetColumnSpacings( 5 );
	AddChild( m_ButtonTable );

	// Add the buttons
	va_list pArgs;
	va_start( pArgs, msg );
	_AddButtons( pArgs );
	va_end( pArgs );

	// Set title and icon
	m_Icon = NULL;
	switch( type )
	{
		case ALERT_INFO: 
			m_Icon = new ImageView( Rect(), "", LoadImage( "alert_info.png" ) );
			SetTitle( "Information - oCADis" );
			break;
		case ALERT_WARNING:
			m_Icon = new ImageView( Rect(), "", LoadImage( "alert_warning.png" ) );
			SetTitle( "Warning - oCADis" );
			break;
		case ALERT_ERROR:
			m_Icon = new ImageView( Rect(), "", LoadImage( "alert_error.png" ) );
			SetTitle( "Error - oCADis" );
			break;
		case ALERT_QUESTION:
			m_Icon = new ImageView( Rect(), "", LoadImage( "alert_question.png" ) );
			SetTitle( "Question - oCADis" );
			break;
		case ALERT_NONE:
			SetTitle( "Message box - oCADis" );
			break;
	}

	if( m_Icon != NULL )
		m_theTable->Attach( m_Icon, 0, 1, 0, 2, 0,0 );

	// Resize alert
	Point cSize = m_theTable->GetPreferredSize( false );
	cSize.x = CadMath::max( cSize.x, m_ButtonTable->GetPreferredSize( false ).x );
	cSize.y += m_ButtonTable->GetPreferredSize( false ).y;
	cSize += Point( 20.0f, 20.0f );

	ResizeTo( cSize );

	// Move alert to center of screen
	CenterInWindow( oCADisApp::GetAppWindow() );

	_Layout();

	m_hMsgPort = -1;
	m_pcInvoker = NULL;
}

void WinDiaAlert ::ShowErrorAlert( os::String title, os::String msg )
{
	WinDiaAlert* alert = new WinDiaAlert( ALERT_ERROR, title, msg, "OK", NULL );
	alert->Go( new Invoker() );
}

WinDiaAlert :: ~WinDiaAlert()
{
	for( uint i = 0 ; i < m_ListOfButtons.size() ; i++ )
		delete m_ListOfButtons[ i ];

	m_ListOfButtons.clear();

	delete m_theTable;
	delete m_ButtonTable;
//	delete m_Title;		// FIXME: Why cannot I delete these three items??
//	delete m_Message;
//	delete m_Icon;

}

void WinDiaAlert :: HandleMessage( os::Message* pcMessage )
{
	if( pcMessage->GetCode() < int32 ( m_ListOfButtons.size() ) )
	{
		if( m_hMsgPort >= 0 )
		{
			send_msg( m_hMsgPort, pcMessage->GetCode(), NULL, 0 );
		}
		else
		{
			if( m_pcInvoker != NULL )
			{
				Message *pcMsg = m_pcInvoker->GetMessage();

				if( pcMsg != NULL )
				{
					pcMsg->AddInt32( "which", pcMessage->GetCode() );
					m_pcInvoker->Invoke();
				}
			}
			PostMessage( M_QUIT );
		}
	}
	else
	{
		Handler::HandleMessage( pcMessage );
	}
}

void WinDiaAlert :: FrameSized( const os::Point& delta )
{
	Window::FrameSized( delta );
	_Layout();
}
	
void WinDiaAlert :: Raise()
{
	if( IsVisible() )
		Hide();

	Show();
	MakeFocus();
}

int WinDiaAlert :: Go()
{
	uint32 nCode;
	int nError;

	m_hMsgPort = create_port( "alert_port", DEFAULT_PORT_SIZE );

	Show();
	MakeFocus();
	if( m_hMsgPort < 0 )
	{
		dbprintf( "Alert::WaitForSelection() failed to create message port\n" );
		PostMessage( M_QUIT );
		return ( -1 );
	}
	nError = get_msg( m_hMsgPort, &nCode, NULL, 0 );
	if( nError < 0 )
	{
		PostMessage( M_QUIT );
		return ( -1 );
	}
	else
	{
		PostMessage( M_QUIT );
		return ( nCode );
	}
}

void WinDiaAlert :: Go( Invoker * pcInvoker )
{
	m_pcInvoker = pcInvoker;
	Show();
	MakeFocus();
}

//////////////////////////////////////////////////////////////////////
//
// P R I V A T E   M E T H O D S
//
//////////////////////////////////////////////////////////////////////


void WinDiaAlert :: _AddButtons( va_list pButtons )
{
	int i = 0;
	const char *pzButName;

	while( ( pzButName = va_arg( pButtons, const char * ) ) != NULL )
	{
		Button *pcButton = new Button( Rect( 0, 0, 1, 1 ), pzButName, pzButName, new Message( i++ ) );

		m_ListOfButtons.push_back( pcButton );
		pcButton->SetTabOrder();

		m_ButtonTable->Resize( 1, m_ButtonTable->GetNumberOfColumns() + 1 );
		m_ButtonTable->Attach( pcButton, 0 + ( m_ButtonTable->GetNumberOfColumns() - 1 ), 1  + ( m_ButtonTable->GetNumberOfColumns() - 1 ) , 0, 1, 0, 0 );
	}

	m_ButtonTable->SetColumnSpacings( 5 );
}

void WinDiaAlert :: _Layout()
{
	Rect cFrame = GetBounds();
	cFrame.Resize( 10, 10, -2, -2 );

	// Set button table
	Point tsize = m_ButtonTable->GetPreferredSize( false );
	Rect tframe( cFrame.right - tsize.x, cFrame.bottom - tsize.y, cFrame.right, cFrame.bottom );
	m_ButtonTable->SetFrame( tframe );

	// Set table
	Rect t1frame( cFrame.left, cFrame.top, cFrame.right, tframe.top - 2.0f );
	m_theTable->SetFrame( t1frame );
}
