// oCADis (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <vector>

#include "tool_multiselect.h"
#include "windialogs/windia_alert.h"
#include "common.h"
#include "main.h"
#include "tool_move/tool_move.h"
#include "toolbox_button.h"
#include "toolbox_point.h"
#include "toolbox_action.h"
#include "tableview.h"

using namespace os;
using namespace std;

///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E
//
///////////////////////////////////////////////////////////////////////////////

class StateMultiSelect :: Private
{
public:
	Private(){};

	void _Layout( Rect cFrame )
	{
		m_Table->SetFrame( Rect( 0, 0, cFrame.Width(), cFrame.Height() ) );
	}


	enum MultiSelectStateMessage { MSG_CANCEL, MSG_SELECT_ALL, MSG_DESELECT_ALL, MSG_DONE };

	ToolboxAction* m_Action;

	TableView* m_Table;
};

///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S
//
///////////////////////////////////////////////////////////////////////////////
StateMultiSelect :: StateMultiSelect( Tool* theTool, ToolboxSideBar* toolbox ) : ToolState( theTool, toolbox )
{
	// Create the private class
	m = new Private();

	// Create the table
	m->m_Table = new TableView( Rect(), "", 1, 1, false );
	m->m_Table->SetRowSpacings( 4 );
	AddChild( m->m_Table );

	// Create buttons and attach them to table
	m->m_Action = new ToolboxAction( Rect(), "" );
	m->m_Table->Attach( m->m_Action, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), ( TABLE_EXPAND | TABLE_FILL ) );

	// Add action items
	m->m_Action->AddButton( "Cancel", (Image*) LoadImage( "cancel.png" ), new Message( Private::MSG_CANCEL ) , "Cancel the tool and return to main toolbox " );
	m->m_Action->AddButton( "Done", (Image*) LoadImage( "done.png" ), new Message( Private::MSG_DONE ) , "Selection finished " );
	m->m_Action->AddButton( "Select all", (Image*) LoadImage( "select_all.png" ), new Message( Private::MSG_SELECT_ALL ) , "Select all objects " );
	m->m_Action->AddButton( "Deselect all", (Image*) LoadImage( "circle.png" ), new Message( Private::MSG_DESELECT_ALL ) , "Deselect all objects " );

	State::SetToolbox( this );
}

StateMultiSelect :: ~StateMultiSelect()
{
	delete m;
}

String StateMultiSelect :: GetStatusText()
{
	return String( "Click to select a objects. Use Shift to select multiple objects" );
}

void StateMultiSelect :: PointSelected( Canvas* canvas, Document* doc, Point coord )
{
	canvas->Undraw();

	// Unselect all objects unless shift is pressed
	if( ! (GetQualifiers() & QUAL_SHIFT) )
	{
		doc->DeselectAllObjects();
		canvas->Refresh();
	}

	// Calculate selection sensitivity
	double sens = GET_CANVAS()->ScreenToWorld( (float) GET_PREFERENCES().GetSelectionSensity() );

	// Check if an object is hit
	Object *obj = doc->HitCheck( coord, sens );

	if( obj != NULL )
	{
		if( obj->IsSelected() )
			doc->DeselectObject( obj );
		else
			doc->SelectObject( obj );

		obj->Draw( canvas );
	}
}

void StateMultiSelect :: MouseBox( Canvas* canvas, Document* doc, Rect box )
{
	// Unselect all objects unless shift is pressed
	if( ! (GetQualifiers() & QUAL_SHIFT) )
	{
		doc->DeselectAllObjects();
		canvas->Refresh();
	}	

	vector< Object* > hits;

	doc->HitCheck( box, hits );

	for( uint i = 0 ; i < hits.size(); i++ )
	{
		Object* obj = hits[ i ];

		doc->SelectObject( obj );

		obj->Draw( canvas );
	}
}


bool StateMultiSelect :: AllowMouseBox( Canvas* canvas, Document* doc, Point coord )
{
	// Unselect all objects unless shift is pressed
	if( ! (GetQualifiers() & QUAL_SHIFT) )
	{
		doc->DeselectAllObjects();
		canvas->Refresh();
	}

	// Calculate selection sensitivity
	double sens = canvas->ScreenToWorld( (float) GET_PREFERENCES().GetSelectionSensity() );

	// Check if we hit an object
	return doc->HitCheck( coord, sens ) == NULL;
}

void StateMultiSelect :: SetFrame( const Rect& cRect, bool bNotifyServer )
{
	Toolbox::SetFrame( cRect, bNotifyServer );
	m->_Layout( GetFrame() );
}

void StateMultiSelect :: AllAttached( void )
{
	Toolbox::AllAttached();
	m->m_Action->SetTarget( this );
}

void StateMultiSelect :: HandleMessage( Canvas* canvas, Document* doc, Message* pcMessage )
{
	switch( pcMessage->GetCode() )
	{	
	case Private::MSG_CANCEL:		
		SetMainToolbox();
		break;
	case Private::MSG_DONE:
		// Make sure at least 1 object is selected
		if( doc->GetNoSelectedObjects() > 0 )
			SelectionDone( canvas, doc );
		else if( !GET_PREFERENCES().IsExpertMode() )
		{
			WinDiaAlert::ShowErrorAlert( "Selection", "No objects are selected." );
		}

		break;
	case Private::MSG_SELECT_ALL:
		doc->SelectAllObjects();
		canvas->Refresh();	
		break;
	case Private::MSG_DESELECT_ALL:
		doc->DeselectAllObjects();
		canvas->Refresh();
		break;
	}
}

void StateMultiSelect :: SelectionDone( Canvas* canvas, Document* doc )
{
	doc->DeselectAllObjects();
	SetMainToolbox();
}
