// oCADis (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "tool_line_state_1.h"
#include "common.h"
#include "object_line.h"
#include "undo.h"
#include "cadmath.h"
#include "tool_line.h"
#include "toolbox_button.h"
#include "toolbox_point.h"
#include "toolbox_action.h"
#include "tableview.h"

using namespace os;

///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E
//
///////////////////////////////////////////////////////////////////////////////

class LineState1 :: Private
{
public:
	Private(){};

	void _Layout( Rect cFrame )
	{
		m_Table->SetFrame( Rect( 0, 0, cFrame.Width(), cFrame.Height() ) );
	}

	enum LineState1Message { MSG_CLOSE, MSG_POINT, MSG_CANCEL };

	ToolboxPoint* m_NextPoint;
	ToolboxAction* m_Action;

	TableView* m_Table;

	Point m_FirstPoint;
	Point m_SavedPoint;
};

///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S
//
///////////////////////////////////////////////////////////////////////////////
LineState1 :: LineState1( Tool* theTool, ToolboxSideBar* toolbox ) : ToolState( theTool , toolbox )
{
	// Create the private class
	m = new Private();

	// Create the table
	m->m_Table = new TableView( Rect(), "", 1, 3, false );
	m->m_Table->SetRowSpacings( 4 );
	AddChild( m->m_Table );

	// Create buttons and attach them to table
	m->m_NextPoint = new ToolboxPoint( Rect(), "Next point", "X: ", "Y: ", new Message( Private::MSG_POINT ), true );
	m->m_Table->Attach( m->m_NextPoint, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m->m_Action = new ToolboxAction( Rect(), "" );
	m->m_Table->Attach( m->m_Action, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), ( TABLE_EXPAND | TABLE_FILL ) );

	// Add action items
	m->m_Action->AddButton( "Cancel", (Image*) LoadImage( "cancel.png" ), new Message( Private::MSG_CANCEL ) , "Cancel the tool and return to main toolbox " );
	m->m_Action->AddButton( "Close", (Image*) LoadImage( "circle.png" ), new Message( Private::MSG_CLOSE ), "Connects the last line with the first line " );

	State::SetToolbox( this );
}

LineState1 :: ~LineState1()
{
	delete m;
}

void LineState1 :: Init( Canvas* canvas, Document* doc )
{
	m->m_FirstPoint = GetTool()->GetPropertyAsPoint( "first_point" );
	m->m_SavedPoint = GetTool()->GetPropertyAsPoint( "saved_point" );
	m->m_NextPoint->SetOrigin( m->m_SavedPoint );

	canvas->PrepareForXORMode();
}

void LineState1 :: GotFocus( Canvas* canvas, Document* doc, Point coord )
{
	coord = doc->MakeSnap( m->m_SavedPoint, coord );
	m->m_NextPoint->SetValue( coord );
}

void LineState1 :: LostFocus( Canvas* canvas, Document* doc, Point coord )
{
	canvas->Undraw();
	m->m_NextPoint->ClearValue();
}

void LineState1 :: MouseMove( Canvas* canvas, Document* doc, Point coord )
{
	canvas->Undraw();
	coord = doc->MakeSnap( m->m_SavedPoint, coord );
	m->m_NextPoint->SetValue( coord );

	canvas->SetStyle( doc->GetCurrentStyle() );
	canvas->DrawLine( m->m_SavedPoint, coord, true );
}

String LineState1 :: GetStatusText()
{
	return String( "Select next point" );
}

void LineState1 :: PointSelected( Canvas* canvas, Document* doc, Point coord )
{
	canvas->Undraw();

	coord = doc->MakeSnap( m->m_SavedPoint, coord );

	// Create object
	ObjectLine* obj = new ObjectLine( m->m_SavedPoint, coord );
	doc->AddObject( obj );
	obj->Draw( canvas );

	// Create Undo object
	UndoAddObject* undo = new UndoAddObject( "Create Line", obj );
	doc->AddUndoAction( undo );

	doc->SetContinuePoint( coord, CadMath::GetAngle( coord, m->m_SavedPoint ) + M_PI / 2.0f );
	doc->SetModification();

	GetTool()->SetProperty( "saved_point", coord );
	GetTool()->SetState( 1 );
}


void LineState1 :: SetFrame( const Rect& cRect, bool bNotifyServer )
{
	Toolbox::SetFrame( cRect, bNotifyServer );
	m->_Layout( GetFrame() );
}

void LineState1 :: AllAttached( void )
{
	Toolbox::AllAttached();

	m->m_NextPoint->SetTarget( this );
	m->m_Action->SetTarget( this );
}

void LineState1 :: HandleMessage( Canvas* canvas, Document* doc, Message* pcMessage )
{
	switch( pcMessage->GetCode() )
	{	
	case Private::MSG_CANCEL:
		SetMainToolbox();
		break;
	case Private::MSG_CLOSE: 
		PointSelected( canvas, doc, m->m_FirstPoint );
		GetTool()->SetState( 0 );
		break;
	case Private::MSG_POINT:
		{
			Point p;

			if( pcMessage->FindPoint( "value", &p ) == EOK )
				PointSelected( canvas, doc, p );
			break;
		}
	}
}
