// oCADis (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <math.h>

#include "tool_circle_state_7.h"
#include "common.h"
#include "cadmath.h"
#include "object_circle.h"

using namespace os;

///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E
//
///////////////////////////////////////////////////////////////////////////////

class CircleState7 :: _Private
{
public:
	_Private(){};

	void _Layout( Rect cFrame )
	{
		m_Table->SetFrame( Rect( 0, 0, cFrame.Width(), cFrame.Height() ) );
	}

	void _CalculateCircle( Point p1, Point p2, Point coord, Point& center, float& radius )
	{
		double A,B,C,D,E,F,G;

		A = p2.x - p1.x;
		B = p2.y - p1.y;
		C = coord.x - p1.x;
		D = coord.y - p1.y;

		E = A * ( p1.x + p2.x ) + B * ( p1.y + p2.y );
		F = C * ( p1.x + coord.x ) + D * ( p1.y + coord.y );

		G = 2.0f * ( A * ( coord.y - p2.y ) - B * ( coord.x - p2.x ) );

		if( G != 0 )
		{
			center.x = ( D * E - B * F ) / G;
			center.y = ( A * F - C * E ) / G;

			radius = CadMath::Distance( p1, center );
		}
		else
		{
			center.x = 10.0f;
			center.y = 10.0f;
			radius = 0.0f;
		}
	}

	enum CircleState7Message { MSG_POINT, MSG_CANCEL };

	ToolboxPoint* m_ThirdPoint;
	ToolboxAction* m_Action;

	os::TableView* m_Table;
};

///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S
//
///////////////////////////////////////////////////////////////////////////////
CircleState7 :: CircleState7( ToolCircle* theTool, ToolboxSideBar* toolbox ) : ToolState( theTool, toolbox )
{
	// Create the private class
	m = new _Private();

	// Create the table
	m->m_Table = new TableView( Rect(), "", 1, 2, false );
	m->m_Table->SetRowSpacings( 4 );
	AddChild( m->m_Table );

	// Create buttons and attach them to table
	m->m_ThirdPoint = new ToolboxPoint( Rect(), "Third point", "X: ", "Y: ", new Message( _Private::MSG_POINT ), true );
	m->m_Table->Attach( m->m_ThirdPoint, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m->m_Action = new ToolboxAction( Rect(), "" );
	m->m_Table->Attach( m->m_Action, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), ( TABLE_EXPAND | TABLE_FILL ) );

	// Add action items
	m->m_Action->AddButton( "Cancel", (Image*) LoadImage( "cancel.png" ), new Message( _Private::MSG_CANCEL ) , "Cancel the tool and return to main toolbox " );

	State::SetToolbox( this );
}

CircleState7 :: ~CircleState7()
{	
	delete m;
}

void CircleState7 :: Init( Canvas* canvas, Document* doc )
{
	m->m_ThirdPoint->SetOrigin( GetTool()->GetPropertyAsPoint( "second_point" ) );
	canvas->PrepareForXORMode();
}

void CircleState7 :: GotFocus( Canvas* canvas, Document* doc, Point coord )
{
	coord = doc->MakeSnap( coord );
	m->m_ThirdPoint->SetValue( coord );
}

void CircleState7 :: LostFocus( Canvas* canvas, Document* doc, Point coord )
{
	m->m_ThirdPoint->ClearValue();
}

void CircleState7 :: MouseMove( Canvas* canvas, Document* doc, Point coord )
{
	coord = doc->MakeSnap( coord );
	m->m_ThirdPoint->SetValue( coord );

	Point center;
	float radius;

	canvas->SetStyle( doc->GetCurrentStyle() );

	m->_CalculateCircle( GetTool()->GetPropertyAsPoint( "first_point" ), GetTool()->GetPropertyAsPoint( "second_point" ), coord, center, radius );

	canvas->DrawCircle( center, radius, true );
}

String CircleState7 :: GetStatusText()
{
	return String( "Select third point of the circle" );
}

void CircleState7 :: PointSelected( Canvas* canvas, Document* doc, Point coord )
{
	canvas->Undraw();

	Point center;
	float radius;

	coord = doc->MakeSnap( coord );
	m->_CalculateCircle( GetTool()->GetPropertyAsPoint( "first_point" ), GetTool()->GetPropertyAsPoint( "second_point" ), coord, center, radius );

	// Create object
	ObjectCircle* obj = new ObjectCircle( center, radius );
	doc->AddObject( obj );
	obj->Draw( canvas );

	// Create Undo object
	UndoAddObject* undo = new UndoAddObject( "Create Circle", obj );
	doc->AddUndoAction( undo );
	doc->SetModification();

	GetTool()->SetState( 5 );
}

void CircleState7 :: SetFrame( const Rect& cRect, bool bNotifyServer )
{
	Toolbox::SetFrame( cRect, bNotifyServer );
	m->_Layout( cRect );
}

void CircleState7 :: AllAttached( void )
{
	Toolbox::AllAttached();

	m->m_ThirdPoint->SetTarget( this );
	m->m_Action->SetTarget( this );
}

void CircleState7 :: HandleMessage( Canvas* canvas, Document* doc, Message* pcMessage )
{
	switch( pcMessage->GetCode() )
	{
		case _Private::MSG_CANCEL:
			SetMainToolbox();
			break;
		case _Private::MSG_POINT:
			{
				Point p;

				if( pcMessage->FindPoint( "value", &p ) == EOK )
					PointSelected( canvas, doc, p );
		
				break;
			}
	}
}

