// oCADis (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <cassert>

#include "object_circle.h"
#include "document.h"
#include "cadmath.h"

using namespace os;
using namespace std;

ObjectCircle :: ObjectCircle( Point center, float radius ) : Object()
{
	m_Center = center;
	m_Radius = radius;
}

ObjectCircle :: ~ObjectCircle()
{
}

void ObjectCircle :: Erase( Canvas* canvas )
{
	canvas->SetStyle( m_Style );
	canvas->SetColour( 1.0f, 1.0f, 1.0f );
	float i = canvas->WorldToScreen( m_Style.GetLineWidth() );
	i+= 3.0f;
	canvas->SetLineWidth( canvas->ScreenToWorld( i ) );
	canvas->DrawCircle( m_Center, m_Radius, false, true );
}

void ObjectCircle :: Draw( Canvas* canvas, bool save_background, bool update_server )
{
	canvas->SetStyle( m_Style );

	if( m_Selected )
		canvas->SetColour( 0, 1.0f, 0 );

	canvas->DrawCircle( m_Center, m_Radius, save_background, update_server );
}

void ObjectCircle :: Draw( Canvas* canvas, Transform& transform, bool save_background, bool update_server )
{
	canvas->SetStyle( m_Style );

	if( m_Selected )
		canvas->SetColour( 0, 1.0f, 0 );

	canvas->DrawCircle( transform.Calculate( m_Center ), m_Radius, save_background, update_server ); // FIXME
}

void ObjectCircle :: DoTransform( Transform& transform )
{
	m_Center = transform.Calculate( m_Center );

// FIXME radius

}

Rect ObjectCircle :: GetBounds()
{
	Rect r( Point( m_Center.x - m_Radius, m_Center.y - m_Radius ), Point( m_Center.x + m_Radius, m_Center.y + m_Radius ) );
	
	// Consider the linewidth
	r.left -= m_Style.GetLineWidth() / 2.0f;
	r.top -= m_Style.GetLineWidth() / 2.0f;
	r.right += m_Style.GetLineWidth() / 2.0f;
	r.bottom += m_Style.GetLineWidth() / 2.0f;

	return r;
}

bool ObjectCircle :: HitCheck( Point coord, double sens )
{
	// Check if point is within bound box
	if( ! GetBounds().DoIntersect(coord)) 
		return false;

	double r = sqrt( ( m_Center.x - coord.x ) * ( m_Center.x - coord.x ) + ( m_Center.y - coord.y ) * ( m_Center.y - coord.y ) );

	return ( ( r > m_Radius - ( m_Style.GetLineWidth() + sens ) ) && ( r < m_Radius + ( m_Style.GetLineWidth() + sens ) ) );
}

Object* ObjectCircle :: Duplicate()
{
	Object* obj = new ObjectCircle( m_Center, m_Radius );

	obj->SetStyle( m_Style );
	obj->SetLayer( m_Layer );

	return obj;
}

bool ObjectCircle :: SetProperty( int which, Variant& value )
{
	switch( which )
	{
		case 0:  // Center
			m_Center = value.AsPoint();
			break;
		case 1:  // Radius
			m_Radius = value.AsDouble();
			break;
		default:
			return false;
	}

	return true;
}

bool ObjectCircle :: GetProperty( int which, Variant& value )
{
	switch( which )
	{
		case 0:  // Center
			value.SetPoint( m_Center );
			break;
		case 1:  // Radius
			value.SetDouble( m_Radius );
			break;
		default:
			return false;
	}

	return true;
}

bool ObjectCircle :: GetSnapPoint( int type, Point p, vector< Point >& list )
{
	if( type & OSNAP_CENPOINT )
	{
		list.push_back( m_Center );
	}

	if( type & OSNAP_QUAPOINT )
	{
		list.push_back( Point( m_Center.x, m_Center.y - m_Radius ) );
		list.push_back( Point( m_Center.x, m_Center.y + m_Radius ) );
		list.push_back( Point( m_Center.x - m_Radius, m_Center.y ) );
		list.push_back( Point( m_Center.x + m_Radius, m_Center.y ) );
	}

	if( type & OSNAP_NEAPOINT )
	{
		double angle = CadMath::GetAngle( m_Center, p );
		list.push_back( Point( m_Center.x + m_Radius * cos( angle ), m_Center.y + m_Radius * sin( angle ) ) );
	}

	return false;
}

String ObjectCircle :: Save()
{
	String data;
	data.Format( "OC, %f, %f, %f", m_Center.x, m_Center.y, m_Radius );
	data += SaveAttributes();
	return String( data );  
}

bool ObjectCircle :: Load( Document* doc, Layer* current_layer, vector< String >& split )
{
	assert( doc != NULL );
	assert( current_layer != NULL );

	// Make sure we have the correct number of arguments
	Object* obj = new ObjectCircle( Point( atof( split[1].c_str() ), atof( split[2].c_str() ) ), atof( split[3].c_str() ) );
	
	obj->SetLayer( current_layer );
	obj->GetStyle().SetStyle( 4, split );

	current_layer->AddObject( obj );

	return true;
}

