// oCADis (C)opyright 2007 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <cassert>

#include "object_arc.h"
#include "cadmath.h"
#include "document.h"

using namespace os;
using namespace std;


ObjectArc :: ObjectArc( Point center, double radius, double start_angle, double stop_angle ) : Object()
{
	m_Center = center;
	m_Radius = radius;
	m_StartAngle = start_angle;
	m_EndAngle = stop_angle;
}

ObjectArc :: ~ObjectArc()
{
}

void ObjectArc :: Erase( Canvas* canvas )
{
	canvas->SetStyle( m_Style );
	canvas->SetColour( 1.0f, 1.0f, 1.0f );
	float i = canvas->WorldToScreen( m_Style.GetLineWidth() );
	i+= 3.0f;
	canvas->SetLineWidth( canvas->ScreenToWorld( i ) );
	canvas->DrawArc( m_Center, m_Radius, m_StartAngle, false, true );
}

void ObjectArc :: Draw( Canvas* canvas, bool save_background, bool update_server )
{
	canvas->SetStyle( m_Style );

	if( m_Selected )
		canvas->SetColour( 0.0f, 1.0f, 0.0f );

	canvas->DrawArc( m_Center, m_Radius, m_StartAngle, m_EndAngle, save_background, update_server );
	canvas->DrawArc( m_Center, m_Radius, m_StartAngle, m_EndAngle, save_background, update_server );
}

void ObjectArc :: Draw( Canvas* canvas, Transform& transform, bool save_background, bool update_server )
{
	canvas->SetStyle( m_Style );

	if( m_Selected )
		canvas->SetColour( 0, 255, 0 );

	Point p1, p2, center;
	double start_angle, end_angle;

	_Calc3P( p1, p2 );

	p1 = transform.Calculate( p1 );
	p2 = transform.Calculate( p2 );
	center = transform.Calculate( m_Center );

	start_angle = CadMath::GetAngle( center, p1 );
	end_angle = CadMath::GetAngle( center, p2 );

	canvas->DrawArc( center, m_Radius, start_angle, end_angle, save_background, update_server );
}

void ObjectArc :: DoTransform( Transform& transform )
{
	Point p1, p2;

	_Calc3P( p1, p2 );

	p1 = transform.Calculate( p1 );
	p2 = transform.Calculate( p2 );
	m_Center = transform.Calculate( m_Center );

	m_StartAngle = CadMath::GetAngle( m_Center, p1 );

	m_EndAngle = CadMath::GetAngle( m_Center, p2 );
}

Rect ObjectArc :: GetBounds()
{
	Rect r;	
	double sx,sy,ex,ey;

	sx = m_Center.x + m_Radius * cos( m_StartAngle );
  	sy = m_Center.y + m_Radius * sin( m_StartAngle );

	ex = m_Center.x + m_Radius * cos( m_EndAngle );
	ey = m_Center.y + m_Radius * sin( m_EndAngle );

	r.left = CadMath::min( sx, ex ) - m_Style.GetLineWidth();
	r.top = CadMath::min( sy, ey )- m_Style.GetLineWidth();
	r.right = CadMath::max( sx, ex ) + m_Style.GetLineWidth();
	r.bottom = CadMath::max( sy, ey ) + m_Style.GetLineWidth();

	// Check each quadrant
	for( double i = 0.0f ; i <= M_PI * 2.0f ; i += M_PI / 2.0f )
	{
		if( CadMath::BetweenAngles( m_StartAngle, m_EndAngle, i ) )
		{
			sx = m_Center.x + m_Radius * cos( i );
			sy = m_Center.y + m_Radius * sin( i );

			r.left = CadMath::min( sx, r.left );
			r.top = CadMath::min( sy, r.top );
			r.right = CadMath::max( sx, r.right );
			r.bottom = CadMath::max( sy, r.bottom );
		}
	}

	r.left -= m_Style.GetLineWidth() / 2.0f;
	r.top -= m_Style.GetLineWidth() / 2.0f;
	r.right += m_Style.GetLineWidth() / 2.0f;
	r.bottom += m_Style.GetLineWidth() / 2.0f;


	return r;
}

bool ObjectArc :: HitCheck( Point coord, double sens )
{
	// Check if point is within bound box
	if( ! GetBounds().DoIntersect(coord)) 
		return false;

	// Check if we are within the radius
	double r = sqrt( ( m_Center.x - coord.x ) * ( m_Center.x - coord.x ) + ( m_Center.y - coord.y ) * ( m_Center.y - coord.y ) );

	if( ! ( r > m_Radius - ( m_Style.GetLineWidth() + sens ) ) && ( r < m_Radius + ( m_Style.GetLineWidth() + sens ) ) )
		return false;

	// Check we are within the angles
	double angle = CadMath::GetAngle( m_Center, coord );
	return CadMath::BetweenAngles( m_StartAngle, m_EndAngle, angle );
}

Object* ObjectArc :: Duplicate()
{
	Object* obj = new ObjectArc( m_Center, m_Radius, m_StartAngle, m_EndAngle );
	obj->SetStyle( m_Style );
	obj->SetLayer( m_Layer );

	return obj;
}

bool ObjectArc :: SetProperty( int which, Variant& value )
{
	switch( which )
	{
		case 0:  // Center
			m_Center = value.AsPoint();
			break;
		case 1:  // Radius
			m_Radius = value.AsDouble();
			break;
		case 2:  // Start angle
			m_StartAngle = value.AsDouble();
			break;
		case 3:  // End angle
			m_EndAngle = value.AsDouble();
			break;
		default:
			return false;
	}

	return true;
}

bool ObjectArc :: GetProperty( int which, Variant& value )
{
	switch( which )
	{
		case 0:  // Center
			value.SetPoint( m_Center );
			break;
		case 1:  // Radius
			value.SetDouble( m_Radius );
			break;
		case 2:  // Start angle
			value.SetDouble( m_StartAngle );
			break;
		case 3:  // End angle
			value.SetDouble( m_EndAngle );
			break;
		default:
			return false;
	}

	return true;
}

bool ObjectArc :: GetSnapPoint( int type, Point p, vector< Point >& list )
{
	if( type & OSNAP_ENDPOINT )
	{
		list.push_back( Point( m_Radius * cos( m_StartAngle ) + m_Center.x, m_Radius * sin( m_StartAngle ) + m_Center.y ) );
		list.push_back( Point( m_Radius * cos( m_EndAngle ) + m_Center.x, m_Radius * sin( m_EndAngle ) + m_Center.y ) );
	}

	if( type & OSNAP_MIDPOINT )
	{
		double a;

		if( m_EndAngle > m_StartAngle )
			a = ( m_EndAngle + m_StartAngle ) / 2.0f;
		else
			a = ( 2.0f * M_PI - m_StartAngle + m_EndAngle ) / 2.0f + m_StartAngle;

		CadMath::FixAngle( a );

		list.push_back( Point( m_Radius * cos( a ) + m_Center.x, m_Radius * sin( a ) + m_Center.y ) );
	}

	if( type & OSNAP_CENPOINT )
	{
		list.push_back( m_Center );
	}

	if( type & OSNAP_QUAPOINT )
	{
		Point q;
		double a;

		q = Point( m_Center.x, m_Center.y - m_Radius );
		a = CadMath::GetAngle( m_Center, q );
		if( CadMath::BetweenAngles( m_StartAngle, m_EndAngle, a ) )
			list.push_back( q );

		q = Point( m_Center.x, m_Center.y + m_Radius );
		a = CadMath::GetAngle( m_Center, q );
		if( CadMath::BetweenAngles( m_StartAngle, m_EndAngle, a ) )
			list.push_back( q );

		q = Point( m_Center.x - m_Radius, m_Center.y );
		a = CadMath::GetAngle( m_Center, q );
		if( CadMath::BetweenAngles( m_StartAngle, m_EndAngle, a ) )
			list.push_back( q );

		q = Point( m_Center.x + m_Radius, m_Center.y );
		a = CadMath::GetAngle( m_Center, q );
		if( CadMath::BetweenAngles( m_StartAngle, m_EndAngle, a ) )
			list.push_back( q );
	}

	
	if( type & OSNAP_NEAPOINT )
	{
		double angle = CadMath::GetAngle( m_Center, p );
		
		if( !CadMath::BetweenAngles( m_StartAngle, m_EndAngle, angle ) )
		{
			if( fabs( m_StartAngle - angle ) < fabs( m_EndAngle - angle ) )
				angle = m_StartAngle;
			else
				angle = m_EndAngle;
		}

		list.push_back( Point( m_Center.x + m_Radius * cos( angle ), m_Center.y + m_Radius * sin( angle ) ) );
	}

	return false;
}

String ObjectArc :: Save()
{
	String data;
	data.Format( "OA, %f, %f, %f, %f, %f", m_Center.x, m_Center.y, m_Radius, m_StartAngle, m_EndAngle );
	data += SaveAttributes();
	return String( data );  
}

bool ObjectArc :: Load( Document* doc, Layer* current_layer, vector< String >& split, bool old_ocadis )
{
	assert( doc != NULL );
	assert( current_layer != NULL );

	Object* obj;

	// Should lod it according to new or old ocadis
	if( old_ocadis )
	{
		// Make sure we have the correct number of arguments
		obj = new ObjectArc( Point( atof( split[1].c_str() ), atof( split[2].c_str() ) ), 
							 atof( split[3].c_str() ),
						     CadMath::ConvertDeg2Rad( atof( split[4].c_str() ) ), CadMath::ConvertDeg2Rad( atof( split[5].c_str() ) ) );
	}
	else
	{
		// Make sure we have the correct number of arguments
		obj = new ObjectArc( Point( atof( split[1].c_str() ), atof( split[2].c_str() ) ), 
							 atof( split[3].c_str() ), 
							 atof( split[4].c_str() ), atof( split[5].c_str() ) );
	}

	obj->SetLayer( current_layer );
	obj->GetStyle().SetStyle( 6, split );

	current_layer->AddObject( obj );

	return true;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////7
//
// P R I V A T E   M E T H O D S
//
/////////////////////////////////////////////////////////////////////////////////////////////////////7

void ObjectArc :: _Calc3P( os::Point& p1, os::Point& p2 )
{
	p1.x = m_Center.x + m_Radius * cos( m_StartAngle );
	p1.y = m_Center.y + m_Radius * sin( m_StartAngle );

	p2.x = m_Center.x + m_Radius * cos( m_EndAngle );
	p2.y = m_Center.y + m_Radius * sin( m_EndAngle );
}
