// oCADis (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <util/string.h>

#include "cadmath.h"
#include "style.h"
#include "dialogs/dialog_style.h"

using namespace os;


DialogStyle :: DialogStyle( DialogSideBar* parent ) : Dialog( parent, String( "Style" ) )
{
	// Create the table
	m_Table = new TableView( Rect(), "", 2, 4, false );
	m_Table->SetRowSpacings( 4 );
	AddChild( m_Table );

	// COLOUR FRAME

	// Create the colour frame
	m_ColourFrame = new ToolboxFrame( Rect(0,0,100,100), "Colour", "Colour" );
	m_Table->Attach( m_ColourFrame, 0, 2, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create colour table
	m_ColourTable = new TableView( Rect(), "", 1, 4, false );
	m_ColourTable->SetRowSpacings( 4 );
	m_ColourTable->SetRowSpacing( 2, 0 );
	m_ColourFrame->AddTableView( m_ColourTable );

	// Create widgets in colour table
	m_ColourLayer = new RadioButton( Rect(), "1", "By layer", new Message( MSG_COLOUR_LAYER ) );
	m_ColourTable->Attach( m_ColourLayer, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m_ColourBlock = new RadioButton( Rect(), "2", "By block", new Message( MSG_COLOUR_BLOCK ) );
	m_ColourTable->Attach( m_ColourBlock, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m_ColourUser = new RadioButton( Rect(), "3", "Userdefined", new Message( MSG_COLOUR_USER ) );
	m_ColourTable->Attach( m_ColourUser, 0, 1, 2, 3, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m_Colour = new ColourButton( Rect(), new Message( MSG_COLOUR ) );
	m_ColourTable->Attach( m_Colour, 0, 1, 3, 4, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// LINEWIDTH FRAME

	// Create the linewidth frame
	m_WidthFrame = new ToolboxFrame( Rect(), "Linewidth", "Linewidth" );
	m_Table->Attach( m_WidthFrame, 0, 2, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create linewidth table
	m_WidthTable = new TableView( Rect(), "", 1, 4, false );
	m_WidthTable->SetRowSpacings( 4 );
	m_WidthTable->SetRowSpacing( 2, 0 );
	m_WidthFrame->AddTableView( m_WidthTable );

	// Create widgets in width table
	m_WidthLayer = new RadioButton( Rect( ), "1", "By layer", new Message( MSG_WIDTH_LAYER ) );
	m_WidthTable->Attach( m_WidthLayer, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m_WidthBlock = new RadioButton( Rect(), "2", "By block", new Message( MSG_WIDTH_BLOCK ) );
	m_WidthTable->Attach( m_WidthBlock, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m_WidthUser = new RadioButton( Rect(), "3", "Userdefined", new Message( MSG_WIDTH_USER ) );
	m_WidthTable->Attach( m_WidthUser, 0, 1, 2, 3, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m_Width = new RTSLTextView( Rect(), "", new Message( MSG_WIDTH ) );
	m_Width->SetMinWidth( 60 );
	m_Width->SetText( "0" );
	m_WidthTable->Attach( m_Width, 0, 1, 3, 4, ( TABLE_SHRINK | TABLE_EXPAND | TABLE_FILL ), 0 );

	// PATTERN FRAME

	// Create the pattern frame
	m_PatternFrame = new ToolboxFrame( Rect(), "Pattern", "Pattern" );
	m_Table->Attach( m_PatternFrame, 0, 2, 2, 3, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create pattern table
	m_PatternTable = new TableView( Rect(), "", 1, 4, false );
	m_PatternTable->SetRowSpacings( 4 );
	m_PatternFrame->AddTableView( m_PatternTable );

	// Create widgets in colour table
	m_PatternLayer = new RadioButton( Rect( ), "1", "By layer", new Message( MSG_PATTERN_LAYER ) );
	m_PatternTable->Attach( m_PatternLayer, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m_PatternBlock = new RadioButton( Rect(), "2", "By block", new Message( MSG_PATTERN_BLOCK ) );
	m_PatternTable->Attach( m_PatternBlock, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m_PatternUser = new RadioButton( Rect(), "3", "Userdefined", new Message( MSG_PATTERN_USER ) );
	m_PatternTable->Attach( m_PatternUser, 0, 1, 2, 3, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m_Pattern = new DropdownMenu( Rect(), "" );
	m_Pattern->SetSelectionMessage( new Message( MSG_PATTERN ) );
	m_Pattern->SetReadOnly();
	m_PatternTable->Attach( m_Pattern, 0, 1, 3, 4, ( TABLE_EXPAND | TABLE_FILL ), 0 );
}

DialogStyle ::~DialogStyle()
{
	delete m_ColourFrame;
	delete m_ColourTable;
	delete m_ColourLayer;
	delete m_ColourBlock;
	delete m_ColourUser;
	delete m_Colour;

	delete m_WidthFrame;
	delete m_WidthTable;
	delete m_WidthLayer;
	delete m_WidthBlock;
	delete m_WidthUser;
	delete m_Width;

	delete m_PatternFrame;
	delete m_PatternTable;
	delete m_PatternLayer;
	delete m_PatternBlock;
	delete m_PatternUser;
	delete m_Pattern;

	delete m_Table;
}

void DialogStyle :: SetFrame( const Rect& cRect, bool bNotifyServer )
{
	Dialog::SetFrame( cRect, bNotifyServer );
	_Layout();
}

void DialogStyle :: AllAttached( void )
{
	Dialog::AllAttached();

	m_ColourLayer->SetTarget( this );
	m_ColourBlock->SetTarget( this );
	m_ColourUser->SetTarget( this );
	m_Colour->SetTarget( this );

	m_WidthLayer->SetTarget( this );
	m_WidthBlock->SetTarget( this );
	m_WidthUser->SetTarget( this );
	m_Width->SetTarget( this );

	m_PatternLayer->SetTarget( this );
	m_PatternBlock->SetTarget( this );
	m_PatternUser->SetTarget( this );
	m_Pattern->SetTarget( this );
}

void DialogStyle :: HandleMessage( Canvas* canvas, Document* doc, os::Message* pcMessage )
{
	Style style = doc->GetCurrentStyle();

	switch( pcMessage->GetCode() )
	{
		case MSG_COLOUR_LAYER:
			style.SetColourType( Style::BY_LAYER );
			m_Colour->SetEnable( false );
			break;
		case MSG_COLOUR_BLOCK:
			style.SetColourType( Style::BY_BLOCK );
			m_Colour->SetEnable( false );
			break;
		case MSG_COLOUR_USER:
			{
				Color32_s col = m_Colour->GetColour();
				style.SetColourType( Style::BY_USER );
				style.SetColour( col.red / 255.0f, col.blue / 255.0f, col.green / 255.0f );
				m_Colour->SetEnable( true );
				break;
			}
		case MSG_COLOUR:
			{
				if( m_ColourUser->GetValue().AsBool() )
				{
					Color32_s col = m_Colour->GetColour();
					style.SetColourType( Style::BY_USER );
					style.SetColour( col.red / 255.0f, col.blue / 255.0f, col.green / 255.0f );
				}
				else
					m_Colour->SetEnable( false );
				break;
			}

		case MSG_WIDTH_LAYER:
			m_Width->SetEnable( false );
			style.SetLineWidthType( Style::BY_LAYER );
			break;
		case MSG_WIDTH_BLOCK:
			m_Width->SetEnable( false );
			style.SetLineWidthType( Style::BY_BLOCK );
			break;
		case MSG_WIDTH_USER:
		{
			double width = CadMath::Strtod( m_Width->GetText().c_str() );
			style.SetLineWidthType( Style::BY_USER );
			style.SetLineWidth( width );
			m_Width->SetEnable( true );
			break;
		}
		case MSG_WIDTH:
		{
			if( m_WidthUser->GetValue().AsBool() )
			{
				double width = CadMath::Strtod( m_Width->GetText().c_str() );
				style.SetLineWidthType( Style::BY_USER );
				style.SetLineWidth( width );
				m_Width->SetEnable( true );
			}
			else
				m_Width->SetEnable( false );
			break;
		}

		case MSG_PATTERN_LAYER:
			style.SetPatternType( Style::BY_LAYER );
			m_Pattern->SetEnable( false );
			break;
		case MSG_PATTERN_BLOCK:
			style.SetPatternType( Style::BY_BLOCK );
			m_Pattern->SetEnable( false );
			break;
		case MSG_PATTERN_USER:
		{
			style.SetPatternType( Style::BY_USER );
			style.SetPatternType( Style::BY_USER );
			int sel = m_Pattern->GetSelection();
			if( sel < 0 )
				sel = 0;

			style.SetPattern( doc->GetPattern( sel ) );
			m_Pattern->SetEnable( true );
			break;
		}
		case MSG_PATTERN:
		{
			bool final;
			int32 selection;

			if( pcMessage->FindBool( "final", &final ) < 0 )
				break;

			if( pcMessage->FindInt32( "selection", &selection ) < 0 )
				break;

			if( final && selection >= 0 )
			{
				style.SetPatternType( Style::BY_USER );
				style.SetPattern( doc->GetPattern( selection ) );
				m_Pattern->SetEnable( true );	
			}
			break;
		}
	}

	doc->SetCurrentStyle( style );
}

void DialogStyle :: Init( Document* doc )
{
	_Update( doc );
}

/////////////////////////////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E   M E T H O D S
//
/////////////////////////////////////////////////////////////////////////////////////////////////////

void DialogStyle :: _Layout()
{
	Rect cFrame = ConvertFromParent( GetFrame() );
	m_Table->SetFrame( cFrame );	

	Rect rec = m_Colour->GetFrame();
	rec.left += 14;
	m_Colour->SetFrame( rec );

	rec = m_Width->GetFrame();
	rec.left += 14;
	m_Width->SetFrame( rec );

	rec = m_Pattern->GetFrame();
	rec.left += 14;
	m_Pattern->SetFrame( rec );
}

void DialogStyle :: _Update( Document* doc )
{
	Style style = doc->GetCurrentStyle();
	double r, g, b;

 	m_ColourLayer->SetValue( style.GetColourType() == Style::BY_LAYER );
	m_ColourBlock->SetValue( style.GetColourType() == Style::BY_BLOCK );
	m_ColourUser->SetValue( style.GetColourType() == Style::BY_USER );
	if( style.GetColourType() == Style::BY_USER )
	{
		style.GetColour( r, b, g );
		m_Colour->SetColour( Color32_s( (int) ( r * 255.0f ), (int) ( g * 255.0f ) , (int) ( b * 255.0f ) ) );
		m_Colour->SetEnable( true );
	}
	else
		m_Colour->SetEnable( false );

 	m_WidthLayer->SetValue( style.GetLineWidthType() == Style::BY_LAYER );
	m_WidthBlock->SetValue( style.GetLineWidthType() == Style::BY_BLOCK );
	m_WidthUser->SetValue( style.GetLineWidthType() == Style::BY_USER );
	if( style.GetLineWidthType() == Style::BY_USER )
	{
		String d;

		d.Format( "%f", style.GetLineWidth() );
		m_Width->SetText( d );

		m_Width->SetEnable( true );
	}
	else
		m_Width->SetEnable( false );

 	m_PatternLayer->SetValue( style.GetPatternType() == Style::BY_LAYER );
	m_PatternBlock->SetValue( style.GetPatternType() == Style::BY_BLOCK );
	m_PatternUser->SetValue( style.GetPatternType() == Style::BY_USER );

	// Set up pattern menu
	_UpdatePattern( doc );

	if( style.GetPatternType() == Style::BY_USER )
	{
		m_Pattern->SetEnable( true );
	}
	else
		m_Pattern->SetEnable( false );
}

void DialogStyle :: _UpdatePattern( Document* doc )
{
	m_Pattern->Clear();

	for( int i = 0 ; i < doc->GetPatternCount() ; i++ )
	{
		Pattern* pattern = doc->GetPattern( i );
		m_Pattern->AppendItem( pattern->GetName() );
	}

	m_Pattern->SetSelection(0, false );
}
