// oCADis (C)opyright 2007 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "windialogs/windia_addlayer.h"
#include "windialogs/windia_alert.h"
#include "postoffice.h"
#include "messages.h"
#include "main.h"
#include "cadmath.h"

using namespace os;

WinDiaAddLayer :: WinDiaAddLayer( const Rect& cFrame, Document* current ) : Window( cFrame, "add_layer", "Add Layer - oCADis" )
{
	// Create the mailbox
	AddMailbox( "Document" );

	// Create the table
	m_theTable = new TableView( Rect(), "", 1, 2, false );
	m_theTable->SetRowSpacings( 4 );
	AddChild( m_theTable );
	
	//////////////////////////////////////////////
	// Create the attribute frame
	m_AttrFrame = new ToolboxFrame( Rect(0,0,100,100), "Attributes", "Attributes" );
	m_theTable->Attach( m_AttrFrame, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create attribute table
	m_AttrTable = new TableView( Rect(), "", 2, 3, false );
	m_AttrTable->SetRowSpacings( 4 );
	m_AttrFrame->AddTableView( m_AttrTable );

	m_LabelName = new StringView( Rect(), "", "Name: " );
	m_AttrTable->Attach( m_LabelName, 0, 1, 0, 1, 0, 0 );
	m_Name = new TextView( Rect(), "", "" );
	m_AttrTable->Attach( m_Name, 1, 2, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m_Visible = new CheckBox( Rect(), "", "Layer is visible", NULL );
	m_Visible->SetValue( true );
	m_AttrTable->Attach( m_Visible, 0, 2, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m_Editable = new CheckBox( Rect(), "", "Layer is editable", NULL );
	m_Editable->SetValue( true );
	m_AttrTable->Attach( m_Editable, 0, 2, 2, 3, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	//////////////////////////////////////////////
	// Create the style frame
	m_StyleFrame = new ToolboxFrame( Rect(0,0,100,100), "Style", "Style" );
	m_theTable->Attach( m_StyleFrame, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create style table
	m_StyleTable = new TableView( Rect(), "", 2, 4, false );
	m_StyleTable->SetRowSpacings( 4 );
	m_StyleFrame->AddTableView( m_StyleTable );

	// Create colour
	m_LabelColour = new StringView( Rect(), "", "Colour: " );
	m_StyleTable->Attach( m_LabelColour, 0, 1, 0, 1, 0, 0 );
	m_Colour = new ColourButton( Rect(), new Message( MSG_COLOUR ) );
	m_StyleTable->Attach( m_Colour, 1, 2, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create linewidth
	m_LabelLinewidth = new StringView( Rect(), "", "Linewidth: " );
	m_StyleTable->Attach( m_LabelLinewidth, 0, 1, 1, 2, 0, 0 );
	m_Linewidth = new TextView( Rect(), "", "0" );
	m_StyleTable->Attach( m_Linewidth, 1, 2, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create pattern
	m_LabelPattern = new StringView( Rect(), "", "Pattern: " );
	m_StyleTable->Attach( m_LabelPattern, 0, 1, 2, 3, 0, 0 );
	m_Pattern = new DropdownMenu( Rect(), "" );
	m_Pattern->SetSelectionMessage( new Message( MSG_PATTERN ) );
	m_Pattern->SetReadOnly();
	m_StyleTable->Attach( m_Pattern, 1, 2, 2, 3, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	//////////////////////////////////////////////
	// Create the button table
	m_ButtonTable = new TableView( Rect(), "", 3, 1, true );
	m_ButtonTable->SetColumnSpacings( 5 );	
	AddChild( m_ButtonTable );

	// Create the buttons
	m_Apply = new Button( Rect(), "", "Apply", new Message( MSG_APPLY ) );
	m_ButtonTable->Attach( m_Apply, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );	

	m_OK = new Button( Rect(), "", "OK", new Message( MSG_OK ) );
	m_ButtonTable->Attach( m_OK, 1, 2, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );	

	m_Cancel = new Button( Rect(), "", "Cancel", new Message( MSG_CANCEL ) );
	m_ButtonTable->Attach( m_Cancel, 2, 3, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );	
	
	// Fix layout
	_Layout();

	// Close dialog with escape
	ShortcutKey s( VK_ESCAPE ); 
	AddShortcut( s, new Message( MSG_CANCEL ) );

	// Init pointers
	m_CurrentDocument = current;
	_UpdatePattern();
}

WinDiaAddLayer :: ~WinDiaAddLayer()
{
	// Remove the mailbox
	RemoveMailbox( "Document" );

	delete m_theTable;
	delete m_AttrFrame;
	delete m_AttrTable;
	delete m_LabelName;
	delete m_Name;
	delete m_Visible;
	delete m_Editable;
	delete m_StyleFrame;
	delete m_StyleTable;
	delete m_LabelColour;
	delete m_Colour;
	delete m_LabelLinewidth;
	delete m_Linewidth;
	delete m_LabelPattern;
	delete m_Pattern;
	delete m_ButtonTable;
	delete m_Apply;
	delete m_OK;
	delete m_Cancel;
}

void WinDiaAddLayer :: HandleMessage( os::Message* pcMessage )
{
	switch( pcMessage->GetCode() )	//Get the message code from the message
	{		
		case MSG_APPLY:
			_Apply();
			break;
		case MSG_OK:
			if( _Apply() )
				Hide();
			break;
		case MSG_CANCEL:
			Hide();
			break;

		// Broadcast messages
		case MSG_BROADCAST_NEW_DOCUMENT:
			if( pcMessage->FindPointer( "document", (void**) &m_CurrentDocument ) != EOK )
				m_CurrentDocument = NULL;
			_UpdatePattern();
			break;
		default:
		{
			Window::HandleMessage( pcMessage );
			break;
		}
	}
}

bool WinDiaAddLayer ::OkToQuit( void )
{
	Hide();
	return false;
}

void WinDiaAddLayer :: FrameSized( const os::Point& delta )
{
	Window::FrameSized( delta );
	_Layout();
}
	
void WinDiaAddLayer :: Raise()
{
	if( IsVisible() )
		Hide();

	Show();
	MakeFocus();
}

//////////////////////////////////////////////////////////////////////
//
// P R I V A T E   M E T H O D S
//
//////////////////////////////////////////////////////////////////////

void WinDiaAddLayer :: _Layout()
{
	Rect cFrame = GetBounds();
	cFrame.Resize( 2, 2, -2, -2 );

	// Set button table
	Point tsize = m_ButtonTable->GetPreferredSize( false );
	Rect tframe( cFrame.right - tsize.x, cFrame.bottom - tsize.y, cFrame.right, cFrame.bottom );
	m_ButtonTable->SetFrame( tframe );

	// Set table
	Rect t1frame( cFrame.left, cFrame.top, cFrame.right, tframe.top - 2.0f );
	m_theTable->SetFrame( t1frame );
}

void WinDiaAddLayer :: _UpdatePattern()
{
	m_Pattern->Clear();

	if( m_CurrentDocument == NULL )
		return;

	for( int i = 0 ; i < m_CurrentDocument->GetPatternCount() ; i++ )
	{
		Pattern* pattern = m_CurrentDocument->GetPattern( i );
		m_Pattern->AppendItem( pattern->GetName() );
	}

	m_Pattern->SetSelection(0, false );
}

bool WinDiaAddLayer :: _Apply()
{
	if( m_CurrentDocument == NULL )
	{
		WinDiaAlert::ShowErrorAlert( "Add new layer", "No valid document." );
		return false;
	}

	// Check name
	String name;
	name = m_Name->GetValue().AsString().Strip();

	// Check for empty string
	if( name == "" )
	{
		WinDiaAlert::ShowErrorAlert( "Add new layer", "A layer must have a name." );
		return false;
	}

	// Check for name duplicates
	for( int i = 0 ; i < m_CurrentDocument->GetLayerCount() ; i++ )
	{
		Layer* layer = m_CurrentDocument->GetLayer( i );

		if( layer->GetName() == name )
		{
			WinDiaAlert::ShowErrorAlert( "Add new layer", "A layer with this name already exists." );
			return false;
		}
	}

	// Extract the other variables
	double linewidth = CadMath :: Strtod( m_Linewidth->GetValue().AsString() );
	int sel = m_Pattern->GetSelection();
	if( sel < 0 )
	{
		WinDiaAlert::ShowErrorAlert( "Add new layer", "No pattern selected." );
		return false;
	}
	Pattern* pattern = m_CurrentDocument->GetPattern( sel );

	// Lets create our layer
	Layer* layer = new Layer( name, m_Colour->GetColour(), linewidth, pattern, m_Visible->GetValue().AsBool(), m_Editable->GetValue().AsBool() );

	m_CurrentDocument->AddLayer( layer );

	m_CurrentDocument->SetCurrentLayer( layer );

	Message* msg = new Message( MSG_BROADCAST_ADD_LAYER  );
	msg->AddPointer( "document", (void*) m_CurrentDocument );
	msg->AddPointer( "layer", (void*) layer );
	
	Mail( "AddLayer", msg );

	return true;
}
