// oCADis (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <math.h>

#include "tool_circle_state_4.h"
#include "common.h"
#include "object_circle.h"

using namespace os;

///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E
//
///////////////////////////////////////////////////////////////////////////////

class CircleState4 :: _Private
{
public:
	_Private(){};

	void _Layout( Rect cFrame )
	{
		m_Table->SetFrame( Rect( 0, 0, cFrame.Width(), cFrame.Height() ) );
	}

	void _CalculateCircle( Point first, Point second, Point& center, float& radius )
	{		
		center.x = ( second.x - first.x ) / 2.0f + first.x;
		center.y = ( second.y - first.y ) / 2.0f + first.y;
		Point dist = second - first;
		radius = sqrt( dist.x * dist.x + dist.y * dist.y ) / 2.0f;
	}

	enum CircleState4Message { MSG_POINT, MSG_CANCEL };

	ToolboxPoint* m_SecondPoint;
	ToolboxAction* m_Action;

	os::TableView* m_Table;

};

///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S
//
///////////////////////////////////////////////////////////////////////////////
CircleState4 :: CircleState4( ToolCircle* theTool, ToolboxSideBar* toolbox ) : ToolState( theTool, toolbox )
{
	// Create the private class
	m = new _Private();

	// Create the table
	m->m_Table = new TableView( Rect(), "", 1, 2, false );
	m->m_Table->SetRowSpacings( 4 );
	AddChild( m->m_Table );

	// Create buttons and attach them to table
	m->m_SecondPoint = new ToolboxPoint( Rect(), "Second point", "X: ", "Y: ", new Message( _Private::MSG_POINT ), true );
	m->m_Table->Attach( m->m_SecondPoint, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m->m_Action = new ToolboxAction( Rect(), "" );
	m->m_Table->Attach( m->m_Action, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), ( TABLE_EXPAND | TABLE_FILL ) );

	// Add action items
	m->m_Action->AddButton( "Cancel", (Image*) LoadImage( "cancel.png" ), new Message( _Private::MSG_CANCEL ) , "Cancel the tool and return to main toolbox " );

	State::SetToolbox( this );
}

CircleState4 :: ~CircleState4()
{	
	delete m;
}

void CircleState4 :: Init( Canvas* canvas, Document* doc )
{
	m->m_SecondPoint->SetOrigin( GetTool()->GetPropertyAsPoint( "first_point" ) );
	canvas->PrepareForXORMode();
}

void CircleState4 :: GotFocus( Canvas* canvas, Document* doc, Point coord )
{
	coord = doc->MakeSnap( GetTool()->GetPropertyAsPoint( "first_point" ), coord );
	m->m_SecondPoint->SetValue( coord );
}

void CircleState4 :: LostFocus( Canvas* canvas, Document* doc, Point coord )
{
	m->m_SecondPoint->ClearValue();
}

void CircleState4 :: MouseMove( Canvas* canvas, Document* doc, Point coord )
{
	coord = doc->MakeSnap( GetTool()->GetPropertyAsPoint( "first_point" ), coord );
	m->m_SecondPoint->SetValue( coord );

	Point center;
	float radius;

	canvas->SetStyle( doc->GetCurrentStyle() );

	m->_CalculateCircle( GetTool()->GetPropertyAsPoint( "first_point" ), coord, center, radius );

	canvas->DrawCircle( center, radius, true );
}

String CircleState4 :: GetStatusText()
{
	return String( "Select second point of the circle" );
}

void CircleState4 :: PointSelected( Canvas* canvas, Document* doc, Point coord )
{
	coord = doc->MakeSnap( GetTool()->GetPropertyAsPoint( "first_point" ), coord );
	canvas->Undraw();

	Point center;
	float radius;

	m->_CalculateCircle( GetTool()->GetPropertyAsPoint( "first_point" ), coord, center, radius );

	// Create object
	ObjectCircle* obj = new ObjectCircle( center, radius );
	doc->AddObject( obj );
	obj->Draw( canvas );

	// Create Undo object
	UndoAddObject* undo = new UndoAddObject( "Create Circle", obj );
	doc->AddUndoAction( undo );
	doc->SetModification();

	GetTool()->SetState( 3 );
}

void CircleState4 :: SetFrame( const Rect& cRect, bool bNotifyServer )
{
	Toolbox::SetFrame( cRect, bNotifyServer );
	m->_Layout( cRect );
}

void CircleState4 :: AllAttached( void )
{
	Toolbox::AllAttached();

	m->m_SecondPoint->SetTarget( this );
	m->m_Action->SetTarget( this );
}

void CircleState4 :: HandleMessage( Canvas* canvas, Document* doc, Message* pcMessage )
{
	switch( pcMessage->GetCode() )
	{
		case _Private::MSG_CANCEL:
			SetMainToolbox();
			break;
		case _Private::MSG_POINT:
			{
				Point p;

				if( pcMessage->FindPoint( "value", &p ) == EOK )
					PointSelected( canvas, doc, p );
		
				break;
			}
	}
}
