// oCADis (C)opyright 2007 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

// Filter for postscript export

#include <string>

#include "filters/filter_ps.h"
#include "main.h"
#include "cadmath.h"

using namespace os;
using namespace std;
using namespace CadMath;

#define WEIGHT_RED 0.299f
#define WEIGHT_GREEN 0.587f
#define WEIGHT_BLUE 0.114f

#define MM2UNIT 2.83464f

void FilterPS :: Save( String& path, Document& doc )
{
	// Setup global variables
	m_PageLandscape = false;
	m_PageHeight = 210.0f;
	m_PageWidth = 297.0f;
	m_PlotCenter = Point( 297.0f / 2.0f, 210.0f / 2.0f );
	m_PlotPan = Point( 0.0f, 0.0f );
	m_ZoomFactor = 1.0f;

	// Get bounding box of document
	Rect bound = doc.GetBoundingBox();
	m_PlotCenter = Point( bound.Width() / 2.0f + bound.left, bound.Height() / 2.0f + bound.top );

	// Open file
	ofstream fout( path.c_str() );

	// Check that file is opened
	if( !fout.is_open() )
		throw String( "Unable to save file\n" );

	try
	{
		_GenerateHeader( fout, doc );
		_GenerateObjects( fout, doc );
		_GenerateEOF( fout, doc );
	}
	catch( ... )
	{
		printf( "eeror!\n" );
	}
	fout.flush();
	fout.close();
}

String FilterPS :: GetExtension()
{
	return String( "ps" );
}

String FilterPS ::GetMenuString()
{
	return String( "PostScript" );
}

///////////////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E   M E T H O D S
//
///////////////////////////////////////////////////////////////////////////////////////

void FilterPS :: _GenerateHeader( ofstream& fout, Document& doc )
{
	fout << "%!PS-Adobe-3.0\n";

	string title( doc.GetTitle() );

	fout << "%%Title: " << title << "\n";

 	fout << "%%Creator: oCADis\n";
 
	fout << "%%Pages: 1\n";

 	fout << "%%EndComments\n";

	if( m_PageLandscape )
	{
		fout << m_PageHeight / 2.0f * MM2UNIT << " " << m_PageWidth / 2 * MM2UNIT << " translate\n";
		fout << "90 rotate\n";
	}
	else
		fout << m_PageWidth / 2.0f * MM2UNIT << " " << m_PageHeight / 2 * MM2UNIT << " translate\n";
}

void FilterPS :: _GenerateObjects( ofstream& fout, Document& doc )
{
	for( int i = 0 ; i < doc.GetLayerCount() ; i++ )
	{		
		Layer* layer = doc.GetLayer( i );

		if( layer->IsVisible() )
		{
			_SetLayer( fout, layer );

			for( int j = 0 ; j < layer->GetObjectCount() ; j++ )
			{
				Object* obj = layer->GetObject( j );

				bool changed = _SetObject( fout, obj );

				// Figure out object type
				if( typeid( *obj ) == typeid( ObjectLine ) )
				{
					_ObjectLine( fout, obj );
				}
				else if( typeid( *obj ) == typeid( ObjectCircle ) )
				{
					_ObjectCircle( fout, obj );
				}
				else if( typeid( *obj ) == typeid( ObjectArc ) )
				{
					_ObjectArc( fout, obj );
				}
				else
					printf( "Unknown object type %s\n", typeid( *obj ).name() );

				if( changed )
					fout << "  grestore\n";
			}

			// Restore graphic settings
			fout << "grestore   % End Layer: "<< string( layer->GetName() ) << "\n";
		}

	}
}

void FilterPS :: _GenerateEOF( ofstream& fout, Document& doc )
{
	fout <<  "showpage\n%%EOF\n";
}

double FilterPS :: _GetIntensity( double r, double g, double b )
{
  return WEIGHT_RED * r + WEIGHT_GREEN * g + WEIGHT_BLUE * b;
}

void FilterPS :: _SetLayer( ofstream& fout, Layer* layer )
{
	Style style = layer->GetStyle();

	fout << "gsave      % Start Layer: " << string( layer->GetName() ) << "\n";

	// Set colour of the layer
	double r,g,b;
	style.GetColour( r, g, b );
	fout << "  " << _GetIntensity( r, g, b ) << " setgray\n";

	// Set linewidth of the layer
	fout << "  " << _WorldToPS( style.GetLineWidth() ) << " setlinewidth\n";

	// Set pattern
	Pattern* pattern = style.GetPattern();

	if( pattern->IsSolid() )
		fout << "  [] 0 setdash\n";
	else
	{
		fout << "  [ ";

		for( int i = 0 ; i < pattern->GetDashesCount() ; i++ )
			fout << pattern->GetDash( i ) << " ";		// FIXME: Doesnt consider zooming! 

		fout << "] 0 setdash\n";
    }
}

bool FilterPS :: _SetObject( ofstream& fout, Object* obj )
{
	bool ret = false;

	Style style = obj->GetStyle();

	// Check colour
	if( style.GetColourType() == Style::BY_USER )
	{
		if( !ret )
			fout << "  gsave\n";

		double r,g,b;
		style.GetColour( r, g, b );
		fout << "    " << _GetIntensity( r, g, b ) << " setgray\n";

		ret = true;
	}

	// Check linewidth
	if( style.GetLineWidthType() == Style::BY_USER )
	{
		if( !ret )
			fout << "  gsave\n";

		fout << "    " << _WorldToPS( style.GetLineWidth() ) << " setlinewidth\n";

		ret = true;
	}

	// Check pattern
	if( style.GetPatternType() == Style::BY_USER )
	{
		if( !ret )
			fout << "  gsave\n";

		Pattern* pattern = style.GetPattern();

		if( pattern->IsSolid() )
			fout << "    [] 0 setdash\n";
		else
		{
			fout << "    [ ";

			for( int i = 0 ; i < pattern->GetDashesCount() ; i++ )
				fout << pattern->GetDash( i ) << " ";		// FIXME: Doesnt consider zooming! 

			fout << "] 0 setdash\n";
    	}

		ret = true;
	}
	return ret;
}

os::Point FilterPS :: _WorldToPS( Point wld )
{
	Point ret;

	ret.x = MM2UNIT*( ( wld.x - m_PlotCenter.x ) * m_ZoomFactor + m_PlotPan.x );
	ret.y = MM2UNIT*( ( m_PlotCenter.y - wld.y ) * m_ZoomFactor + m_PlotPan.y );

	return ret;
}

double FilterPS :: _WorldToPS( double value )
{
	return value * m_ZoomFactor * MM2UNIT;
}

///////////////////////////////////////////////////////////////////////////////
//
// O B J E C T S
//
///////////////////////////////////////////////////////////////////////////////

void FilterPS :: _ObjectLine( ofstream& fout, Object* line )
{
	Variant value;

	line->GetProperty( 0, value );
	Point p1 = _WorldToPS( value.AsPoint() );

	line->GetProperty( 1, value );
	Point p2 = _WorldToPS( value.AsPoint() );

	fout << "  newpath " << p1.x << " " << p1.y << " moveto " << p2.x << " " << p2.y << " lineto stroke\n";
}

void FilterPS :: _ObjectArc( ofstream& fout, Object* arc )
{
	Variant value;

	arc->GetProperty( 0, value );
	Point center = _WorldToPS( value.AsPoint() );

	arc->GetProperty( 1, value );
	double radius = _WorldToPS( value.AsDouble() );

	arc->GetProperty( 2, value );
	double start_angle = CadMath::ConvertRad2Deg( value.AsDouble() );

	arc->GetProperty( 3, value );
	double end_angle = CadMath::ConvertRad2Deg( value.AsDouble() );

	fout << "  newpath " << center.x << " " << center.y << " " << radius << " " << start_angle << " " << end_angle << " arc stroke\n";
}

void FilterPS :: _ObjectCircle( ofstream& fout, Object* circle )
{
	Variant value;

	circle->GetProperty( 0, value );
	Point center = _WorldToPS( value.AsPoint() );

	circle->GetProperty( 1, value );
	double radius = _WorldToPS( value.AsDouble() );

	fout << "  newpath " << center.x << " " << center.y << " " << radius << " 0 360 arc stroke\n";
}


