// oCADis (C)opyright 2007 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <gui/point.h>
#include "postoffice.h"
#include "messages.h"
#include "document.h"
#include "cadmath.h"
#include "main.h"

using namespace std;
using namespace os;
using namespace CadMath;

Document :: Document() : Locker( "" )
{
	m_CurrentLayer = NULL;
	m_NeedSave = false;
	m_ContinuePointAvailable = false;
	m_SnapActive = false;
	m_SnapOrigo = Point( 0.0f, 0.0f );
	m_SnapDistance = Point( 10.0f, 10.0f );
	m_SnapAngle = 0.0f; //M_PI / 4.0f; // 0.0f;
	m_GridActive = false;
	m_GridSkip = Point( 1.0f, 1.0f );
	m_OrthoMode = OrthoNone;
	m_Pan = Point( 0, 0 );
	m_ZoomFactor = 1.0f;
	m_OSnap = OSNAP_NONE;
}

Document :: ~Document()
{
	for( uint i = 0 ; i < m_ListOfLayers.size() ; i++ )
		delete m_ListOfLayers[ i ];
}

void Document :: Check()
{
	// Make sure there at least some linestyles
	if( GetPatternCount() < 1 )
	{
		// Create the default ones
		AddPattern( new Pattern( "Solid" ) );
		AddPattern( new Pattern( "DotLine", 2, 0.1f, 6.2f ) );
		AddPattern( new Pattern( "DotLine2", 2, 0.1f, 3.1f ) );
		AddPattern( new Pattern( "DotLineX2", 2, 0.1f, 12.4f ) );

		AddPattern( new Pattern( "DashLine", 2, 12.0f, 6.0f ) );
		AddPattern( new Pattern( "DashLine2", 2, 6.0f, 3.0f ) );
		AddPattern( new Pattern( "DashLineX2" ,2, 24.0f, 12.0f ) );

		AddPattern( new Pattern( "DashDotLine", 4, 12.0f, 5.95f, 0.1f, 5.95f ) );
		AddPattern( new Pattern( "DashDotLine2", 4, 6.0f, 2.95f, 0.1f, 2.95f ) );
		AddPattern( new Pattern( "DashDotLineX2", 4, 24.0f, 11.95f, 0.1f, 11.95f ) );

		AddPattern( new Pattern( "DivideLine", 6, 12.0f, 5.9f, 0.15f, 5.9f, 0.15f, 5.9f ) );
		AddPattern( new Pattern( "DivideLine2", 6, 6.0f, 2.9f, 0.15f, 2.9f, 0.15f, 2.9f ) );
		AddPattern( new Pattern( "DivideLineX2", 6, 24.0f, 11.9f, 0.15f, 11.9f, 0.15f, 11.9f ) );

		AddPattern( new Pattern( "CenterLine", 4, 32.0f, 6.0f, 6.0f, 6.0f ) );
		AddPattern( new Pattern( "CenterLine2", 4, 16.0f, 3.0f, 3.0f, 3.0f ) );
		AddPattern( new Pattern( "CenterLineX2", 4, 64.0f, 12.0f, 12.0f, 12.0f ) );

		AddPattern( new Pattern( "BorderLine", 6, 12.0f, 6.0f, 12.0f, 5.95f, 0.1f, 5.95f ) );
		AddPattern( new Pattern( "BorderLine2", 6, 6.0f, 3.0f, 6.0f, 2.95f, 0.1f, 2.95f ) );
		AddPattern( new Pattern( "BorderLineX2", 6, 24.0f, 12.0f, 24.0f, 11.95f, 0.1f, 11.95f ) );

		AddPattern( new Pattern( "BlockLine", 2, 0.5f, 0.5f ) );
	}

	// Make sure there is at least one layer
	if( GetLayerCount() < 1 )
	{
		AddLayer( new Layer( "Default layer", GetPattern( 0 ) ) );
	}

	// Set up pointers
	if( m_CurrentLayer == NULL )
		m_CurrentLayer = GetLayer( 0 );
}
void Document :: SetCurrentLayer( Layer* layer )
{
	m_CurrentLayer = layer;
}

void Document :: AddLayer( Layer* layer )
{
	m_ListOfLayers.push_back( layer );

//	Message* msg = new Message( MSG_BROADCAST_ADD_LAYER );
//	msg->AddPointer( "layer", (void**) &layer );
	
//	Broadcast( msg );

}

void Document :: AddObject( Object* obj )
{
	if( m_CurrentLayer != NULL )
	{
		obj->SetStyle( m_CurrentStyle );
		obj->SetLayer( m_CurrentLayer );
		m_CurrentLayer->AddObject( obj );
	}
	
}

void Document :: AddObject( Layer* layer, Object* obj )
{
	if( m_CurrentLayer != NULL )
	{
		obj->SetLayer( layer );
		layer->AddObject( obj );
	}
}

void Document :: Redraw( Canvas* canvas )
{
	Lock();

	for( uint i = 0 ; i < m_ListOfLayers.size(); i++ )
		m_ListOfLayers[ i ]->Redraw( canvas );

	Unlock();
}

Pattern* Document :: GetPattern( String name )
{
	for( uint i = 0 ; i < m_ListOfPatterns.size() ; i++ )
	{
		if( m_ListOfPatterns[ i ]->GetName() == name )
			return m_ListOfPatterns[ i ];
	}
	
	return NULL;
}

void Document :: SetModification( bool save )
{ 
	if( m_NeedSave != save )
	{
		m_NeedSave = save; 

		// Update the title
		Message* msg = new Message( MSG_BROADCAST_UPDATE_TITLE );
		msg->AddString( "title", GetTitle() );
		Mail( "Title", msg );
	}
}

String Document :: GetTitle()
{
	String p;

	if( m_Path != "" )
		p = m_Path;
	else
		p = "Untitled";

	if( m_NeedSave )
		p = String( "*" ) + p;

	p += " - oCADis";
	return p;
}

Object* Document :: HitCheck( Point coord, double sens )
{
	// For each layer
	for( int i = 0 ; i < GetLayerCount() ; i ++ )
	{
		Layer* layer = GetLayer( i );

		// Check layer only if it is visible
		if( layer->IsVisible() )
		{
			// Check all objects in the layer
			for( int j = 0 ; j < layer->GetObjectCount();  j++ )
			{
				Object* object = layer->GetObject( j ) ;

				if( object->HitCheck( coord, sens ) )
					return object;
			}	
		}
	}

	// No object found
	return NULL;
}

void Document :: HitCheck( Rect box, vector< Object* >& hits )
{
	// For each layer
	for( int i = 0 ; i < GetLayerCount() ; i ++ )
	{
		Layer* layer = GetLayer( i );

		// Check layer only if it is visible
		if( layer->IsVisible() )
		{
			// Check all objects in the layer
			for( int j = 0 ; j < layer->GetObjectCount();  j++ )
			{
				Object* object = layer->GetObject( j ) ;

				Rect bounds = object->GetBounds();

				if( box.Includes( bounds ) )
					hits.push_back( object );
			}	
		}
	}
}

void Document :: RemoveObject( Object* obj )
{
	// For each layer
	for( int i = 0 ; i < GetLayerCount() ; i ++ )
	{
		Layer* layer = GetLayer( i );

		if( layer->RemoveObject( obj ) )
			return;
	}	
}

void Document :: ReplaceObject( Object* old_obj, Object* new_obj )
{
	// For each layer
	for( int i = 0 ; i < GetLayerCount() ; i ++ )
	{
		Layer* layer = GetLayer( i );

		if( layer->ReplaceObject( old_obj, new_obj ) )
			return;
	}	
}

void Document :: SelectObject( Object* obj )
{
	// Check so that object is not already selected
	if( ! obj->IsSelected() )
	{
		obj->SetSelected( true );
		m_ListOfSelectedObjects.push_back( obj );
	}
}

void Document :: DeselectObject( Object* obj )
{
	// Check so that object is already selected
	if( obj->IsSelected() )
	{
		obj->SetSelected( false );

		for( vector< Object* >::iterator it = m_ListOfSelectedObjects.begin() ; it != m_ListOfSelectedObjects.end() ; it++ )
		{
			if( *it == obj )
			{
				m_ListOfSelectedObjects.erase( it );
				break;
			}
		}
	}
}

void Document :: SelectAllObjects()
{
	// For each layer
	for( int i = 0 ; i < GetLayerCount() ; i ++ )
	{
		Layer* layer = GetLayer( i );

		// Check all objects in the layer
		for( int j = 0 ; j < layer->GetObjectCount();  j++ )
		{
			Object* object = layer->GetObject( j ) ;

			SelectObject( object );
		}	
	}
}

void Document :: DeselectAllObjects( Canvas* canvas )
{
	for( uint i = 0 ; i < m_ListOfSelectedObjects.size() ; i++ )
	{
		m_ListOfSelectedObjects[i]->SetSelected( false );
		if( canvas != NULL )
			m_ListOfSelectedObjects[i]->Draw( canvas );
	}
	m_ListOfSelectedObjects.clear();
}


void Document :: TransformAllObjects( Transform& trans )
{
	// For each layer
	for( int i = 0 ; i < GetLayerCount() ; i ++ )
	{
		Layer* layer = GetLayer( i );

		// Check all objects in the layer
		for( int j = 0 ; j < layer->GetObjectCount();  j++ )
		{
			Object* object = layer->GetObject( j ) ;
		
			object->DoTransform( trans );
		}	
	}
}

Rect Document :: GetBoundingBox()
{
	bool first = true;
	Rect ret;

	// For each layer
	for( int i = 0 ; i < GetLayerCount() ; i ++ )
	{
		Layer* layer = GetLayer( i );

		// Check all objects in the layer
		for( int j = 0 ; j < layer->GetObjectCount();  j++ )
		{
			Object* object = layer->GetObject( j ) ;
			Rect bound = object->GetBounds();

			if( first )
			{
				ret = bound;
				first = false;
			}
			else
				ret |= bound;
		}	
	}

	return ret;
}

void Document :: SetContinuePoint( os::Point p, double tangent )
{
	m_ContinuePointAvailable = true;	
	m_ContinuePoint = p;
	m_ContinueTangent = tangent;
}

void Document :: SetContinuePoint( Point center, Point endpoint )
{
	m_ContinuePointAvailable = true;	
	m_ContinuePoint = endpoint;
	m_ContinueTangent = CadMath::GetAngle( endpoint, center ) + M_PI;
	CadMath::FixAngle( m_ContinueTangent );
}

bool Document :: GetContinuePoint( os::Point& p, double& tangent )
{
	p = m_ContinuePoint;
	tangent = m_ContinueTangent;

	return m_ContinuePointAvailable;
}

Point Document :: MakeSnap( Point p1 )
{
	if( m_SnapActive)
	{
		p1 = CadMath::RotatePoint( m_SnapAngle, m_SnapOrigo, p1 );

		p1.x = rint( ( p1.x - m_SnapOrigo.x ) / m_SnapDistance.x ) * m_SnapDistance.x +  m_SnapOrigo.x;
		p1.y = rint( ( p1.y - m_SnapOrigo.y ) / m_SnapDistance.y ) * m_SnapDistance.y +  m_SnapOrigo.y;

  		p1 = CadMath::RotatePoint( -m_SnapAngle, m_SnapOrigo, p1);
	}

	if( m_OSnap != OSNAP_NONE )
	{
		// Remove previous snap point
		Point p = GET_CANVAS()->WorldToScreen( ValidOSnapPoint() );  // A little ugly since we need to call the canvas object from the document object
		Rect r( p.x - 5, p.y - 5, p.x + 5, p.y + 5 );
		GET_CANVAS()->Invalidate( r );

		p1 = _OSnap( p1 );
		
		// We need to perform a invalidation
		if( m_FoundOSnap )
		{
			Point p = GET_CANVAS()->WorldToScreen( ValidOSnapPoint() );
			Rect r( p.x - 5, p.y - 5, p.x + 5, p.y + 5 );

			GET_CANVAS()->Invalidate( r );
			GET_CANVAS()->Sync();
		}
		return p1;
	}
	else
		return p1;
}

Point Document :: MakeSnap( Point p1, Point p2 )
{
	if( m_SnapActive)
	{
		p2 = CadMath::RotatePoint( m_SnapAngle, m_SnapOrigo, p2 );

		p2.x = rint( ( p2.x - m_SnapOrigo.x ) / m_SnapDistance.x ) * m_SnapDistance.x +  m_SnapOrigo.x;
		p2.y = rint( ( p2.y - m_SnapOrigo.y ) / m_SnapDistance.y ) * m_SnapDistance.y +  m_SnapOrigo.y;

	  	p2 = CadMath::RotatePoint( -m_SnapAngle, m_SnapOrigo, p2 );
	}

	switch( m_OrthoMode )
	{
		case OrthoNone:
			break;
		case OrthoVert:
			p2.x = p1.x;
			break;
		case OrthoHoriz:
			p2.y = p1.y;
			break;
		case OrthoBoth:
		{	
			p1 = CadMath::RotatePoint( m_SnapAngle, m_SnapOrigo, p1 );
	  		p2 = CadMath::RotatePoint( m_SnapAngle, m_SnapOrigo, p2 );

			if( fabs( p2.x - p1.x ) > fabs( p2.y - p1.y  ) )
				p2.y = p1.y;
			else
				p2.x = p1.x;

  			p2 = CadMath::RotatePoint( -m_SnapAngle, m_SnapOrigo, p2 );
			break;
		}
	}

	if( m_OSnap != OSNAP_NONE )
		return _OSnap( p2 );
	else
		return p2;
}


void Document :: DeleteLayer( int x )
{
	// Make some sanity check
	if( !( x >= 0 && x < GetLayerCount() ) )
	{
		printf( "Wrong input DeleteLayers %d\n", x );
		return;
	}

	try
	{
		int i = 0;

		for( vector< Layer* >::iterator it = m_ListOfLayers.begin() ; it != m_ListOfLayers.end() ; it++, i++ )
		{
			if( i == x )
			{
				// Check if this is the current layer
				if( (*it) == m_CurrentLayer )
				{
					if( x == 0 )
						m_CurrentLayer = GetLayer( 1 );
					else
						m_CurrentLayer = GetLayer( x - 1 );
				}
				delete *it;
				m_ListOfLayers.erase( it );
				return;
			}
		}
	}
	catch( ... )
	{
		printf( "Error while deleting layers\n" );
	}
}

bool Document :: LayerExists( Layer* layer )
{ 
	for( int i = 0 ; i < GetLayerCount() ; i++ )
	{ 
		if( GetLayer( i ) == layer ) 
			return true; 
	}

	return false;
}

void Document :: SwapLayers( int x, int y )
{
	try
	{
		Layer* l1 = m_ListOfLayers.at( x );
		Layer* l2 = m_ListOfLayers.at( y );
		m_ListOfLayers.at( x ) = l2;
		m_ListOfLayers.at( y ) = l1;
	}
	catch( ... )
	{
		printf( "Error while swapping layers\n" );
	}
}

Rect Document :: GetSelectedBoundingBox()
{
	bool first = true;
	Rect ret;

	// For each selected object
	for( uint i = 0 ; i < m_ListOfSelectedObjects.size() ; i ++ )
	{
		Rect bound =m_ListOfSelectedObjects[ i ]->GetBounds();

		if( first )
		{
			ret = bound;
			first = false;
		}
		else
			ret |= bound;
	}	

	return ret;
}

///////////////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E   M E T H O D S
//
///////////////////////////////////////////////////////////////////////////////////////

Point Document :: _OSnap( os::Point p )
{
	m_OsnapPoint = p;
	m_FoundOSnap = false;

	// This is ugly but we need to know the canvas area so check if a osnap
	// is within it or not
	Rect scr = GET_CANVAS()->GetViewPortSize();

	// For each layer
	for( int i = 0 ; i < GetLayerCount() ; i ++ )
	{
		Layer* layer = GetLayer( i );

		// Check layer only if it is visible
		if( layer->IsVisible() )
		{
			// Check all objects in the layer
			for( int j = 0 ; j < layer->GetObjectCount();  j++ )
			{
				Object* object = layer->GetObject( j ) ;

				vector< Point > list;
				object->GetSnapPoint( m_OSnap, p, list );

				if( ! list.empty() )
				{
					// Check all returned osnap point from object
					for( uint k = 0 ; k < list.size() ; k++ )
					{
						Point t = list[ k ];

						// Make sure the osnap is within the screen
						if( ! scr.DoIntersect( t ) )
							continue;

						if( !m_FoundOSnap )
						{
							m_OsnapPoint = t;
							m_FoundOSnap = true; 
						}
						// If new osnap is closer than the old use the new one instead
						else if( CadMath::Distance( p,  t  ) < CadMath::Distance( p,  m_OsnapPoint  ) )
							m_OsnapPoint = t;											
					}
				}

			}	
		}
	}

	return m_OsnapPoint;
}

