// oCADis (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


#include <storage/file.h>
#include <gui/requesters.h>
#include <util/resources.h>
#include <util/settings.h>
#include <gui/toolbar.h>
#include <gui/button.h>
#include <gui/dropdownmenu.h>
#include <gui/filerequester.h>
#include <gui/statusbar.h>
#include <gui/menu.h>
#include <gui/checkmenu.h>
#include <gui/layoutview.h>

#include "main.h"
#include "appwindow.h"
#include "messages.h"
#include "common.h"
#include "layer.h"
#include "postoffice.h"
#include "tableview.h"
#include "viewcoord.h"
#include "tool_main/tool_main.h"

#include "filters/filter_ocadis.h"
#include "filters/filter_ocadis_old.h"
#include "filters/filter_ps.h"
#include "filters/filter_dxf.h"

#include "gui/splitter.h"

#include "sidebar_toolbox.h"
#include "sidebar_dialog.h"

#include "dialogs/dialog_style.h"
#include "dialogs/dialog_layer.h"
#include "dialogs/dialog_guide.h"
#include "dialogs/dialog_osnap.h"

#include "windialogs/windia_pagesetup.h"
#include "windialogs/windia_about.h"
#include "windialogs/windia_alert.h"


using namespace os;
using namespace std;



///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E
//
///////////////////////////////////////////////////////////////////////////////

class AppWindow :: _Private
{
public:
	_Private() {};

	Menu* _CreateMenus()
	{		
		m_MenuBar=new Menu( Rect(), "main_menu", ITEMS_IN_ROW, CF_FOLLOW_LEFT | CF_FOLLOW_RIGHT | CF_FOLLOW_TOP );

		// Create the menus
		m_AppMenu = new Menu( Rect(), "oCADis", ITEMS_IN_COLUMN);
		m_AppMenu->AddItem( "Quit", new Message( MSG_MENU_QUIT ), "Ctrl+Q" );
		m_AppMenu->AddItem( new MenuSeparator() );
		m_AppMenu->AddItem( "About", new Message( MSG_MENU_ABOUT ) );

		// The file menu
		m_FileMenu = new Menu( Rect(), "File", ITEMS_IN_COLUMN );
		m_FileNew = new MenuItem( "New", new Message( MSG_MENU_NEW ), "Ctrl+N" );
		m_FileMenu->AddItem( m_FileNew );
		m_FileMenu->AddItem( new MenuSeparator() );
		m_FileOpen = new MenuItem( "Open...", new Message( MSG_MENU_OPEN ), "Ctrl+O" );
		m_FileMenu->AddItem( m_FileOpen );
		m_FileMenu->AddItem( new MenuSeparator() );
		m_FileClose = new MenuItem( "Close", new Message( MSG_MENU_CLOSE ), "Ctrl+W" );
		m_FileMenu->AddItem( m_FileClose );
		m_FileSave = new MenuItem( "Save", new Message( MSG_MENU_SAVE ), "Ctrl+S" );
		m_FileMenu->AddItem( m_FileSave );
		m_FileSaveAs = new MenuItem( "Save as...", new Message( MSG_MENU_SAVEAS ), "" );
		m_FileMenu->AddItem( m_FileSaveAs );

		// The export menu and its submenu
		m_FileExportAs = new Menu( Rect(), "Export", ITEMS_IN_COLUMN );

		// add the filters to the export
		// Skip the first filter as that is the oCADis filter
		for( uint i = 1 ; i < m_ListOfFilters.size() ; i++ )
		{
			String e = m_ListOfFilters[ i ]->GetMenuString();

			if( e != "" )
			{
				Message* msg = new Message( MSG_MENU_EXPORT );
				msg->AddInt8( "filter_no", i );
				MenuItem* tmp = new MenuItem( e, msg, "" );
				m_FileExportAs->AddItem( tmp );
			}
		}	

		// Only add the export menu if we have filter
		if( m_FileExportAs->GetItemCount() > 0 )
			m_FileMenu->AddItem( m_FileExportAs );

		m_FileMenu->AddItem( new MenuSeparator() );
		m_FilePageSetup = new MenuItem( "Page setup...", new Message( MSG_MENU_PAGESETUP ), "" );
		m_FileMenu->AddItem( m_FilePageSetup );
		m_FilePrint = new MenuItem( "Print", new Message( MSG_MENU_PRINT ), "Ctrl+P" );
		m_FileMenu->AddItem( m_FilePrint );

		// The edit menu
		m_EditMenu = new Menu( Rect(), "Edit", ITEMS_IN_COLUMN );
		m_EditUndo = new MenuItem( "Undo", new Message( MSG_MENU_UNDO ), "Ctrl+Z" );
		m_EditMenu->AddItem( m_EditUndo );
		m_EditRedo = new MenuItem( "Redo", new Message( MSG_MENU_REDO ), "" );
		m_EditMenu->AddItem( m_EditRedo );

		// The dialog submenu
		m_ViewDialogs = new Menu( Rect(), "Dialogs", ITEMS_IN_COLUMN );
		m_ViewLayer = new MenuItem( "Layer", new Message( MSG_MENU_LAYER ), "" );
		m_ViewDialogs->AddItem( m_ViewLayer );
		m_ViewStyle = new MenuItem( "Style", new Message( MSG_MENU_STYLE ), "" );
		m_ViewDialogs->AddItem( m_ViewStyle );
		m_ViewGuide = new MenuItem( "Guide", new Message( MSG_MENU_GUIDE ), "" );
		m_ViewDialogs->AddItem( m_ViewGuide );
		m_ViewOSnap = new MenuItem( "Object Snap", new Message( MSG_MENU_OSNAP ), "" );
		m_ViewDialogs->AddItem( m_ViewOSnap );
		m_ViewDialogs->AddItem( new MenuSeparator() );
		m_ViewHide = new MenuItem( "Hide Dialog", new Message( MSG_MENU_HIDE ), "" );
		m_ViewDialogs->AddItem( m_ViewHide );

		// The view menu
		m_ViewMenu = new Menu( Rect(), "View", ITEMS_IN_COLUMN );
		m_ViewMenu->AddItem( m_ViewDialogs );
		m_ViewMenu->AddItem( new MenuSeparator() );
		m_ViewToolbox = new MenuItem( "Hide Toolbox", new Message( MSG_MENU_TOOLBOX ), "" );

		m_ViewMenu->AddItem( m_ViewToolbox );
		m_ViewMenu->AddItem( new MenuSeparator() );
		m_ViewRefresh = new MenuItem( "Refresh", new Message( MSG_MENU_REFRESH ), "" );
		m_ViewMenu->AddItem( m_ViewRefresh );
		m_ViewZoomAll = new MenuItem( "Zoom All", new Message( MSG_MENU_ZOOM_ALL ), "" );
		m_ViewMenu->AddItem( m_ViewZoomAll );
		m_ViewMenu->AddItem( new MenuSeparator() );
		m_ViewZoomMode = new CheckMenu( "Zoom mode", new Message( MSG_MENU_ZOOM_MODE ) );
		m_ViewMenu->AddItem( m_ViewZoomMode );
		m_ViewPanMode = new CheckMenu( "Pan mode", new Message( MSG_MENU_PAN_MODE ), true );
		m_ViewMenu->AddItem( m_ViewPanMode );

		// Attach the menus
		m_MenuBar->AddItem( m_AppMenu );	
		m_MenuBar->AddItem( m_FileMenu );	
		m_MenuBar->AddItem( m_EditMenu );	
		m_MenuBar->AddItem( m_ViewMenu );	

		return m_MenuBar;
	}

	void _UpdateMenuStatus()
	{
		// If there is no document available
		if( m_CurrentDocument == NULL )
		{
			m_FileClose->SetEnable( false );
			m_FileSave->SetEnable( false );
			m_FileSaveAs->SetEnable( false );
			return;
		}

		m_FileSave->SetEnable( m_CurrentDocument->IsModified() );
	}

	void _UpdateUndo()
	{
		if( m_CurrentDocument == NULL )
		{
			m_EditUndo->SetEnable( false );
			m_EditRedo->SetEnable( false );

			return;
		}

		// Create new undo menuitem
		String s( "Undo - " );
		s = s + m_CurrentDocument->GetUndoName();
		m_EditUndo->SetLabel( s );
		m_EditUndo->SetEnable( m_CurrentDocument->HasUndoObjects() );
	
		// Check for redo objects
		String t( "Redo - " );
		t = t + m_CurrentDocument->GetRedoName();
		m_EditRedo->SetLabel( t );
		m_EditRedo->SetEnable( m_CurrentDocument->HasRedoObjects() );
	}

	void _CreateFilters()
	{
		m_ListOfFilters.push_back( new FilteroCADis() );
		m_ListOfFilters.push_back( new FilteroCADisOld() );
		m_ListOfFilters.push_back( new FilterPS() );
		m_ListOfFilters.push_back( new FilterDXF() );
	}

	void _OpenDocument( String path, AppWindow* t )
	{
		Document* doc;

		// Read the first 1024 bytes in the file
		File g( path );
		char buffer[ 1025 ];
		g.Read( buffer, 1024 );

		String data( buffer );

		// Find correct filter for this file
		for( uint i = 0 ; i < m_ListOfFilters.size() ; i++ )
		{
			if( m_ListOfFilters[ i ]->Identify( path, data ) )
			{
				// Create new document
				doc = new Document();
		
				// Load file using this filter
				try
				{
					m_ListOfFilters[ i ]->Open( path, *doc );

					doc->Check();
					doc->SetTitle( path );
	
					t->SetCurrentDocument( doc );
				}
				catch( String& e )
				{
					WinDiaAlert::ShowErrorAlert( "Error open file", e );
					delete doc;
				}

				return;
			}
		}

		WinDiaAlert::ShowErrorAlert( "Error open file", "The file type is not supported\n and cannot be opened." );
	}

	void _UpdateTitle( AppWindow* t )
	{
		if( m_CurrentDocument == NULL )
			t->SetTitle( "oCADis" );	
		else
			t->SetTitle( m_CurrentDocument->GetTitle() );	
	}

	void _SetSideBarDialog( Dialog* dia  )
	{
		m_DialogSidebar->SetDialog( dia );
		
		// Show dialog if hidden
		if( !m_Splitter2->IsView2Visible() )
		{
			m_Splitter2->ShowView2();			
			m_Splitter2->AdjustLayout();	
		}

		m_ViewHide->SetLabel( "Hide Dialog" );
	}

	// List of filters
	vector< FilterBase* > m_ListOfFilters;

	// Some window dialogs
	WinDiaPageSetup* m_PageSetup;
	WinDiaAbout* m_About;

	// Here are the settings
	Preferences m_Preferences;

	// Menus
	Menu* m_MenuBar;
	Menu* m_AppMenu;

	// The file menu and its menuitems
	Menu* 	  m_FileMenu;
	MenuItem* m_FileNew;
	MenuItem* m_FileOpen;
	MenuItem* m_FileClose;
	MenuItem* m_FileSave;
	MenuItem* m_FileSaveAs;
	Menu* m_FileExportAs;
	MenuItem* m_FilePageSetup;
	MenuItem* m_FilePrint;

	// The edit menu
	Menu* 	  m_EditMenu;
	MenuItem* m_EditUndo;	
	MenuItem* m_EditRedo;

	// The View menu and its menuitems
	Menu* 	  m_ViewMenu;
	Menu*	  m_ViewDialogs;
	MenuItem* m_ViewLayer;	
	MenuItem* m_ViewStyle;
	MenuItem* m_ViewGuide;	
	MenuItem* m_ViewOSnap;
	MenuItem* m_ViewHide;
	MenuItem* m_ViewToolbox;
	MenuItem* m_ViewRefresh;
	MenuItem* m_ViewZoomAll;
	CheckMenu* m_ViewZoomMode;
	CheckMenu* m_ViewPanMode;

	// Splitters
	Splitter* m_Splitter1;
	Splitter* m_Splitter2;

	// Sidebars
	SideBar* m_CurrentSideBar;
	ToolboxSideBar* m_ToolboxSidebar;

	SideBar* m_CurrentDialogSideBar;
	DialogSideBar* m_DialogSidebar;

	// Crreate all sidebar dialogs
	DialogStyle* m_DialogStyle;
	DialogLayer* m_DialogLayer;
	DialogGuide* m_DialogGuide;
	DialogOSnap* m_DialogOSnap;

	// Draw area
	Canvas* m_Canvas;

	// The coordnate viewer
	ViewCoord* m_ViewCoord;

	// The status bar
	StatusBar* m_StatusBar;
	
	// The document
	Document* m_CurrentDocument;

	// Create all default tools
	ToolMain* m_ToolMain;

	// Current state
	State* m_CurrentState;
};

///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S
//
///////////////////////////////////////////////////////////////////////////////

AppWindow :: AppWindow( const Rect& cFrame ) : Window( cFrame, "main_window", "oCADis" )
{	
	// Create the private class
	m = new _Private();

	// Init variables	
	m->m_CurrentState = NULL;
	m->m_CurrentDocument = NULL;
	m->m_PageSetup = NULL;
	m->m_About = NULL;

	// Create filters
	m->_CreateFilters();

	// Load the settings
	m->m_Preferences.Load();

	// Create widgets
	LayoutNode*	root_node = new VLayoutNode( "pcRootNode" );
	LayoutNode*	statusbar_node = new HLayoutNode( "pcRootNode", 0.0f );
	LayoutView*	root_view = new LayoutView( GetBounds(), "pcRootView" );
	root_view->SetRoot( root_node );
	AddChild( root_view );

	// Make all menus
	m->_CreateMenus();
	m->m_MenuBar->SetTargetForItems( this );

	// Create canvas
	m->m_Canvas = new Canvas( this, Rect() );

	// Create sidebars
	m->m_ToolboxSidebar = new ToolboxSideBar( Rect() );

	// Set current sidebar
	m->m_CurrentSideBar = m->m_ToolboxSidebar;

	// Create dialog sidebars
	m->m_DialogSidebar = new DialogSideBar( Rect() );

	// Set current dialogsidebar
	m->m_CurrentDialogSideBar = m->m_DialogSidebar;

	// Create sidebar dialogs
	m->m_DialogStyle = new DialogStyle( m->m_DialogSidebar );
	m->m_DialogLayer = new DialogLayer( m->m_DialogSidebar );
	m->m_DialogGuide = new DialogGuide( m->m_DialogSidebar );
	m->m_DialogOSnap = new DialogOSnap( m->m_DialogSidebar );

	// Create splitters
	m->m_Splitter2 = new Splitter( Rect(), "", m->m_Canvas, m->m_DialogSidebar, HORIZONTAL );
	m->m_Splitter2->SetSplitRatio( 0.5f );

	m->m_Splitter1 = new Splitter( Rect(), "", m->m_ToolboxSidebar, m->m_Splitter2, HORIZONTAL );
	m->m_Splitter1->SetSplitRatio( 0.1f );

	// Create the statusbar
	m->m_StatusBar = new StatusBar( Rect(), "" );
	m->m_StatusBar->AddPanel( "message", "" );

	// View coordinate
	m->m_ViewCoord = new ViewCoord( Rect(), "" );

	// Create standard tools
	m->m_ToolMain = new ToolMain( m->m_ToolboxSidebar );	

	// Add widgets to the layout system
	statusbar_node->AddChild( m->m_StatusBar );
	statusbar_node->AddChild( m->m_ViewCoord );

	root_node->AddChild( m->m_MenuBar, 0.0f );
	root_node->AddChild( m->m_Splitter1 );
	root_node->AddChild( statusbar_node );
	root_view->InvalidateLayout();

	// Fix some welcome text
	SetStatusbar( "Welcome to oCADis! Please select action." );

	// Create mailboxes
	AddMailbox( "Undo" );
	AddMailbox( "Title" );

	// Add some global keys
	AddShortcut( ShortcutKey( 27 ), new Message( MSG_MAIN_TOOLBOX ) );	// ESC

	// Create an empty document
	SetCurrentDocument( new Document() );
}

AppWindow :: ~AppWindow()
{
	delete m;
}

void AppWindow :: HandleMessage( Message* pcMessage )
{
	switch( pcMessage->GetCode() )	//Get the message code from the message
	{		
///////// APP MENU		
		case MSG_MENU_ABOUT:
			if( m->m_About == NULL )
				m->m_About = new WinDiaAbout();

			m->m_About->Raise( );
			break;
		case MSG_MENU_QUIT:
			if( m->m_CurrentDocument == NULL || !m->m_CurrentDocument->IsModified() )
				Application::GetInstance()->PostMessage( M_QUIT );
			else
			{
				WinDiaAlert* t = new WinDiaAlert( WinDiaAlert::ALERT_WARNING, "Quit application", "You have unsaved documents. Are you sure that you want to quit?", "No", "Yes", NULL );
				t->Go( new Invoker( new Message( MSG_ALERT_QUIT ), this ) );
			}
			break;

///////// FILE MENU

		case MSG_MENU_NEW:
			break;
		case MSG_MENU_OPEN:
		{
			FileRequester* req = new FileRequester( FileRequester::LOAD_REQ, new Messenger(this), "" );
			req->CenterInWindow( this );
			req->Show( true );
			req->MakeFocus();
			break;
		}
		case M_LOAD_REQUESTED:
		{
			const char* path;
			SetMainToolbox();

//			FileRequester *req;
//			if( pcMessage->FindPointer( "source", (void**)req ) == EOK )
//				req->Close();

			if( pcMessage->FindString( "file/path", &path ) == 0 && m->m_CurrentDocument != NULL )
			{
				m->_OpenDocument( String( path ), this );

				m->m_Canvas->Refresh();
			}

			break;
		}
		case MSG_MENU_CLOSE:
			break;
		case MSG_MENU_SAVE:
			break;
		case MSG_MENU_SAVEAS:
		{
			FileRequester* req = new FileRequester( FileRequester::SAVE_REQ, new Messenger(this), "" );
			req->CenterInWindow( this );
			req->Show( true );
			req->MakeFocus();
			
			break;
		}
		case M_SAVE_REQUESTED:
		{
			const char* path;

//			FileRequester *req;
//			if( pcMessage->FindPointer( "source", (void**)req ) == EOK )
//				req->Close();

			if( pcMessage->FindString( "file/path", &path ) == 0 && m->m_CurrentDocument != NULL )
			{
				String p( path );

				// The first filter is always oCADis
				try
				{
					m->m_ListOfFilters.at( 0 )->Save( p, *m->m_CurrentDocument );

					m->m_CurrentDocument->SetModification( false );
					m->m_CurrentDocument->SetTitle( p );
					m->_UpdateTitle( this );
				}
				catch( String& e )
				{
					printf( "Error saving file: %s\n", e.c_str() );
				}
			}

			break;
		}
		case MSG_MENU_EXPORT:
		{
			int8 filter;
			
			if( pcMessage->FindInt8( "filter_no", &filter ) != EOK )
				return;

			Message* msg = new Message( MSG_EXPORT_REQUESTED );
			msg->AddInt8( "filter_no", filter );

			FileRequester* req = new FileRequester( FileRequester::SAVE_REQ, new Messenger(this), "", FileRequester::NODE_FILE, false, msg );
			req->CenterInWindow( this );
			req->Show( true );
			req->MakeFocus();
			
			break;
		}
		case MSG_EXPORT_REQUESTED:
		{
			int8 filter;
			const char* path;

			if( pcMessage->FindString( "file/path", &path ) != EOK)
				return;
			
			if( pcMessage->FindInt8( "filter_no", &filter ) != EOK )
				return;

			String p( path );

			// Check the extension and add it if neccessary
			string ext = string( "." ) + string( m->m_ListOfFilters[ filter ]->GetExtension() );

			if( string(p).rfind( ext ) == string::npos )
			{
				p += ext;
			}

			try
			{
				m->m_ListOfFilters[ filter ]->Save( p, *m->m_CurrentDocument );
			}
			catch( String& e )
			{
				printf( "Error saving file: %s\n", e.c_str() );
			}
			break;
		}
		case MSG_MENU_PAGESETUP:
		{
			if( m->m_PageSetup == NULL )
				m->m_PageSetup = new WinDiaPageSetup( Rect( 100, 100, 300, 300 ), m->m_CurrentDocument );

			m->m_PageSetup->Raise( );
			break;
		}
		case MSG_MENU_PRINT:
		{
			break;
		}

///////// EDIT MENU

		case MSG_MENU_UNDO:
			m->m_CurrentDocument->DoUndo( m->m_Canvas, m->m_CurrentDocument );
			m->_UpdateUndo();
			break;

		case MSG_MENU_REDO:
			m->m_CurrentDocument->DoRedo( m->m_Canvas, m->m_CurrentDocument );
			m->_UpdateUndo();
			break;

///////// VIEW MENU
		case MSG_MENU_LAYER:
		{
			m->_SetSideBarDialog( m->m_DialogLayer  );
			break;
		}
		case MSG_MENU_STYLE:
		{
			m->_SetSideBarDialog( m->m_DialogStyle  );
			break;
		}
		case MSG_MENU_GUIDE:
		{
			m->_SetSideBarDialog( m->m_DialogGuide );
			break;
		}
		case MSG_MENU_OSNAP:
		{
			m->_SetSideBarDialog( m->m_DialogOSnap );
			break;
		}
		case MSG_MENU_HIDE:
		{	
			if( m->m_Splitter2->IsView2Visible() )
			{
				m->m_ViewHide->SetLabel( "Show Dialog" );
				m->m_Splitter2->HideView2();			
			}
			else
			{
				m->m_ViewHide->SetLabel( "Hide Dialog" );
				m->m_Splitter2->ShowView2();
			}

			m->m_Splitter2->AdjustLayout();
			break;
		}
		case MSG_MENU_TOOLBOX:
		{
			if( m->m_Splitter1->IsView1Visible() )
			{
				m->m_ViewToolbox->SetLabel( "Show Toolbox" );
				m->m_Splitter1->HideView1();			
			}
			else
			{
				m->m_ViewToolbox->SetLabel( "Hide Toolbox" );
				m->m_Splitter1->ShowView1();
			}

			m->m_Splitter1->AdjustLayout();
			break;
		}

		case MSG_MENU_REFRESH:
		{
			m->m_Canvas->Refresh();
			break;
		}
		case MSG_MENU_ZOOM_ALL:
		{
			m->m_Canvas->ZoomBox( m->m_CurrentDocument->GetBoundingBox() );
			m->m_Canvas->Refresh();
			break;
		}
		case MSG_MENU_ZOOM_MODE:
		{
			break;
		}
		case MSG_MENU_PAN_MODE:
		{
			break;
		}

		// Broadcast messages
		case MSG_BROADCAST_UPDATE_UNDO:
			m->_UpdateUndo();
			break;
		case MSG_BROADCAST_UPDATE_TITLE:
			m->_UpdateTitle( this );
			break;

		case MSG_ALERT_QUIT:
		{
			int32 t;
			if( pcMessage->FindInt32( "which", &t ) == EOK )
			{
				if( t == 1 ) // Yes to quit
					Application::GetInstance()->PostMessage( M_QUIT );
			}
			break;
		}

		case MSG_MAIN_TOOLBOX:
		{
			SetMainToolbox();
			break;
		}
		default:
		{
			Window::HandleMessage( pcMessage );
			break;
		}
	}
}

bool AppWindow :: OkToQuit( void )
{
	if( m->m_CurrentDocument == NULL || !m->m_CurrentDocument->IsModified() )
	{
		Application::GetInstance()->PostMessage( M_QUIT );
		return true;
	}

	WinDiaAlert* t = new WinDiaAlert( WinDiaAlert::ALERT_WARNING, "Quit application", "You have unsaved documents. Are you sure that you want to quit?", "No", "Yes", NULL );
	t->Go( new Invoker( new Message( MSG_ALERT_QUIT ), this ) );

	return false;
}

void AppWindow :: Init()
{
	m->m_ToolMain->Init();	

	// Set correct sidebar dialog
	m->m_DialogSidebar->SetDialog( m->m_DialogLayer );

	// Force an update
	SetCurrentDocument( m->m_CurrentDocument );
}

void AppWindow :: SetCoord( Point coord )
{
	if( m->m_ViewCoord != NULL )
		m->m_ViewCoord->SetCoord( coord );
}

void AppWindow :: ClearCoord()
{
	if( m->m_ViewCoord != NULL )
		m->m_ViewCoord->ClearCoord();
}

void AppWindow :: SetStatusbar( os::String what )
{
	if( m->m_StatusBar != NULL )
		m->m_StatusBar->SetText( "message", what );
}

void AppWindow :: SetCurrentState( State* state )
{
	m->m_CurrentState = state;
	m->m_ToolboxSidebar->SetToolbox( state->GetToolbox() );
	SetStatusbar( state->GetStatusText() );
	state->Init( GetCanvas(), GetCurrentDocument() );
}

State* AppWindow :: GetCurrentState()
{
	return m->m_CurrentState;
}

void AppWindow :: ClearCurrentState()
{ 
	m->m_CurrentState = NULL;
}

void AppWindow :: SetMainToolbox()
{
	int s = m->m_CurrentDocument->GetOSnapMode();
	m->m_CurrentDocument->SetOSnapMode( OSNAP_NONE );
	// Remove previous snap point
	Point p = m->m_Canvas->WorldToScreen( m->m_CurrentDocument->ValidOSnapPoint() );
	Rect r( p.x - 5, p.y - 5, p.x + 5, p.y + 5 );
	m->m_Canvas->Invalidate( r );
	m->m_CurrentDocument->SetOSnapMode( s );

	m->m_Canvas->Undraw();
	m->m_CurrentDocument->DeselectAllObjects( m->m_Canvas );
	m->m_ToolMain->Init();	
}

Canvas* AppWindow :: GetCanvas()
{ 
	return m->m_Canvas; 
}

void AppWindow :: SetCurrentDocument( Document* doc )
{
	m->m_CurrentDocument = doc;
	m->m_CurrentDocument->Check();

	// Set title
	SetTitle( m->m_CurrentDocument->GetTitle() );

	// Update undostatus, menu and sidebars
	m->_UpdateUndo();
	m->_UpdateMenuStatus();
	if( m->m_DialogSidebar != NULL )
		m->m_DialogSidebar->SetDocument( doc );

	// Broadcast the new document	
	Message* msg = new Message( MSG_BROADCAST_NEW_DOCUMENT  );
	msg->AddPointer( "document", (void*) doc );
	Mail( "Document", msg );
}

Document* AppWindow :: GetCurrentDocument()
{ 
	return m->m_CurrentDocument;
}

Preferences& AppWindow :: GetPreferences()
{
	return m->m_Preferences;
}

