// EFileBrowser	 (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <gui/window.h>
#include <gui/guidefines.h>
#include <storage/path.h>
#include <storage/directory.h>
#include <storage/fsnode.h>

#include "sltextview.h"

using namespace os;
using namespace std;

SLTextView :: SLTextView( Rect cFrame, const String& cName, Message* pcMessage, bool do_automation, uint32 nResizeMask, uint32 nFlags ) : 
	TextView( cFrame, cName, "", nResizeMask, nFlags)
{
	m_Invoker = new Invoker( pcMessage );

	m_Automation = do_automation;
	m_MinWidth = TextView::GetPreferredSize( false ).x,

	TextView::SetMultiLine( false );
	m_theMessage = pcMessage;

	// Create popup menu
	m_ContextMenu = new Menu(Rect(0,0,10,10),"",ITEMS_IN_COLUMN);
	if( m_Automation )
	{
		m_ContextBrowser = new MenuItem( "Open browser...", new Message( SLTextView::MSG_OPEN_BROWSER ), "Ctrl + B" );
		m_ContextMenu->AddItem( m_ContextBrowser );
		m_ContextMenu->AddItem( new MenuSeparator() );
	}

	m_ContextCut = new MenuItem( "Cut", new Message( SLTextView::MSG_CUT ), "Ctrl + X" );
	m_ContextMenu->AddItem( m_ContextCut );
	m_ContextCopy = new MenuItem( "Copy", new Message( SLTextView::MSG_COPY ), "Ctrl + C" );
	m_ContextMenu->AddItem(  m_ContextCopy );
	m_ContextPaste = new MenuItem( "Paste", new Message( SLTextView::MSG_PASTE ), "Ctrl + V" );
	m_ContextMenu->AddItem( m_ContextPaste );
	m_ContextMenu->AddItem( new MenuSeparator()) ;
	m_ContextSelectAll = new MenuItem( "Select All", new Message( SLTextView::MSG_SELECT_ALL ), "Ctrl + A" );
	m_ContextMenu->AddItem( m_ContextSelectAll );
}

SLTextView :: ~SLTextView()
{
	delete m_ContextMenu;
}

void SLTextView :: AttachedToWindow()
{
	m_ContextMenu->SetTargetForItems( this );
}

void SLTextView :: MouseDown( const Point &cPosition, uint32 nButtons )
{
	if( nButtons == 2 )
		m_ContextMenu->Open( ConvertToScreen( cPosition ) );	// Open the menu where the mouse is
	else
		TextView::MouseDown(cPosition, nButtons);
}

void SLTextView :: KeyDown( const char* pzString, const char* pzRawString, uint32 nQualifiers )
{
	switch( (unsigned char)(*pzString) )
	{
		case VK_RETURN:
			if( m_theMessage != NULL )
			{	
				m_theMessage->RemoveName("file/path" );
				m_theMessage->AddString( "file/path", GetValue() );
				m_Invoker->Invoke( m_theMessage );
			}
			break;
		case VK_TAB:
			if( m_Automation )
				DoAutoCompletion();
			else
				TextView::KeyDown( pzString, pzRawString, nQualifiers );

			break;
		
		default:
			TextView::KeyDown( pzString, pzRawString, nQualifiers );
	}
}

void SLTextView :: HandleMessage( Message* msg )
{
	switch( msg->GetCode() )	//Get the message code from the message
	{	
	case MSG_OPEN_BROWSER:
		// Create fileselector
		m_Browser = new FileRequester( FileRequester::LOAD_REQ, new Messenger(this), GetValue().AsString(), FileRequester::NODE_DIR );
		m_Browser->Lock();
		m_Browser->Start();
		m_Browser->CenterInWindow( GetWindow() );	
		m_Browser->Show(true);
		m_Browser->MakeFocus();
		m_Browser->Unlock();
		break;
	case M_LOAD_REQUESTED:
		{
			const char* path;
			FileRequester* browser;

			if( msg->FindPointer( "source", (void**)&browser ) == EOK )
				browser->Close();

			if( msg->FindString("file/path", &path ) == 0 )
			{
				m_theMessage->RemoveName("file/path" );
				m_theMessage->AddString( "file/path", path );
				GetWindow()->HandleMessage( m_theMessage );
			}
			break;
		}
	case MSG_CUT:
		if(	!TextSelected() )
			TextView::SelectAll();
		TextView::Cut();
		break;
	case MSG_COPY:
		if(	!TextSelected() )
		{
			TextView::SelectAll();
			TextView::Copy();
			TextView::ClearSelection();
		}
		else
			TextView::Copy();
		break;
	case MSG_PASTE:
		if(	!TextSelected() )
			TextView::SelectAll();

		TextView::Paste();
		break;
	case MSG_SELECT_ALL:
		TextView::SelectAll();
		break;
	default:
		TextView::HandleMessage( msg );
		break;
	}
	
}


Point SLTextView :: GetPreferredSize( bool bLargest ) const
{
	if( bLargest )
		return TextView::GetPreferredSize( bLargest );
	else
		return Point( m_MinWidth, TextView::GetPreferredSize( false ).y );
}

void SLTextView :: DoAutoCompletion()
{
	string path;	
	int x, y;		

	// Find path to check
	GetCursor( &x, &y );
	path = string( GetText() );
	path.resize( x );

	string n =  AutoCompletion( path );
	int cursor_x = n.size();
	n += string( GetText() ).substr( x );
	SetValue( String( n ) );
	SetCursor( cursor_x, y );
}

string SLTextView :: AutoCompletion( string& path )
{
	try
	{
		string root, leaf;
	
		// Get the root dir
		int i = path.rfind( "/" );
		if( (uint) i == string::npos )
			root = "/";
		else
			root = path.substr( 0, i + 1 );

		// Get the leaf
		if( (uint) i == string::npos )
			leaf = path;
		else
			leaf = path.substr( i + 1 );

		// check if user wants to take one step back
		if( leaf == ".." )
		{
			i = root.rfind( "/", root.size() - 2 );
			if( (uint) i == string::npos )
				root = "/";
			else
				root = root.substr( 0, i + 1 );

			return root;
		}
	
		// Create a list of valid files
		vector<string> valid_filenames;

		String name;
		Directory dir( root );
		dir.Rewind();	
		
		while( dir.GetNextEntry( &name ) == 1 )
		{
			if( !( name == "." ) && !( name == ".." ) )
			{
				String new_path, path;
				dir.GetPath( &path );
				new_path = path + String( "/" ) + name;
	
				FSNode fs( new_path );
	
				// We are only intresseted in directories
				if( fs.IsDir() )
				{
					if( name.find( leaf ) == 0 )
					{
						valid_filenames.push_back( name );
					}
				}
			}
		}
	
		// Have we found any files
		if( valid_filenames.size() == 1)
		{
			return String( root + valid_filenames[ 0 ] + "/" );
		}
		else if( valid_filenames.size() > 1)
		{
			// Find the common string
			return String( root + CommonCharacters( valid_filenames ) );
		}
	} 
	catch( std::exception & ) { }
	
	return path;
}

string SLTextView :: CommonCharacters( vector<string>& valid_filenames )
{
	string res = "";
	int i = 0;
	string first_name = valid_filenames[0];	

	while( 1 )
	{
		// Check so hat we havnt got the end of the string
		if( (uint) i > first_name.size() )
			return res;

		char c = first_name[i];

		// Check if this is valid for all strings
		bool hit = true;
		for( int y = 1 ; (uint) y < valid_filenames.size() ; y++ )
		{
			string name = valid_filenames[y];
		
			// Check that we are not outside the string
			if( (uint) i > name.size() )
				return res;

			if( c != name[ i ] )
			{
				hit = false;
				break;
			}
		}

		if( !hit )
		{
			return res;		
		}
	
		res += c;

		// Check next character
		i++;
	}
}

bool SLTextView :: TextSelected()
{
	IPoint start, end;
	TextView::GetSelection( &start, &end );

	return !( start.x == end.x && start.y == end.y );
}

void SLTextView :: SetTarget( Handler* pcHandler )
{
	m_Invoker->SetTarget( pcHandler );
}
