// oCADis (C)opyright 2007 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "filters/filter_ocadis.h"
#include "main.h"
#include "cadmath.h"
#include "object_line.h"
#include "object_circle.h"
#include "object_arc.h"

using namespace os;
using namespace std;
using namespace CadMath;

bool FilteroCADis :: Identify( String& fs, os::String r )
{
	return( r.find( "# HEADER" ) != std::string::npos && r.find( "H, \"oCADis 2.0\"" ) != std::string::npos );
}
	
void FilteroCADis :: Open( String& path, Document& doc )
{
	// Open file
	ifstream fin( path.c_str() );

	// Check that file is opened
	if( !fin.is_open() )
		throw String( "Unable to open file\n" );

	Layer* layer = NULL;
	Style current_style;

	// For each line
	String line;
	if( !_ReadLine( fin, line ) )
		throw String( "Unable to read header" );

	// Check for valid header
	vector<String> split;
	_SplitLine( line, split );

	if( ! ( split[ 0 ] == "H" && split[ 1 ] == "oCADis 2.0" ) )
		throw String( "This is not a oCADis file!" );

	while( _ReadLine( fin, line ) )
	{
		vector<String> split;

		_SplitLine( line, split );

		switch( split[ 0 ][ 0 ] )
		{
			case 'V':   // Viewport
			{
				GET_CANVAS()->SetViewport( Point( Strtod( split[1] ), Strtod( split[2] ) ), 
										   Strtod( split[3] ) );
				break;
			}
			case 'L':   // Layer
			{
				layer = new Layer( String( split[ 1 ] ), 	// Name of layer
								   strtod( split[2].c_str(), NULL ), strtod( split[3].c_str(), NULL ), strtod( split[4].c_str(), NULL ), // Colour of layer
								   strtod( split[5].c_str(), NULL ),  		// Linewidth of layer
								   split[6],								// Linepattern
								   (bool) strtod( split[7].c_str(), NULL ), // Visible
								   (bool) strtod( split[8].c_str(), NULL )); // Editable
				doc.AddLayer( layer );
				doc.SetCurrentLayer( layer );
				break;
			}
			case 'G':	// Snap & ortho
			{
				doc.SetSnapActive( (int) Strtod( split[ 1 ] ) );
				doc.SetSnapOrigo( Point( Strtod( split[ 2 ] ), Strtod( split[ 3 ] ) ) );
				doc.SetSnapDistance( Point( Strtod( split[ 5 ] ), Strtod( split[ 5 ] ) ) );
				doc.SetSnapAngle( Strtod( split[ 6 ] ) );
				doc.SetOrthoMode( (enum Document::OrthoMode) Strtod( split[ 7 ] ) );
				break;
			}
			case 'R':	// GRid
			{
				doc.SetGridActive( (int) Strtod( split[ 1 ] ) );
				doc.SetGridSkip( Point( Strtod( split[ 2 ] ), Strtod( split[ 3 ] ) ) );
				break;
			}
			case 'S':   // Current style
			{
				current_style = doc.GetCurrentStyle();
				current_style.SetStyle( 1, split );
				// Dont set style yet as we have no current layer yet
				break;
			}
			case 'P':   // Pattern
			{
				Pattern* pattern = new Pattern( String( split[ 1 ] ) );
				doc.AddPattern( pattern );
				pattern->SetOffset( Strtod( split[2] ) );
				pattern->SetScale( Strtod( split[3] ) );

				for( int i = 0 ; i < (int)Strtod( split[4] ) ; i++ )
					pattern->AddPattern( Strtod( split[ i + 5 ] ) );
				break;
			}
			case 'O':   // Object
			{
				switch( split[ 0 ][ 1 ] )
				{
					case 'L':	// Line
						ObjectLine :: Load( &doc, layer, split );
						break;
					case 'A':	// Arc
						ObjectArc :: Load( &doc, layer, split, false );
						break;
					case 'C':	// Circle
						ObjectCircle :: Load( &doc, layer, split );
						break;
					default:
						printf( "Unknown object type \"%c\"\n", split[ 0 ][ 1 ] );
						break;
				}

				break;
			}
			default:
				printf( "Unknown token \"%c\"\n", split[ 0 ][ 0 ] );
				break;
		}
	}

	// Close the file afterwards
	fin.close();

	// Set up a few pointers
	// Should be moved!
	doc.SetCurrentLayer( doc.GetLayer( 0 ) );
	doc.SetCurrentStyle( current_style );
}

void FilteroCADis :: Save( String& path, Document& doc )
{
	// Open file
	ofstream fout( path.c_str() );

	// Check that file is opened
	if( !fout.is_open() )
		throw String( "Unable to save file\n" );

	// Write header
	fout << "# HEADER\n";
	fout << "H, \"oCADis 2.0\"\n";
	
	// Save view port
	fout << "\n# VIEWPORT\n";
	String v;
	v.Format( "V, %f, %f, %f\n", doc.GetPan().x, doc.GetPan().y, doc.GetZoom() );
	fout << string( v );

	// Save snap & grid info
	fout << "\n# SNAP AND GRID\n";
	String g;
	g.Format( "G, %d, %f, %f, %f, %f, %f, %d\nR, %d, %f, %f\n", 
			  doc.GetSnapActive(), doc.GetSnapOrigo().x, doc.GetSnapOrigo().y, doc.GetSnapDistance().x, doc.GetSnapDistance().y, doc.GetSnapAngle(), 
			  doc.GetOrthoMode(),
			  doc.GetGridActive(), doc.GetGridSkip().x, doc.GetGridSkip().y );
	fout << string( g );

	// Save current style
	fout << "\n# CURRENT STYLE\n";
	fout << "S" << string( doc.GetCurrentStyle().SaveAttributes() );

	// Line patterns
	fout << "\n# LINEPATTERNS\n";
	for( int i = 0 ; i < doc.GetPatternCount() ; i++ )
	{
		fout << string( doc.GetPattern( i )->Save() );
	}

	// For each layer
	fout << "\n# LAYERS AND OBJECTS\n";
	for( int i = 0 ; i < doc.GetLayerCount() ; i ++ )
	{
		Layer* layer = doc.GetLayer( i );
		fout << string( layer->Save() );

		// Save all objects
		for( int j = 0 ; j < layer->GetObjectCount();  j++ )
		{
			fout << "  " << string( layer->GetObject( j )->Save() );
		}
	}

	fout.flush();
	fout.close();
}

///////////////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E   M E T H O D S
//
///////////////////////////////////////////////////////////////////////////////////////

bool FilteroCADis :: _ReadLine( ifstream& fin, String& result )
{
	do
	{
		result="";
		bool exit = true;
		bool inside = false;	
		char ch;

		while( exit )
		{

			if( !fin.good() )
				return false;

			ch = fin.get();

			if( !inside )
			{
				switch( ch )
				{
					case '\n':
						exit = false;
						break;
					case '#':
						// Read to next line
						while( fin.good() && fin.get() !='\n' );
						exit = false;
						break;
					case '\"':
						inside = true;
						break;
					default:
						if( !isspace( ch ) )
							result += ch;
						break;
				}
			}
			else
			{
				switch( ch )
				{
					case '\"':
						inside = false;
						break;
					default:
						result += ch;
						break;
				}
			}
		}

		result.Strip();

	} while( result == "" );
	
	return true;
}

void FilteroCADis :: _SplitLine( String line, vector< String >& split )
{
	split.clear();
	String add;

	for( uint i = 0 ; i < line.size(); i++ )
	{
		if( line[ i ] == ',' )
		{
			split.push_back( add );
			add = "";
		}
		else
			add += line[ i ];
	}
	
	// Add also the last word
	split.push_back( add );
}
