// oCADis (C)opyright 2007 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <util/string.h>
#include <util/message.h>

#include "main.h"
#include "dialogs/dialog_layer.h"
#include "windialogs/windia_alert.h"
#include "postoffice.h"
#include "messages.h"
#include "cadmath.h"
#include "common.h"

#define ALERT_DELETE_LAYER 0x12333

using namespace os;

class LayerListRow : public ListViewRow
{
public:
	#define COLUMN_VISIBLE 0
	#define COLUMN_EDITABLE 1
	#define COLUMN_NAME 2

	LayerListRow( Layer* layer )
	{
		m_Layer = layer;

		m_Editable = LoadImage( "layer_editable.png" );
		m_NotEditable = LoadImage( "layer_not_editable.png" );
		m_Visible = LoadImage( "layer_visible.png" );
		m_NotVisible = LoadImage( "layer_not_visible.png" );

		m_Current = false;
	}

	~LayerListRow()
	{
		delete m_Editable;
		delete m_NotEditable;
		delete m_Visible;
		delete m_NotVisible;
	}

	void AttachToView( View* pcView, int nColumn )
	{
	}

	void SetRect( const Rect& cRect, int nColumn )
	{
	}
    
	float GetWidth( View* pcView, int nColumn )
	{
		switch( nColumn )
		{
			case COLUMN_VISIBLE:
				return CadMath::max( m_Visible->GetSize().x, m_NotVisible->GetSize().x );
			case COLUMN_EDITABLE:
				return CadMath::max( m_Editable->GetSize().x, m_NotEditable->GetSize().x );
			case COLUMN_NAME:
				return 0.0f;
		}

		return 0.0f;	// Should not happen!
	}

	float GetHeight( View* pcView )
	{
		float height;

		height = CadMath::max( m_Visible->GetSize().y, m_NotVisible->GetSize().y );
		height = CadMath::max( height, m_Editable->GetSize().y );
		height = CadMath::max( height, m_NotEditable->GetSize().y );

		font_height sHeight;

		pcView->GetFontHeight( &sHeight );

		height = CadMath::max( height,sHeight.ascender + sHeight.descender );

		return height + 2.0f;
	}

	void Paint( const Rect& cFrame, View* pcView, uint nColumn,
			 	bool bSelected, bool bHighlighted, bool bHasFocus )
	{
		pcView->SetFgColor( 255, 255, 255 );
		pcView->FillRect( cFrame );

		if( bSelected || bHighlighted )
		{
			Rect cSelectFrame = cFrame;
			if( nColumn == 0 )
			{
				cSelectFrame.left += 2;
				cSelectFrame.top += 2;
				cSelectFrame.bottom -= 2;
			}

			if( bSelected )
				pcView->SetFgColor( 186, 199, 227 );
			else
				pcView->SetFgColor( 0, 50, 200 );
			pcView->FillRect( cSelectFrame );
	
			// Round edges
			if( nColumn == 0 )
			{
				pcView->DrawLine( os::Point( cSelectFrame.left + 2, cSelectFrame.top - 2 ), 
				  				  os::Point( cSelectFrame.right, cSelectFrame.top - 2 ) );
				pcView->DrawLine( os::Point( cSelectFrame.left, cSelectFrame.top - 1 ), 
								  os::Point( cSelectFrame.right, cSelectFrame.top - 1 ) );
			
				pcView->DrawLine( os::Point( cSelectFrame.left - 2, cSelectFrame.top + 2 ), 
								  os::Point( cSelectFrame.left - 2, cSelectFrame.bottom - 2 ) );
				pcView->DrawLine( os::Point( cSelectFrame.left - 1, cSelectFrame.top ), 
								  os::Point( cSelectFrame.left - 1, cSelectFrame.bottom ) );
								
				pcView->DrawLine( os::Point( cSelectFrame.left + 2, cSelectFrame.bottom + 2 ), 
								  os::Point( cSelectFrame.right, cSelectFrame.bottom + 2 ) );
				pcView->DrawLine( os::Point( cSelectFrame.left, cSelectFrame.bottom + 1 ), 
								  os::Point( cSelectFrame.right, cSelectFrame.bottom + 1 ) );
			} 
		}

		switch( nColumn )
		{
			case COLUMN_VISIBLE:
			{
				pcView->SetDrawingMode( DM_BLEND );

				if( m_Layer->IsVisible() )
					m_Visible->Draw( Point( cFrame.left + 3.0f,  cFrame.top + 2.0f ), pcView );
				else
					m_NotVisible->Draw( Point( cFrame.left + 3.0f,  cFrame.top + 2.0f ), pcView );

				break;	
			}
			case COLUMN_EDITABLE:
			{
				pcView->SetDrawingMode( DM_BLEND );

				if( m_Layer->IsEditable() )
					m_Editable->Draw( Point( cFrame.left + 3.0f,  cFrame.top + 2.0f ), pcView );
				else
					m_NotEditable->Draw( Point( cFrame.left + 3.0f,  cFrame.top + 2.0f ), pcView );
				break;	
			}
			case COLUMN_NAME:
			{
				pcView->SetDrawingMode( DM_OVER );
				font_height sHeight;
				Font* f = pcView->GetFont();

				f->GetHeight( &sHeight );
				float vFontHeight = sHeight.ascender + sHeight.descender;
				float vBaseLine = cFrame.top + ( cFrame.Height() + 1.0f ) / 2 - vFontHeight / 2 + sHeight.ascender;

				if( m_Current )
					f->SetFlags( FPF_ITALIC | FPF_BOLD | FPF_SMOOTHED );
				else
					f->SetFlags( FPF_SMOOTHED );

				pcView->SetFont( f );

				if( bHighlighted )
				{
					pcView->SetFgColor( 255, 255, 255 );
					pcView->SetBgColor( 0, 50, 200 );
				}
				else if( bSelected )
				{
					pcView->SetFgColor( 0, 0, 0 );
					pcView->SetBgColor( 186, 199, 227 );
				}
				else
				{
					pcView->SetBgColor( 255, 255, 255 );
					pcView->SetFgColor( 0, 0, 0 );
				}

				pcView->MovePenTo( cFrame.left + 3.0f, vBaseLine );
				pcView->DrawString( m_Layer->GetName() );				
				break;	
			}
			default:
				break;
		}
	}

	bool HitTest( View* pcView, const Rect& cFrame, int nColumn, Point cPos )
	{
		return true;
	}

	bool IsLessThan( const ListViewRow* pcOther, uint nColumn ) const
	{
		return false;
	}

	void SetCurrent( bool value )
	{
		m_Current = value;
	}

	bool GetCurrent()
	{
		return m_Current;
	}

private:
	Layer* m_Layer;

	// Set to true if this is the current layer
	bool m_Current;

	// Some images
	Image* m_Editable;
	Image* m_NotEditable;

	Image* m_Visible;
	Image* m_NotVisible;
};

DialogLayer :: DialogLayer( DialogSideBar* parent ) : Dialog( parent, String( "Layer" ) )
{
	// Create the mailboxes
	AddMailbox( "AddLayer" );
	AddMailbox( "Document" );

	// Create the table
	m_theTable = new TableView( Rect(), "", 3, 3, false );
	m_theTable->SetRowSpacings( 4 );
	AddChild( m_theTable );

	// Create the list
	m_LayerList = new ListView( Rect(), "", ListView::F_RENDER_BORDER );
	m_LayerList->SetAutoSort( false );
	m_LayerList->InsertColumn( "Visible", 20 );
	m_LayerList->InsertColumn( "Editable", 20 );
	m_LayerList->InsertColumn( "Name", 1 );
	m_LayerList->SetHasColumnHeader( false );
	m_LayerList->SetInvokeMsg( new Message( MSG_LISTVIEW_INVOKE ) );
	m_theTable->Attach( m_LayerList, 0, 3, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), ( TABLE_EXPAND | TABLE_FILL ) );

	// Create the buttons
	m_Up = new Button( Rect(), "", "Up", new Message( MSG_UP ) );
	m_theTable->Attach( m_Up, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0, 4, 4 );

	m_Current = new Button( Rect(), "", "Current", new Message( MSG_CURRENT ) );
	m_theTable->Attach( m_Current, 1, 2, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0, 4, 4 );

	m_Down = new Button( Rect(), "", "Down", new Message( MSG_DOWN ) );
	m_theTable->Attach( m_Down, 2, 3, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0, 4, 4 );

	m_New = new Button( Rect(), "", "New", new Message( MSG_NEW ) );
	m_theTable->Attach( m_New, 0, 1, 2, 3, ( TABLE_EXPAND | TABLE_FILL ), 0, 4, 4 );

	m_Edit = new Button( Rect(), "", "Edit", new Message( MSG_EDIT ) );
	m_theTable->Attach( m_Edit, 1, 2, 2, 3, ( TABLE_EXPAND | TABLE_FILL ), 0, 4, 4 );

	m_Delete = new Button( Rect(), "", "Delete", new Message( MSG_DELETE ) );
	m_theTable->Attach( m_Delete, 2, 3, 2, 3, ( TABLE_EXPAND | TABLE_FILL ), 0, 4, 4 );

	_Layout();

	// Set up pointers
	m_WinDiaAddLayer = NULL;
	m_WinDiaModifyLayer = NULL;
	m_CurrentDocument = NULL;
	_UpdateLayers();
}

DialogLayer ::~DialogLayer()
{
	// Remove the mailbox
	RemoveMailbox( "AddLayer" );
	RemoveMailbox( "Document" );
	
	// Delete dialogs
	delete m_WinDiaAddLayer;
	delete m_WinDiaModifyLayer;
}

void DialogLayer :: SetFrame( const Rect& cRect, bool bNotifyServer )
{
	Dialog::SetFrame( cRect, bNotifyServer );
	_Layout();
}

void DialogLayer :: AllAttached( void )
{
	Dialog::AllAttached();
	m_LayerList->SetTarget( this );
	m_New->SetTarget( this );
	m_Edit->SetTarget( this );
	m_Delete->SetTarget( this );	
	m_Up->SetTarget( this );
	m_Current->SetTarget( this );
	m_Down->SetTarget( this );
}

void DialogLayer :: HandleMessage( Canvas* canvas, Document* doc, os::Message* pcMessage )
{
	switch( pcMessage->GetCode() )
	{
		case ALERT_DELETE_LAYER:
		{
			int32 t;
			if( pcMessage->FindInt32( "which", &t ) == EOK )
			{
				if( t == 1 ) // Yes to delete
				{
					if( pcMessage->FindInt32( "row", &t ) == EOK )
					{	
						doc->DeleteLayer( t );
						_UpdateLayers();						
					}
				}
			}
			break;
		}
		case MSG_NEW:
		{
			if( m_WinDiaAddLayer == NULL )
				m_WinDiaAddLayer = new WinDiaAddLayer( Rect( 100, 100, 320, 310 ), m_CurrentDocument );
			m_WinDiaAddLayer->Raise();
			break;
		}
		case MSG_EDIT:
		{
			// Find current selection
			int sel = m_LayerList->GetFirstSelected();

			if( sel >= 0 )
			{
				if( m_WinDiaModifyLayer == NULL )
					m_WinDiaModifyLayer = new WinDiaModifyLayer( Rect( 100, 100, 320, 310 ), m_CurrentDocument );

				m_WinDiaModifyLayer->Raise( m_CurrentDocument->GetLayer( sel ) );
			}
			else
				WinDiaAlert::ShowErrorAlert( "Edit layer", "No layer has been selected." );
			break;
		}
		case MSG_DELETE:
		{
			_Delete();					
			break;
		}
		
		case MSG_UP:
			_Up();
			break;

		case MSG_CURRENT:
			_SetCurrent();
			break;

		case MSG_DOWN:
			_Down();
			break;

		case MSG_LISTVIEW_INVOKE:
			_Invoke();
			break;

		case MSG_BROADCAST_MODIFY_LAYER:
		case MSG_BROADCAST_ADD_LAYER:
			_UpdateLayers();
			break;
	}
}

void DialogLayer :: SetDocument( Document* doc )
{
	Dialog::SetDocument( doc );
	_UpdateLayers();
}

/////////////////////////////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E   M E T H O D S
//
/////////////////////////////////////////////////////////////////////////////////////////////////////

void DialogLayer :: _Layout()
{
	Rect cFrame = GetBounds();
	m_theTable->SetFrame( cFrame );
}

void DialogLayer :: _UpdateLayers()
{
	if( m_CurrentDocument == NULL )
		return;

	m_LayerList->Clear();

	// For each layer
	for( int i = 0 ; i < m_CurrentDocument->GetLayerCount() ; i ++ )
	{
		Layer* layer = m_CurrentDocument->GetLayer( i );

		LayerListRow* row = new LayerListRow( layer );

		row->SetCurrent( m_CurrentDocument->GetCurrentLayer() == layer );
		m_LayerList->InsertRow( row );
	}
	m_LayerList->Invalidate();
	m_LayerList->Sync();
}

void DialogLayer :: _SetCurrent()
{
	if( m_CurrentDocument == NULL )
	{
		WinDiaAlert::ShowErrorAlert( "Set current layer", "No valid document." );		
		return;
	}

	// Find current selection
	int sel = m_LayerList->GetFirstSelected();
	if( sel < 0 )
	{
		WinDiaAlert::ShowErrorAlert( "Set current layer", "No layer has been selected." );
		return;
	}

	Layer* layer = m_CurrentDocument->GetLayer( sel );
	if( layer == NULL )
	{
		WinDiaAlert::ShowErrorAlert( "Set current layer", "Cannot find selected layer in document." );
		return;
	}

	// Make sure this layer is not already selected
	if( layer == m_CurrentDocument->GetCurrentLayer() )
	{
		WinDiaAlert::ShowErrorAlert( "Set current layer", "The selected layer is already current layer." );
		return;
	}

	// Clear old current layer
	for( uint i = 0 ; i < m_LayerList->GetRowCount() ; i++ )
	{
		LayerListRow* l = dynamic_cast< LayerListRow* >( m_LayerList->GetRow( i ) );

		if( l != NULL && l->GetCurrent() )
		{
			l->SetCurrent( false );
			m_LayerList->InvalidateRow( i, 0 );
			break;	
		}
	}

	// Set new current layer in listview
	LayerListRow* l = dynamic_cast< LayerListRow* >( m_LayerList->GetRow( sel ) );
	l->SetCurrent( true );
	m_LayerList->InvalidateRow( sel, 0 );

	// Set current layer in document
	m_CurrentDocument->SetCurrentLayer( layer );
}

void DialogLayer :: _Up()
{
	if( m_CurrentDocument == NULL )
	{
		WinDiaAlert::ShowErrorAlert( "Move layer", "No valid document." );
		return;
	}

	// Find current selection
	int sel = m_LayerList->GetFirstSelected();

	// Make some validation of the selected layer
	if( sel < 0 )
	{
		WinDiaAlert::ShowErrorAlert( "Move layer", "No layer has been selected." );
		return;
	}

	if( m_CurrentDocument->GetLayerCount() < 2 )
	{
		WinDiaAlert::ShowErrorAlert( "Move layer", "Only one layer available." );
		return;
	}

	if( sel == 0 )
	{
		WinDiaAlert::ShowErrorAlert( "Move layer", "This layer is already the top layer" );
		return;
	}

	// Swap layers
	m_CurrentDocument->SwapLayers( sel, sel - 1 );

	_UpdateLayers();

	// Set selection
	m_LayerList->Select( sel - 1 );
}

void DialogLayer :: _Down()
{
	if( m_CurrentDocument == NULL )
	{
		WinDiaAlert::ShowErrorAlert( "Move layer", "No valid document." );
		return;
	}

	// Find current selection
	int sel = m_LayerList->GetFirstSelected();

	// Make some validation of the selected layer
	if( sel < 0 )
	{
		WinDiaAlert::ShowErrorAlert( "Move layer", "No layer has been selected." );
		return;
	}

	if( m_CurrentDocument->GetLayerCount() < 2 )
	{
		WinDiaAlert::ShowErrorAlert( "Move layer", "Only one layer available." );
		return;
	}

	if( sel == m_CurrentDocument->GetLayerCount() - 1 )
	{
		WinDiaAlert::ShowErrorAlert( "Move layer", "This layer is already the bottom layer" );
		return;
	}

	// Swap layers
	m_CurrentDocument->SwapLayers( sel, sel + 1 );

	_UpdateLayers();

	// Set selection
	m_LayerList->Select( sel + 1 );
}

void DialogLayer :: _Delete()
{
	if( m_CurrentDocument == NULL )
	{
		WinDiaAlert::ShowErrorAlert( "Move layer", "No valid document." );
		return;
	}

	// Find current selection
	int sel = m_LayerList->GetFirstSelected();

	// Make some validation of the selected layer
	if( sel < 0 )
	{
		WinDiaAlert::ShowErrorAlert( "Delete layer", "No layer has been selected." );
		return;
	}

	if( m_CurrentDocument->GetLayerCount() < 2 )
	{
		WinDiaAlert::ShowErrorAlert( "Delete layer", "Only one layer available. The last\nlayer cannot be deleted." );
		return;
	}

	WinDiaAlert* t = new WinDiaAlert( WinDiaAlert::ALERT_WARNING, "Delete layer", "Are you sure that you want to delete this layer?", "No", "Yes", NULL );
	Message* msg = new Message( ALERT_DELETE_LAYER );		
	msg->AddInt32( "row", sel );
	t->Go( new Invoker( msg, this ) );	
}

void DialogLayer :: _Invoke()
{
	printf( "invoke\n" );
}

