// EFileBrowser (C)opyright 2007 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <storage/registrar.h>
#include <storage/volumes.h>
#include <gui/imagebutton.h>
#include <gui/stringview.h>
#include <gui/layoutview.h>
#include "common.h"
#include "address_field.h"
#include "address_field_button.h"

using namespace os;

///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E
//
///////////////////////////////////////////////////////////////////////////////

static uint8 g_aCDImage[] = {
#include "images/cd.h"
};

static uint8 g_aDiskImage[] = {
#include "images/disk.h"
};

static uint8 g_aFloppyImage[] = {
#include "images/floppy.h"
};

uint8 g_aFileImage[] = {
#include "images/file.h"
};

uint8 g_aFolderImage[] = {
#include "images/folder.h"
};

class AddressField :: _Private
{
public:
	_Private()
	{
		try
		{
			m_Manager = RegistrarManager::Get();
		} 
		catch( ... ) 
		{ 
			m_Manager = NULL; 
		}
	};

	// Code from IconDirectoryView
	BitmapImage* _GetDriveIcon( os::String zPath, fs_info* psInfo ) const
	{	
		BitmapImage* pcItemIcon = NULL;
		Volumes cVolumes;
		String cMountPoint;
	
		for( uint i = 0; i < cVolumes.GetMountPointCount(); i++ )
		{
			if( cVolumes.GetMountPoint( i, &cMountPoint ) == 0 )
			{
				if( zPath == cMountPoint )
				{
					fs_info sInfo;
					if( cVolumes.GetFSInfo( i, &sInfo ) == 0 && ( sInfo.fi_flags & FS_IS_BLOCKBASED ) )
					{
						if( psInfo != NULL )
							*psInfo = sInfo;
				
						String zDevice = os::Path( sInfo.fi_device_path ).GetDir().GetLeaf();
						if( zDevice.Length() > 2 )
						{
							os::StreamableIO* pcSource = NULL;
							if( zDevice[0] == 'c' && zDevice[1] == 'd' )
							{
								try {
									pcSource = new os::File( "/system/icons/cdrom.png" );
								} catch( ... ) {
									pcSource = new MemFile( g_aCDImage, sizeof( g_aCDImage ) );
								}
							}
							else if( zDevice[0] == 'f' && zDevice[1] == 'd' )
							{
								try {
									pcSource = new os::File( "/system/icons/floppy.png" );
								} catch( ... ) {
									pcSource = new MemFile( g_aFloppyImage, sizeof( g_aFloppyImage ) );
								}
							}
							else 
							{
								try {
									pcSource = new os::File( "/system/icons/harddisk.png" );
								} catch( ... ) {
									pcSource = new MemFile( g_aDiskImage, sizeof( g_aDiskImage ) );
								}
							}
						
							pcItemIcon = new os::BitmapImage();
									
							pcItemIcon->Load( pcSource );
							delete( pcSource );						
						}
					}
				}
			}
		}
	
		if( pcItemIcon == NULL )
		{
			/* Default icon */
			MemFile* pcSource = new MemFile( g_aDiskImage, sizeof( g_aDiskImage ) );
			pcItemIcon = new os::BitmapImage();
			pcItemIcon->Load( pcSource );
			delete( pcSource );
		}
	
		if( pcItemIcon && !( pcItemIcon->GetSize() == os::Point( 24, 24 ) ) )
			pcItemIcon->SetSize( os::Point( 24, 24 ) );
	
		return( pcItemIcon );
	}	

	// Code from IconDirectoryView
	Image* _GetDirIcon( String path ) const
	{
		// Look if the path is a mountpoint
		os::Volumes cVolumes;
		os::String cMountPoint;
		for( uint i = 0; i < cVolumes.GetMountPointCount(); i++ )
		{
			if( cVolumes.GetMountPoint( i, &cMountPoint ) == 0 )
			{
				if( path == cMountPoint )
					return _GetDriveIcon( m_Path, NULL );
			}
		}

		String zTempType;
		Image* pcIcon;
	
		if( m_Manager && m_Manager->GetTypeAndIcon( path, os::Point( 24, 24 ),
			&zTempType, &pcIcon ) != 0)
		{
			BitmapImage* pcBitmap = new BitmapImage();
		
			MemFile* pcSource = new MemFile( g_aFolderImage, sizeof( g_aFolderImage ) );
							
			pcBitmap->Load( pcSource );
			delete( pcSource );
			pcIcon = pcBitmap;
		}

		return( pcIcon );
	}

	void _AddButton( String label, String path )
	{
		// Make message for button
		Message* msg = NULL;

		if( m_theMessage != NULL )
		{
			msg = new Message( *m_theMessage );
			msg->AddString( "file/path", path );
		}

		// Create new button
		AddressFieldButton* button = new AddressFieldButton( Rect(), label, _GetDirIcon( path ), msg );
		m_ListOfButtons.push_back( button );
		m_RootNode->AddChild( button );
	}

	void _AddDistance()
	{
		HLayoutSpacer* space = new HLayoutSpacer( "space" );
		space->LimitMaxSize( Point( 10, 100 ) );
		m_RootNode->AddChild( space );
	}

	void _Clean()
	{
		// Clear old buttons
		m_ListOfButtons.clear();
		//delete m_RootView->GetRoot();
		m_RootNode = new HLayoutNode( "pcRootNode" );
		m_RootNode->SetBorders( Rect( 4, 4, 4, 4 ) );		
		m_RootView->SetRoot( m_RootNode );
	}

	void _UpdatePath()
	{
		_Clean();
	
		String path = m_Path;
		String button_path;

		// Sanity check
		if( path[0] != '/' )
			throw String("Not a valid path");

		// Remove the very first separator
		path = String( path, 1, path.size() );

		// Create root button
		_AddButton( String( "/" ), String( "/" ) );
		
		while( path != "" )
		{
			_AddDistance();

			// Find next directory splitter "/"
			size_t s = path.find( "/" );

			// Extract folder
			String label = String(path, 0, s);

			// Remove it from the path
			path = String( path, s+1, path.size() );

			// Create nice path for the buttons
			button_path +=  String("/") + label;

			_AddButton( label, button_path );
	
			// No more path separator found
			if( s == std::string::npos)
				break;
		}

		m_RootView->InvalidateLayout();
	}


	std::vector< AddressFieldButton* > m_ListOfButtons;

	LayoutView*	m_RootView;
	LayoutNode*	m_RootNode;
	RegistrarManager* m_Manager;
	String m_Path;
	Message* m_theMessage;
	Handler* m_theHandler;
	Looper* m_theLooper;
};

///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S
//
///////////////////////////////////////////////////////////////////////////////

AddressField :: AddressField( const String path, Message* msg ) : View( Rect(), "" )
{
	// Create the private class
	m = new _Private();

	// Create widgets
	m->m_RootNode = new HLayoutNode( "pcRootNode" );
	m->m_RootNode->SetBorders( Rect( 4, 4, 4, 4 ) );

	m->m_RootView = new LayoutView( GetBounds(), "pcRootView" );
	m->m_RootView->SetRoot( m->m_RootNode );
	AddChild( m->m_RootView );

	m->m_Path = path;
	m->m_theHandler=NULL;
	m->m_theLooper=NULL;
	m->m_theMessage = msg;
	m->_UpdatePath();	

	SetTarget( m->m_theHandler, m->m_theLooper );
}

AddressField :: ~AddressField()
{
	delete m->m_theMessage;
	delete m;
}

void AddressField :: SetPath( os::String path )
{
	m->m_Path = path;
	m->_UpdatePath();
	SetTarget( m->m_theHandler, m->m_theLooper );
}

String AddressField :: GetPath()
{
	return m->m_Path;
}

Point AddressField :: GetPreferredSize( bool bLargest ) const
{
	if( bLargest )
		return Point( COORD_MAX, COORD_MAX );

	return Point( COORD_MAX, 35 );
}

status_t AddressField :: SetTarget( const os::Handler* pcHandler, const Looper* pcLooper )
{
	m->m_theHandler = (os::Handler*) pcHandler;
	m->m_theLooper = (os::Looper*) pcLooper;

	// Set target for the all button
	for( uint i = 0 ; i < m->m_ListOfButtons.size() ; i++ )
		m->m_ListOfButtons[ i ]->SetTarget( m->m_theHandler, m->m_theLooper );

	return Invoker::SetTarget( m->m_theHandler, m->m_theLooper );
}

