// EFileBrowser (C)opyright 2007 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <map>
#include <storage/registrar.h>
#include <storage/fsnode.h>
#include <storage/file.h>
#include "thumbsdb.h"
#include "../common.h"

using namespace os;
using namespace std;

///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E   -   T H U M B N A I L
//
///////////////////////////////////////////////////////////////////////////////
class Thumbnail :: _Private
{
public:
	_Private() {};

	enum ThumbnailStatus m_Status;
	String m_Path;
	String m_CreateDate;
	Image* m_Image;	
};


///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S - T H U M B N A I L
//
///////////////////////////////////////////////////////////////////////////////

Thumbnail :: Thumbnail( String path )
{
	// Create the private class
	m = new Thumbnail::_Private();

	m->m_Status = NO_IMAGE;
	m->m_Path = path;

	m->m_Image = LoadImage( "image-loading.png" );
}

Thumbnail :: ~Thumbnail()
{
	delete m;
}

enum Thumbnail :: ThumbnailStatus Thumbnail :: GetStatus()
{
	return m->m_Status;
}

void Thumbnail :: SetStatus( enum ThumbnailStatus status )
{
	m->m_Status = status;
}

String Thumbnail :: GetPath() 
{ 
	return m->m_Path; 
}

Image* Thumbnail :: GetDuplicatedImage()
{
	return new BitmapImage( (const BitmapImage&) *(m->m_Image) );
}

void Thumbnail :: SetImage( Image* img )
{
	if( m->m_Image != NULL )
		delete m->m_Image;

	SetStatus( READY );
	m->m_Image = img;

printf( "Setting image for %s\n", m->m_Path.c_str() );
}

///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E   -   T H U M B S D B
//
///////////////////////////////////////////////////////////////////////////////

class ThumbsDB :: _Private
{
public:
	_Private() {};
	
	void _AddThumbnail( Thumbnail* thumb )
	{
		m_Thumbnails[ thumb->GetPath() ] = thumb;
	}

	Thumbnail* _CreateThumbnail( os::String path )
	{
		Thumbnail* thumb = NULL;

		try
		{
			thumb  = new Thumbnail( path );
			_AddThumbnail( thumb );
		}
		catch( ... ) { }

		return thumb;
	}

	RegistrarManager* m_Manager;

	map< String, Thumbnail* > m_Thumbnails;
	
};

///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S   -   T H U M B S D B
//
///////////////////////////////////////////////////////////////////////////////
ThumbsDB* ThumbsDB :: m_Instance = NULL;

ThumbsDB :: ThumbsDB() : Locker( "thumbsdb_lock" )
{
	// Create the private class
	m = new ThumbsDB :: _Private();

	m->m_Manager = RegistrarManager::Get();

}

ThumbsDB :: ~ThumbsDB()
{
	m->m_Manager->Put();

	delete m;
}

ThumbsDB* ThumbsDB :: Get()
{
	if( m_Instance == NULL )
		m_Instance = new ThumbsDB();

	return m_Instance;
}

Image* ThumbsDB :: GetThumbnail( String path, bool& update )
{
	// Does the thumbnail already exists?
	Lock();
	Thumbnail* thumb = m->m_Thumbnails[ path ];
	Unlock();

	if( thumb != NULL )
	{
		update = ( thumb->GetStatus() == Thumbnail::NO_IMAGE ) || ( thumb->GetStatus() == Thumbnail::PROCESSING );
		return thumb->GetDuplicatedImage();
	}

	return NULL;
}

Image* ThumbsDB :: GetUpdatedThumbnail( String path )
{
	Lock();
	Thumbnail* thumb = m->m_Thumbnails[ path ];
	Unlock();

	if( thumb != NULL )
	{
		if( thumb->GetStatus() == Thumbnail::READY )
			return thumb->GetDuplicatedImage();
	}

	return NULL;
}

Thumbnail* ThumbsDB :: CreateThumbnail( os::String path )
{
	String mime, type;

	// Does the thumbnail already exists?
	Thumbnail* thumb = m->m_Thumbnails[ path ];
	
	if( thumb != NULL )
		return thumb;

	// Check mime type
	// Today we can only create thumbnails of image
	try
	{		
		Message msg;
		m->m_Manager->GetTypeAndIcon( path, os::Point( 48, 48 ), &type, NULL, &msg );
		msg.FindString( "mimetype", &mime );
	} catch( ... )
	{
		return NULL;
	}

	if( mime.find( "image/" ) != 0 )
		return NULL;

	Lock();

	thumb = m->_CreateThumbnail( path );

	Unlock();

	return thumb;
}
