/*  libsyllable.so - the highlevel API library for Syllable
 *  Copyright (C) 1999 - 2001 Kurt Skauen
 *  Copyright (C) 2003 The Syllable Team
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of version 2 of the GNU Library
 *  General Public License as published by the Free Software
 *  Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 *  MA 02111-1307, USA
 */

#include "dirkeeper.h"
#include "thumbsdb.h"
#include "fileview.h"
#include <atheos/time.h>
#include <storage/volumes.h>

using namespace os;
using namespace os_priv;

#define LAYOUT_TIME 800000

DirKeeper::DirKeeper( const Messenger & cTarget, const String& cPath ):Looper( "dir_worker" ), m_cTarget( cTarget )
{
	m_bRunning = false;
	m_pcCurrentDir = NULL;
	try
	{
		m_cPath = cPath;
		m_pcCurrentDir = new Directory( cPath );
		m_pcCurrentDir->Rewind();
		m_cMonitor.SetTo( m_pcCurrentDir, NWATCH_DIR | NWATCH_NAME, this, this );
	} catch( std::exception & )
	{
	}
	m_eState = S_IDLE;
	m_nPendingReplies = 0;
	m_bLayoutNecessary = false;
	m_bWaitForLayoutReply = false;
	m_nLastLayout = get_system_time() - LAYOUT_TIME;
	m_Preview = false;
	m_ThumbCreator = NULL;
}

bool DirKeeper:: OkToQuit()
{
	if( m_ThumbCreator )
	{
		m_ThumbCreator->Stop();
		m_ThumbCreator->Terminate();
	}

	return true;
}

status_t DirKeeper::GetNode( os::String zPath, os::FSNode* pcNode, bool* pbBrokenLink )
{	
	if( pcNode->SetTo( *m_pcCurrentDir, zPath, O_RDONLY | O_NONBLOCK ) >= 0 )
	{
		*pbBrokenLink = false;
		return( 0 );
	}
	
	if( pcNode->SetTo( *m_pcCurrentDir, zPath, O_RDONLY | O_NONBLOCK | O_NOTRAVERSE ) >= 0 )
	{
		*pbBrokenLink = true;		
		return( 0 );
	}
		
	return( -EIO );
}

void DirKeeper::HandleMessage( Message * pcMessage )
{
	switch ( pcMessage->GetCode() )
	{
	case M_CHANGE_DIR:
		{
			String cNewPath;
			pcMessage->FindString( "path", &cNewPath );

			// Clear our thumbnail creator
		/*	if(  m_ThumbCreator != NULL )
			{
				m_ThumbCreator->Lock();
				m_ThumbCreator->Stop();
				m_ThumbCreator->Terminate();
				delete m_ThumbCreator;
				m_ThumbCreator = NULL;
			}

	
			// Create a new thread creator
			if( m_Preview )
			{
				m_ThumbCreator = new ThumbCreator( this, cNewPath );
				m_ThumbCreator->Lock();
				m_ThumbCreator->Start();
				m_ThumbCreator->Unlock();
			}
*/
			try
			{
				Stop();
				m_cPath = cNewPath;
				Directory *pcNewDir = new Directory( cNewPath );
				delete m_pcCurrentDir;

				m_pcCurrentDir = pcNewDir;
				m_pcCurrentDir->Rewind();
				
				m_eState = S_READDIR;
				if( !( cNewPath == "/" ) )
					m_cMonitor.SetTo( m_pcCurrentDir, NWATCH_DIR | NWATCH_NAME, this, this );
				else
					m_cMonitor.Unset();
				
			}
			catch( std::exception & )
			{
			}
			
			break;
		}
	case M_PREVIEW:
		{
			pcMessage->FindBool( "active", &m_Preview );

			// Clear our thumbnail creator
			if( m_ThumbCreator != NULL )
			{
				m_ThumbCreator->Lock();
				m_ThumbCreator->Stop();
				m_ThumbCreator->Terminate();
				delete m_ThumbCreator;
				m_ThumbCreator = NULL;
			}
	
			// Create a new thread creator
			if( m_Preview )
			{
				String path;
				m_pcCurrentDir->GetPath( &path );
				m_ThumbCreator = new ThumbCreator( this, path );
				m_ThumbCreator->Start();
			}
			break;
		}
	case M_ENTRIES_ADDED:
	case M_ENTRIES_UPDATED:
	case M_ENTRIES_REMOVED:				
	case M_THUMBNAIL_LAYOUT_UPDATED:	
		{
			m_nPendingReplies--;
			printf("Pending now %i\n", m_nPendingReplies );
			break;
		}
	case M_LAYOUT_UPDATED:
		{
			m_bWaitForLayoutReply = false;
			m_nLastLayout = get_system_time();
			break;
		}
	case M_THUMBNAIL_LAYOUT:
	{
		printf( "DirKeeper uthumbnail update\n" );
		Message cMsg( FileView::M_THUMBNAIL_LAYOUT );		
		m_cTarget.SendMessage( &cMsg );	
		m_nPendingReplies++;
		break;
	}	
	case M_THUMBNAIL_READY:
	{
		if( m_ThumbCreator != NULL )
		{
			m_ThumbCreator->Lock();
			m_ThumbCreator->Stop();
			m_ThumbCreator->Terminate();
			delete m_ThumbCreator;
			m_ThumbCreator = NULL;
		}
		break;
	}

	case M_NODE_MONITOR:
		{
			int32 nEvent = -1;

			pcMessage->FindInt32( "event", &nEvent );
			switch ( nEvent )
			{
			case NWEVENT_CREATED:
				{
					String cName;
					dev_t nDevice;
					int64 nInode;
					
					//std::cout<<"CREATE"<<std::endl;

					pcMessage->FindInt( "device", &nDevice );
					pcMessage->FindInt64( "node", &nInode );
					pcMessage->FindString( "name", &cName );
					SendAddMsg( cName, nDevice, nInode );
					break;
				}
			case NWEVENT_DELETED:
				{
					dev_t nDevice;
					int64 nInode;
					//std::cout<<"DELETE"<<std::endl;

					String cName;

					pcMessage->FindInt( "device", &nDevice );
					pcMessage->FindInt64( "node", &nInode );
					pcMessage->FindString( "name", &cName );
					SendRemoveMsg( cName, nDevice, nInode );
					break;
				}
			case NWEVENT_STAT_CHANGED:
				{
					dev_t nDevice;
					int64 nInode;

					String cName;

					pcMessage->FindInt( "device", &nDevice );
					pcMessage->FindInt64( "node", &nInode );
					pcMessage->FindString( "name", &cName );
					SendUpdateMsg( cName, nDevice, nInode, false );
					break;
				}
			case NWEVENT_ATTR_WRITTEN:
			case NWEVENT_ATTR_DELETED:
				{
				/* Not implemented yet in the kernel/glibc */
				break;
				}
			case NWEVENT_MOVED:
				{
					dev_t nDevice;
					int64 nOldDir;
					int64 nNewDir;
					int64 nInode;

					String cName;

					pcMessage->FindInt( "device", &nDevice );
					pcMessage->FindInt64( "old_dir", &nOldDir );
					pcMessage->FindInt64( "new_dir", &nNewDir );
					pcMessage->FindInt64( "node", &nInode );
					pcMessage->FindString( "name", &cName );


					if( nInode == int64 ( m_pcCurrentDir->GetInode() ) )
					{
						printf( "%Ld moved from %Ld to %Ld\n", nInode, nOldDir, nNewDir );
					}
					else
					{
						if( nOldDir == nNewDir )
						{
							SendUpdateMsg( cName, nDevice, nInode, true );
						}
						else
						{
							if( nOldDir == int64 ( m_pcCurrentDir->GetInode() ) )
							{
								SendRemoveMsg( cName, nDevice, nInode );
							}
							else
							{
								SendAddMsg( cName, nDevice, nInode );
							}
						}
					}
					break;
				}
			}
		}
		
	default:
		Looper::HandleMessage( pcMessage );
		break;
	}
}

void DirKeeper::SendUpdateMsg( const String& cName, dev_t nDevice, ino_t nInode, bool bReloadIcon )
{
	//std::cout<<"Update "<<cName.c_str()<<std::endl;
	try
	{
		
		bool bBrokenLink;
		struct stat sStat;
		
		if( cName == "" )
		{
			bBrokenLink = false;
			sStat.st_dev = nDevice;
			sStat.st_ino = nInode;
		} else
		{
			FSNode cNode;
			if( GetNode( cName, &cNode, &bBrokenLink ) < 0 )
				return;
			cNode.GetStat( &sStat, false );
		}
			
		
		//std::cout<<"Update "<<cName<<" "<<sStat.st_dev<<" "<<sStat.st_ino<<" "<<nInode<<std::endl;


		//FileNode cFileNode( cName, cName, sStat, bBrokenLink, bReloadIcon );
		
		Message cMsg( FileView::M_UPDATE_ENTRY );
		cMsg.AddString( "name", cName );
		cMsg.AddString( "path", cName );
		cMsg.AddData( "stat", T_ANY_TYPE, &sStat, sizeof( sStat ) );
		cMsg.AddBool( "broken_link", bBrokenLink );
		cMsg.AddBool( "reload_icon", bReloadIcon );
			
			

		m_nPendingReplies++;
		m_cTarget.SendMessage( &cMsg );
		//printf("Send Update %s %i %i!\n", cName.c_str(), (int)nInode, m_nPendingReplies);
	} catch( std::exception & )
	{
		m_cTarget.SendMessage( FileView::M_REREAD );
	}
	m_bLayoutNecessary = true;
}

void DirKeeper::SendAddMsg( const String& cName, dev_t nDevice, ino_t nInode )
{
	try
	{
		
		FSNode cNode;
		bool bBrokenLink = true;
		os::NodeMonitor* pcMonitor = NULL;
		struct stat sStat;
		memset( &sStat, 0, sizeof( sStat ) );
		
		if( GetNode( cName, &cNode, &bBrokenLink ) >= 0 )
			cNode.GetStat( &sStat, false );
		else {
			if( nDevice == -1 && nInode == -1 )
				return;
			sStat.st_dev = nDevice;
			sStat.st_ino = nInode;
		}
		if( !bBrokenLink )
			pcMonitor = new os::NodeMonitor( &cNode, NWATCH_STAT | NWATCH_ATTR, this, this );

		//std::cout<<"Add "<<sStat.st_dev<<" "<<sStat.st_ino<<std::endl;
		
		
		Message cMsg( FileView::M_ADD_ENTRY );
		cMsg.AddString( "name", cName );
		cMsg.AddString( "path", cName );
		cMsg.AddData( "stat", T_ANY_TYPE, &sStat, sizeof( sStat ) );
		cMsg.AddBool( "broken_link", bBrokenLink );
		cMsg.AddPointer( "node_monitor", pcMonitor );
		
		
		m_nPendingReplies++;
		m_cTarget.SendMessage( &cMsg );
		
		// Createt humbnail
		ThumbsDB* db = ThumbsDB::Get();
		Thumbnail* thumb = db->CreateThumbnail( m_cPath + "/" + cName );

		if( thumb != NULL && ( thumb->GetStatus() == Thumbnail::NO_IMAGE )) // || thumb->GetStatus() == Thumbnail::PROCESSING ) )
		{
			if( m_ThumbCreator )
				m_ThumbCreator->AddThumbnail( thumb );
		}

//		printf("Send add %s %i %i!\n", cName.c_str(), (int)sStat.st_ino, m_nPendingReplies);
	}
	catch( ... /*std::exception& */  )
	{
	}
	m_bLayoutNecessary = true;
}

void DirKeeper::SendRemoveMsg( const String& cName, dev_t nDevice, ino_t nInode )
{
	struct stat sStat;
	sStat.st_dev = nDevice;
	sStat.st_ino = nInode;
	
	
	Message cMsg( FileView::M_REMOVE_ENTRY );
	cMsg.AddInt32( "device", nDevice );
	cMsg.AddInt64( "node", nInode );
	
	
	m_nPendingReplies++;
	m_cTarget.SendMessage( &cMsg );
	m_bLayoutNecessary = true;
	//printf("Send Remove %i %i!\n", (int)nInode, m_nPendingReplies);
}


void DirKeeper::SendDriveAddMsg( const String& cName, const String& cPath )
{
	try
	{
		
		FSNode cNode( *m_pcCurrentDir, cPath, O_RDONLY | O_NONBLOCK );
		struct stat sStat;

		cNode.GetStat( &sStat, false );

		
		Message cMsg( FileView::M_ADD_ENTRY );
		cMsg.AddString( "name", cName );
		cMsg.AddString( "path", cPath );
		cMsg.AddData( "stat", T_ANY_TYPE, &sStat, sizeof( sStat ) );
		cMsg.AddBool( "broken_link", false );
		cMsg.AddPointer( "node_monitor", NULL );

		m_nPendingReplies++;
		m_cTarget.SendMessage( &cMsg );
		
	}
	catch( ... /*std::exception& */  )
	{
	}
	m_bLayoutNecessary = true;
}

void DirKeeper::SendReadyMsg()
{
	Message cMsg( FileView::M_READY );
	
	m_cTarget.SendMessage( &cMsg );
}

void DirKeeper::Stop()
{
	Lock();
	m_eState = S_IDLE;
	m_bLayoutNecessary = false;
	m_bWaitForLayoutReply = false;
	m_cMonitor.Unset();
	
	m_nPendingReplies = 0;
	
	/* Make sure we delete all pending messages */
	SpoolMessages();	// Copy all messages from the message port to the message queue.
	MessageQueue *pcQueue = GetMessageQueue();
	MessageQueue cTmp;
	Message* pcMsg;
	pcQueue->Lock();
	while( ( pcMsg = pcQueue->NextMessage() ) != NULL )
	{
		switch( pcMsg->GetCode() )
		{
			case M_ENTRIES_ADDED:
			case M_ENTRIES_UPDATED:
			case M_ENTRIES_REMOVED:
			case M_NODE_MONITOR:
				break;
			default:
				cTmp.AddMessage( pcMsg );
		}
	}
	
	while( ( pcMsg = cTmp.NextMessage() ) != NULL )
	{
		pcQueue->AddMessage( pcMsg );
	}
	
	pcQueue->Unlock();
	Unlock();
}

bool DirKeeper::Idle()
{
	switch ( m_eState )
	{
	case S_READDIR:
		{
			if( m_pcCurrentDir == NULL )
			{
				m_eState = S_IDLE;
				SendReadyMsg();
				return ( true );
			}
			os::String cName;
			
			if( m_cPath == "/" )
			{
				/* Show drives */
				os::Volumes cVolumes;
				os::String cMountPoint;
				//char zTemp[PATH_MAX];
				for( uint i = 0; i < cVolumes.GetMountPointCount(); i++ )
				{
					if( cVolumes.GetMountPoint( i, &cMountPoint ) == 0 ) {
						fs_info sInfo;
						if( cVolumes.GetFSInfo( i, &sInfo ) == 0 && ( sInfo.fi_flags & FS_IS_BLOCKBASED ) )
						{
						
							os::String zVolumeName = sInfo.fi_volume_name;
							if( ( zVolumeName == "" ) || ( zVolumeName == "/" ) )
								zVolumeName = "Unknown";
								
							/* Remove '/' */
							os::String cVolumePath = cMountPoint.substr( 1, cMountPoint.Length() - 1 );
							SendDriveAddMsg( zVolumeName, cVolumePath );
						}
					}
				}
				m_eState = S_IDLE;
				SendReadyMsg();
				return ( true );
			} 
			else
			{
				/* Show files */
				if( m_bLayoutNecessary )
				{
					if( m_nPendingReplies > 0 )
					{
						if( get_system_time() - m_nLastLayout > LAYOUT_TIME && !m_bWaitForLayoutReply )
						{
							/* Update regulary */
							Message cMsg( FileView::M_LAYOUT );
							m_cTarget.SendMessage( &cMsg );
							m_bWaitForLayoutReply = true;
							m_bLayoutNecessary = false;
							m_nLastLayout = get_system_time();
						}
					}
				}
			
				if( m_nPendingReplies > 5 )
					return( false );
					
				if( m_pcCurrentDir->GetNextEntry( &cName ) == 1 )
				{
					if( !( cName == "." ) && !( cName == ".." ) )
					{
						SendAddMsg( cName, -1, -1 );
					}
					return ( true );
				}
				else
				{
					m_eState = S_IDLE;
					SendReadyMsg();
					return ( true );
				}
			}
			break;
		}
	case S_IDLE:
		if( m_bLayoutNecessary )
		{
			if( m_nPendingReplies > 0 )
				return( false );

			/* Final layout */			
			Message cMsg( FileView::M_LAYOUT );
			m_cTarget.SendMessage( &cMsg );
			m_nLastLayout = get_system_time();
			m_bWaitForLayoutReply = true;
			m_bLayoutNecessary = false;
		}
		return ( false );
	}
	
	
	return ( false );
}

