// EFileBrowser (C)opyright 2007 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <stack>
#include <util/message.h>
#include <gui/window.h>	// Needed due to volumes.h
#include <storage/fsnode.h>

#include <storage/volumes.h>
#include "tab_browser.h"

using namespace os;

///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E
//
///////////////////////////////////////////////////////////////////////////////

class BrowserTab :: _Private
{
public:
	_Private() {};

	// Stack to keep track of previous path (for the Back button)
	std::stack< String > m_PreviousPaths;
	
};

///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S
//
///////////////////////////////////////////////////////////////////////////////

BrowserTab :: BrowserTab( String path, Message* dir_chg, Message* ready, Message* sel_chg, Message* tab_open ) : FileView( path )
{
	// Create the private class
	m = new _Private();

	dir_chg->AddPointer( "browsertab", (void**) this );
	SetDirChangeMsg( dir_chg );
	ready->AddPointer( "browsertab", (void**) this );
	SetReReadReadyMsg( ready );
	sel_chg->AddPointer( "browsertab", (void**) this );
	SetSelChangeMsg( sel_chg );
	tab_open->AddPointer( "browsertab", (void**) this );
	SetOpenTabMsg( tab_open );

	SetDirectoryLocked( true );
	SetAutoSort( true );

	ReRead();
}

BrowserTab :: ~BrowserTab()
{
	delete m;
}

String BrowserTab :: GetName()
{
	Path p = GetPath();

	if( GetPath() == "/" )
		return String( "Mounted Drives" );

	return p.GetLeaf();
}

void BrowserTab :: SetPath( String new_path, bool save_path )
{
	// Remeber the previous path
	if( save_path )	
		m->m_PreviousPaths.push( GetPath() );

	FileView::SetPath( new_path );
	ReRead();
}

void BrowserTab :: Up()
{
	String current_path = GetPath();

	// If we are already at the root then stop
	if( current_path == "/" )
		return;

	int i = std::string( current_path ).rfind( "/" );

	if( (uint) i == std::string::npos )
		return;

	if( i == 0 )
		current_path.resize( 1 );
	else
		current_path.resize( i );

	SetPath( current_path );	
}

void BrowserTab :: Back()
{
	if( m->m_PreviousPaths.empty() )
		return;

	String path = m->m_PreviousPaths.top();
	m->m_PreviousPaths.pop();

	SetPath( path, false );
}

void BrowserTab :: GetStat( struct DirectoryStat* stat )
{
	stat->no_folders = 0;
	stat->no_files = 0;

	stat->no_sel_folders = 0;
	stat->no_sel_files = 0;

	stat->total_sel_size = 0;
	stat->total_size = 0;

	if( GetPath() != "/" )
	{
		// Get number of selected icons and calculate some statistics for the directory
		for( uint i = 0 ; i < GetIconCount() ; i ++ )
		{	
		 	DirectoryIconData* data = static_cast< DirectoryIconData* >( GetIconData( i ) );
			String path( GetPath() + "/" + data->m_zPath );
			FSNode fs( path );

			if( fs.IsFile() )
			{
				stat->total_size += (uint) fs.GetSize();
				stat->no_files++;
			}

			if( fs.IsDir() )
				stat->no_folders++;

			if( GetIconSelected( i ) )
			{				
				if( fs.IsDir() )
					stat->no_sel_folders++;
			
				if( fs.IsFile() )
				{
					stat->total_sel_size += fs.GetSize();
					stat->no_sel_files++;
				}
			}
		}
	}
	else
	{	
		// Get number of selected icons and calculate some statistics for the drives
		for( uint i = 0 ; i < GetIconCount() ; i ++ )
		{	
		 	DirectoryIconData* data = static_cast< DirectoryIconData* >( GetIconData( i ) );
			String path( GetPath() + data->m_zPath );			
			Volumes cVolumes;
			String cMountPoint;

			for( uint i = 0; i < cVolumes.GetMountPointCount(); i++ )
			{
				if( cVolumes.GetMountPoint( i, &cMountPoint ) == 0 )
				{
					if( path == cMountPoint )
					{
						fs_info sInfo;
				
						if( cVolumes.GetFSInfo( i, &sInfo ) == 0 && ( sInfo.fi_flags & FS_IS_BLOCKBASED ) )
						{
							stat->no_files++;

							if( GetIconSelected( i ) )
							{
								stat->no_sel_files++;
								stat->total_sel_size = ((int)sInfo.fi_free_blocks) * 100 / ((int)sInfo.fi_total_blocks);
							}
						}
					}
				}
			}
		}
	}	
}
