 #include <gui/font.h>

#include "etreeview.h"

#include <typeinfo>

using namespace os;

/** \internal */
class ETreeViewNode::Private
{
        public:
        bool            m_bExpanded;            // Node is expanded
        uint            m_nIndent;                      // Indentation depth
        ETreeView*       m_pcOwner;
        uint32          m_nLPos;                        // Trunk lines
        bool            m_bLastSibling;         // true if this is the last sibling
                                                                        // (means that the last trunk segment is of half length)
};


ETreeViewNode::ETreeViewNode()
{
        m = new Private;
        m->m_nIndent = 0;
        m->m_bExpanded = true;
        m->m_pcOwner = NULL;
        m->m_nLPos = 0;
        m->m_bLastSibling = false;
}

ETreeViewNode::~ETreeViewNode()
{
	delete m;
}

bool ETreeViewNode::HitTest( View* pcView, const Rect& cFrame, int nColumn, Point cPos )
{
//printf( "Select: %d\n", m->m_pcOwner->GetLastSelected() );

        if( nColumn == 0 ) {
                Rect r2 = _ExpanderCrossPos( cFrame );
                //r2 += Point( cFrame.left, cFrame.top );
                if( r2.DoIntersect( cPos ) && m->m_pcOwner ) {
                        if( IsExpanded() ) {
                                m->m_pcOwner->Collapse( this );
                        } else {                        
                                m->m_pcOwner->Expand( this );
                        }
printf( "Select: %d\n", m->m_pcOwner->GetLastSelected() );
                        return false;
                }
        }
        return true;
}

/** Set indentation depth
 *      \par    Description:
 *                      Set indentation depth for a node. The hierarchical relations
 *                      between nodes is decided by their indentation depths. A node with
 *                      a higher indentation depth than the node before it, is considered
 *                      a sub node.
 *  \param      nIndent Indentation depth.
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
void ETreeViewNode::SetIndent( uint nIndent )
{
        m->m_nIndent = nIndent;
}

/** Get indentation depth
 *      \par    Description:
 * \sa SetIndent()
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
uint ETreeViewNode::GetIndent() const
{
        return m->m_nIndent;
}

/** Set expanded state
 *      \par    Description:
 *                      Set expanded state of a node.
 * \sa ETreeView::Expand(), GetExpanded()
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/ 
void ETreeViewNode::SetExpanded( bool bExpanded )
{
	ETreeView* parent = GetOwner();

	// Make sure there is a TreeView
	if( parent == NULL )
		return;

	m->m_bExpanded = bExpanded;

	std::vector<ETreeViewNode*> children;
	std::vector<ETreeViewNode*>::iterator it;

	GetChildren( children );

	for( it = children.begin() ; it != children.end() ; it++ )
		(*it)->SetIsVisible( bExpanded );
}

void ETreeViewNode::SetIsVisible( bool bVisible )
{
	ETreeView* parent = GetOwner();

	// Make sure there is a TreeView
	if( parent == NULL )
		return;

	ListViewRow::SetIsVisible( bVisible );

	if( !IsExpanded() )
		return;

	std::vector<ETreeViewNode*> children;
	std::vector<ETreeViewNode*>::iterator it;

	GetChildren( children );

	for( it = children.begin() ; it != children.end() ; it++ )
		(*it)->SetIsVisible( bVisible );	
}

/** Get expanded state
 *      \par    Description:
 *                      Returns true if the node is in expanded state (ie. sub nodes are
 *                      displayed).
 * \sa SetIndent()
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
bool ETreeViewNode::IsExpanded() const
{
        return m->m_bExpanded;
}

/** Set owner
 *      \par    Description:
 *                      This method is used by ETreeView to tell nodes which ETreeView they
 *                      belong to. It is called automatically by ETreeView::InsertNode(),
 *                      so you should never need to use this method directly.
 *                      
 * \sa ETreeView::InsertNode(), GetOwner()
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/ 
void ETreeViewNode::_SetOwner( ETreeView* pcOwner )
{
        m->m_pcOwner = pcOwner;
}

/** Get owner
 *      \par    Description:
 *                      Returns a pointer to the ETreeView this node belongs to. Returns
 *                      NULL if the node is not attached to a ETreeView.
 *                      
 * \sa _SetOwner()
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/ 
ETreeView* ETreeViewNode::GetOwner() const
{
        return m->m_pcOwner;
}

/** Set line positions
 *      \par    Description:
 *                      A 'one' in the line positions integer means that a tree trunk
 *                      segment should be drawn at the corresponding indentation depth
 *                      on the current row.
 * \sa _GetLinePositions()
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/ 
void ETreeViewNode::_SetLinePositions( uint32 nLPos )
{
        m->m_nLPos = nLPos;
}

/** Get line positions
 *      \par    Description:
 * \sa _SetLinePositions()
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/ 
uint32 ETreeViewNode::_GetLinePositions() const
{
        return m->m_nLPos;
}

/** Check if node has more siblings
 *      \par    Description:
 *                      Check if there is a sibling following the current node. Used when
 *                      rendering the "tree trunk", to know if it should continue to the
 *                      next node or if it should be finished at this node.
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/ 
bool ETreeViewNode::IsLastSibling() const
{
        return m->m_bLastSibling;
}

/** Set LastSibling
 *      \par    Description:
 *                      Used by ETreeView to keep the LastSibling attribute up to date.
 *                      Should not be called elsewhere.
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/ 
void ETreeViewNode::_SetLastSibling( bool bIsLast )
{
        m->m_bLastSibling = bIsLast;
}

/** Insert child
 *      \par    Description:
 *                      Insert a node as a child to this node. Note that the parent
 *                      node must be attached to a TreeView
 * \param      cpNode Node to insert as a child
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/ 
void ETreeViewNode::InsertChild( ETreeViewNode* pcNode, bool bUpdate )
{
	ETreeView* parent = GetOwner();

	// Make sure there is a TreeView
	if( parent == NULL )
		return;

//	parent->ClearSelection();

	// Insert child node
	uint count = parent->GetRowCount();

	for( uint i = 0; i < count; i++)
	{
		ETreeViewNode* lvr = (ETreeViewNode*) parent->GetRow( i );

		if( lvr == this ) 
		{
			parent->InsertNode( i + 1, pcNode, false );  

			break;
        }                       
	}			

	pcNode->SetIndent( GetIndent() + 1 );
	pcNode->SetExpanded( false );
	pcNode->SetIsVisible( IsExpanded() && IsVisible() );

	if( bUpdate )
		parent->RefreshLayout();

}

/** Get subnodes
 *      \par    Description:
 *                      Fills a std::vector with the sub nodes (children) of this node
 *      \param  cvecChildren std::vector to be filled with pointers to nodes.
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
void ETreeViewNode::GetChildren( std::vector<ETreeViewNode*>& cvecChildren )
{
	ETreeView* parent = GetOwner();

	// Make sure there is a TreeView
	if( parent == NULL )
		return;

	uint count = parent->GetRowCount();
	uint j = GetIndent();
	bool begin = false;

	for( uint i = 0 ; i < count; i++ )
	{
		ETreeViewNode* node = (ETreeViewNode*)parent->GetRow( i );

		if( begin ) {
			if( node->GetIndent() == j + 1 ) 
			{
				cvecChildren.push_back( node );
			} else if( node->GetIndent() <= j ) {
				return;
			}
		}

		if( node == this )
			begin = true;
	}
}

/** Removed all children
 *      \par    Description:
 *                     Removes all children to the node
 *  \param      bUpdate Set to true to refresh ETreeView.
 * \author Jonas Jarvoll (jonas-jarvoll@syllable-norden.info)
 *****************************************************************************/
void ETreeViewNode::RemoveChildren( bool bUpdate )
{
	std::vector<ETreeViewNode*> children;

	GetChildren( children );

	for( std::vector<ETreeViewNode*>::iterator it = children.begin() ; it != children.end() ; it++ )
	{
		(*it)->RemoveChildren( false );
	
		if( (*it)->IsSelected() )
			GetOwner()->ClearSelection();
 
		GetOwner()->RemoveNode( *it, false );
	}

	if( bUpdate )
		GetOwner()->RefreshLayout();
}

/** Get expander position
 *      \par    Description:
 *                      Returns a rect with dimensions and position for where the expander
 *                      image should be drawn.
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/ 
Rect ETreeViewNode::_ExpanderCrossPos( const Rect &cFrame ) const
{
        if( m->m_pcOwner ) {
                Rect cImgFrame( m->m_pcOwner->GetExpanderImageBounds() );
        
                cImgFrame.right += 2;
                cImgFrame.bottom += 2;
        
                cImgFrame += Point( GetIndent() * GetOwner()->GetIndentWidth(),
                                                (cFrame.Height()/2 - cImgFrame.Height()/2) );
        
                return cImgFrame;
        } else {
                return Rect( 0, 0, 10, 10 );
        }
}

/** Draw expander cross
 *      \par    Description:
 *                      The default expander cross drawing routine.
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/ 
void ETreeViewNode::_DrawExpanderCross( View *pcView, const Rect& cRect )
{
        if( !m->m_pcOwner ) return;

        pcView->SetFgColor( Color32_s(0, 0, 0) );

        pcView->SetDrawingMode( DM_COPY );

        if( !IsExpanded() ) {   
        if( !m->m_pcOwner->GetCollapsedImage() ) {
                        pcView->DrawLine( Point( cRect.left + cRect.Width()/2, cRect.top+2 ),
                                        Point( cRect.left + cRect.Width()/2, cRect.bottom-2 ) );
                        pcView->DrawLine( Point( cRect.left + 2, cRect.top + cRect.Height()/2 ),
                                        Point( cRect.right - 2, cRect.top + cRect.Height()/2 ) );
                } else {
            BitmapImage* pcImage = (BitmapImage*)m->m_pcOwner->GetCollapsedImage();
            Point cImgPos( cRect.Width()/2 - pcImage->GetSize().x/2 + cRect.left,
                        cRect.Height()/2 - pcImage->GetSize().y/2 + cRect.top );
            pcImage->Draw( cImgPos, pcView );
                }
        } else {
        if( !m->m_pcOwner->GetExpandedImage() ) {
                        pcView->DrawLine( Point( cRect.left + 2, cRect.top + cRect.Height()/2 ),
                                        Point( cRect.right - 2, cRect.top + cRect.Height()/2 ) );
                } else {
            BitmapImage* pcImage = (BitmapImage*)m->m_pcOwner->GetExpandedImage();
            Point cImgPos( cRect.Width()/2 - pcImage->GetSize().x/2 + cRect.left,
                        cRect.Height()/2 - pcImage->GetSize().y/2 + cRect.top );
            pcImage->Draw( cImgPos, pcView );
                }
        }

        if( m->m_pcOwner->GetDrawExpanderBox() ) {
                pcView->SetDrawingMode( DM_INVERT );
                pcView->DrawFrame( cRect, FRAME_TRANSPARENT | FRAME_THIN );
        }

        pcView->SetDrawingMode( DM_COPY );
}

/** \internal */
class ETreeViewStringNode::Private
{
        public:
    std::vector< std::pair<String,float> > m_cStrings;
    Image*      m_pcIcon;
        uint32  m_nTextFlags;
    
    Private() {
        m_pcIcon = NULL;
        m_nTextFlags = DTF_IGNORE_FMT;
    }
    
    ~Private() {
        if( m_pcIcon ) delete m_pcIcon;
    }
};

ETreeViewStringNode::ETreeViewStringNode()
{
        m = new Private;
}
ETreeViewStringNode::~ETreeViewStringNode()
{
        delete m;
}

void ETreeViewStringNode::AttachToView( View* pcView, int nColumn )
{
        Point   cSize = pcView->GetTextExtent( m->m_cStrings[nColumn].first );
    m->m_cStrings[nColumn].second = cSize.x + 5.0f;

        if( nColumn == 0 && GetIcon() ) {
                m->m_cStrings[0].second += GetIcon()->GetSize().x + 5;
        }
}

void ETreeViewStringNode::SetRect( const Rect& cRect, int nColumn )
{
}

void ETreeViewStringNode::AppendString( const String& cString )
{
    m->m_cStrings.push_back( std::make_pair( cString, 0.0f ) );
}

void ETreeViewStringNode::SetString( int nIndex, const String& cString )
{
    m->m_cStrings[nIndex].first = cString;
}

const String& ETreeViewStringNode::GetString( int nIndex ) const
{
    return( m->m_cStrings[nIndex].first );
}

float ETreeViewStringNode::GetWidth( View* pcView, int nIndex )
{
    return( m->m_cStrings[nIndex].second );
}

float ETreeViewStringNode::GetHeight( View* pcView )
{
        Point   cSize( 0, 0 );
        uint i;
        
        for( i = 0; i < m->m_cStrings.size(); i++ ) {
                Point s = pcView->GetTextExtent( m->m_cStrings[i].first, m->m_nTextFlags );
                if( s.y > cSize.y ) cSize.y = s.y;
        }
     
    if( GetIcon() ) {
        if( cSize.y < GetIcon()->GetSize().y )
                cSize.y = GetIcon()->GetSize().y;
    }

    return cSize.y;
}

uint32 ETreeViewStringNode::GetTextFlags() const
{
        return m->m_nTextFlags;
}

void ETreeViewStringNode::SetTextFlags( uint32 nTextFlags ) const
{
        m->m_nTextFlags = nTextFlags;
}

void ETreeViewStringNode::Paint( const Rect& cFrame, View* pcView, uint nColumn,
                               bool bSelected, bool bHighlighted, bool bHasFocus )
{
        ETreeView *pcOwner = GetOwner();
        uint nIndentWidth = 10;
        bool bExp = false;

        if( pcOwner ) {
                nIndentWidth = pcOwner->GetIndentWidth();
                bExp = pcOwner->HasChildren( this );    // FIXME: slow!
        }

        uint nIndent = GetIndent() * nIndentWidth;

        Rect cItemRect( cFrame );

        Point   cTextSize;
        if( nColumn <= m->m_cStrings.size() ) {
                cTextSize = pcView->GetTextExtent( m->m_cStrings[nColumn].first, m->m_nTextFlags );
        }

        if( nIndent && !nColumn ) {     
                cItemRect.left += nIndent + 15;
                cItemRect.right += nIndent + 15;
        } else {
                nIndent = 0;
        }

        pcView->SetDrawingMode( DM_COPY );

    pcView->SetFgColor( 255, 255, 255 );
    pcView->FillRect( cFrame );
  //  pcView->DrawFrame( cFrame, FRAME_TRANSPARENT | FRAME_THIN );

        float vTextPos = cItemRect.left + 3.0f;

        if( GetIcon() && nColumn == 0 ) {
                vTextPos += GetIcon()->GetSize().x + 5;
        }
        
        if( bSelected || bHighlighted )
        {
                Rect cSelectFrame = cFrame;
                cSelectFrame.left = cItemRect.left;
                if( nColumn == 0 ) {
                        cSelectFrame.left += 2;
                        cSelectFrame.top += 2;
                        cSelectFrame.bottom -= 2;
                }
                if( bSelected )
                        pcView->SetFgColor( 186, 199, 227 );
                else
                        pcView->SetFgColor( 0, 50, 200 );
                pcView->FillRect( cSelectFrame );
        
                /* Round edges */
                if( nColumn == 0 )
                {
                        pcView->DrawLine( os::Point( cSelectFrame.left + 2, cSelectFrame.top - 2 ), 
                                                                os::Point( cSelectFrame.right, cSelectFrame.top - 2 ) );
                        pcView->DrawLine( os::Point( cSelectFrame.left, cSelectFrame.top - 1 ), 
                                                                os::Point( cSelectFrame.right, cSelectFrame.top - 1 ) );
                        
                        pcView->DrawLine( os::Point( cSelectFrame.left - 2, cSelectFrame.top + 2 ), 
                                                                os::Point( cSelectFrame.left - 2, cSelectFrame.bottom - 2 ) );
                        pcView->DrawLine( os::Point( cSelectFrame.left - 1, cSelectFrame.top ), 
                                                                os::Point( cSelectFrame.left - 1, cSelectFrame.bottom ) );
                                                                
                        pcView->DrawLine( os::Point( cSelectFrame.left + 2, cSelectFrame.bottom + 2 ), 
                                                                os::Point( cSelectFrame.right, cSelectFrame.bottom + 2 ) );
                        pcView->DrawLine( os::Point( cSelectFrame.left, cSelectFrame.bottom + 1 ), 
                                                                os::Point( cSelectFrame.right, cSelectFrame.bottom + 1 ) );
                } 
        }
 
    if ( bHighlighted ) {
pcView->SetFgColor( 255, 255, 255 );
                pcView->SetBgColor( 0, 50, 200 );
    } else if ( bSelected ) {
                pcView->SetFgColor( 0, 0, 0 );
                pcView->SetBgColor( 186, 199, 227 );
    } else {
                pcView->SetBgColor( 255, 255, 255 );
                pcView->SetFgColor( 0, 0, 0 );
    }
/*
    if ( bSelected && nColumn == 0 ) {
                Rect cRect = cItemRect;
                cRect.right  = vTextPos + cTextSize.x + 1;
                pcView->FillRect( cRect, Color32_s( 0, 0, 0, 0 ) );
    }*/

        if( nColumn <= m->m_cStrings.size() ) {
                Rect cTextRect( vTextPos, cItemRect.top, cItemRect.right, cItemRect.bottom );
        pcView->DrawText( cTextRect, m->m_cStrings[nColumn].first, m->m_nTextFlags );
    }

        if( GetIcon() && nColumn == 0 ) {
                pcView->SetDrawingMode( DM_BLEND );
                GetIcon()->Draw( Point( cItemRect.left + 2, cItemRect.top + (cItemRect.Height() / 2 - GetIcon()->GetSize().y / 2) ), pcView );
        }

        /* Draw Trunk (a.k.a connecting lines) */
        if( nColumn == 0 && pcOwner && pcOwner->GetDrawTrunk() ) {
                pcView->SetFgColor( 0, 0, 0 );

                uint32 bits = _GetLinePositions();
                int i = 0;
                for( ; bits ; bits >>= 1, i++ ) {
                        if( bits & 1 ) {
                                if( !( bits >> 1 ) && IsLastSibling() ) {
                                        pcView->DrawLine( Point( cFrame.left + ( i + .5 ) * (float)nIndentWidth, cFrame.top ),
                                                        Point( cFrame.left + ( i + .5 ) * (float)nIndentWidth, cFrame.top + cFrame.Height()/2 ) );
                                } else {
                                        pcView->DrawLine( Point( cFrame.left + ( i + .5 ) * (float)nIndentWidth, cFrame.top ),
                                                        Point( cFrame.left + ( i + .5 ) * (float)nIndentWidth, cFrame.bottom ) );
                                }
                        }
                }
                
                if( i ) {
                        if( bExp ) {
                                pcView->DrawLine( Point( cFrame.left + ( i - .5 ) * (float)nIndentWidth, cFrame.top + cFrame.Height()/2 ),
                                                Point( _ExpanderCrossPos( cFrame ).left-1, cFrame.top + cFrame.Height()/2 ) );
                        } else {
                                pcView->DrawLine( Point( cFrame.left + ( i - .5 ) * (float)nIndentWidth, cFrame.top + cFrame.Height()/2 ),
                                                Point( cItemRect.left, cFrame.top + cFrame.Height()/2 ) );
                        }
                }
        }

        if( nIndent && bExp ) {    
                Rect cExpCr( _ExpanderCrossPos( cFrame ) );

            _DrawExpanderCross( pcView, cExpCr + Point( cFrame.left, cFrame.top ) );

                if( IsExpanded() && pcOwner && pcOwner->GetDrawTrunk() ) {
                pcView->DrawLine( Point( nIndent + cFrame.left + nIndentWidth/2, cExpCr.bottom + cFrame.top ),
                                        Point( nIndent + cFrame.left + nIndentWidth/2, cFrame.bottom ) );
                }
        }
        
//      pcView->Flush();
//      pcView->Sync();
}

bool ETreeViewStringNode::IsLessThan( const ListViewRow* pcOther, uint nColumn ) const
{
        if( typeid( *pcOther ) == typeid( *this ) ) {
                return( GetString( nColumn ) < ((ETreeViewStringNode*)pcOther)->GetString( nColumn ) );
        } else {
                return true;
        }
}
        
void ETreeViewStringNode::SetIcon( Image* pcIcon )
{
        if( m->m_pcIcon ) delete m->m_pcIcon;
        m->m_pcIcon = pcIcon;
}

Image* ETreeViewStringNode::GetIcon() const
{
        return m->m_pcIcon;
}






/** \internal */
class ETreeView::Private
{
        public:
    Image*              m_pcExpandedImage ;             /** Expander Image used for expanded items */
    Image*              m_pcCollapsedImage;             /** Expander Image used for collapsed items */
    bool                m_bDrawBox;                             /** Draw box around expander image */
        bool        m_bDrawTrunk;                       /** Draw lines to each node */
        bool        m_bTrunkValid;                      /** Lines to each node are already set up correctly */
    bool                m_bBoldFont;                    /** Use bold font for expanded nodes */
    Message*    m_pcExpandMsg;                  /** Message sent when expanding/collapsing node */
    uint                m_nIndentWidth;                 /** Width of each indent step in pixels */
    Rect                m_cExpImgBounds;                /** Cache for expander image bounds */

        void CalcExpImgBounds() {
                m_cExpImgBounds.Resize( 0, 0, 0, 0 );

                if( m_pcExpandedImage ) {
                         m_cExpImgBounds.right = m_pcExpandedImage->GetSize().x;
                         m_cExpImgBounds.bottom = m_pcExpandedImage->GetSize().y;
                }

                if( m_pcCollapsedImage ) {
                        if( m_pcCollapsedImage->GetSize().x > m_cExpImgBounds.right )
                                 m_cExpImgBounds.right = m_pcCollapsedImage->GetSize().x;
                        if( m_pcCollapsedImage->GetSize().y > m_cExpImgBounds.bottom )
                                 m_cExpImgBounds.bottom = m_pcCollapsedImage->GetSize().y;
                }

                if( !m_cExpImgBounds.right )
                        m_cExpImgBounds.Resize( 0, 0, 10, 10 );
        }

    Private()
    : m_cExpImgBounds( 0, 0, 10, 10 )
    {
            m_pcExpandedImage = NULL;
            m_pcCollapsedImage = NULL;
                m_pcExpandMsg = NULL;
            m_bDrawBox = true;
            m_bBoldFont = true;
            m_nIndentWidth = 10;
            m_bDrawTrunk = true;
            m_bTrunkValid = false;
    }
    
    ~Private() {
        if( m_pcExpandedImage ) delete m_pcExpandedImage;
        if( m_pcCollapsedImage ) delete m_pcCollapsedImage;
                if( m_pcExpandMsg ) delete m_pcExpandMsg;
    }
};

ETreeView::ETreeView( const Rect& cFrame, const String& cName, uint32 nModeFlags,
                    uint32 nResizeMask, uint32 nFlags )
        : ListView( cFrame, cName, nModeFlags, nResizeMask, nFlags ) {
    m = new Private;
}
    
ETreeView::~ETreeView()
{
        delete m;
}

/** Insert a node at the end
 *      \par    Description:
 *                      Insert a node at the end of the tree.
 *  \param      nPos Zero-based index to insert position.
 *  \param      pcNode Pointer to node to insert.
 *  \param      bUpdate Set to true to refresh ETreeView.
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
void ETreeView::InsertNode( ETreeViewNode* pcNode, bool bUpdate )
{
	pcNode->_SetOwner( this );
    m->m_bTrunkValid = false;
    ListView::InsertRow( 1 << 15, pcNode, bUpdate );
}

/** Insert a node at a certain position
 *      \par    Description:
 *                      Insert a node at position nPos.
 *  \param      nPos Zero-based index to insert position.
 *  \param      pcNode Pointer to node to insert.
 *  \param      bUpdate Set to true to refresh ETreeView.
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
void ETreeView::InsertNode( int nPos, ETreeViewNode* pcNode, bool bUpdate )
{
        pcNode->_SetOwner( this );
    m->m_bTrunkValid = false;
        ListView::InsertRow( nPos, pcNode, bUpdate );
}

/** Expand a node
 *      \par    Description:
 *                      Expands a node so that subnodes are made visible.
 *  \param      pcNode Pointer to node to expand.
 *
 *  \sa Collapse
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
void ETreeView::Expand( ETreeViewNode* pcNode )
{
printf( "Expand\n" );
	pcNode->SetExpanded( true );
     
	RefreshLayout();
   
	if( m->m_pcExpandMsg ) 
	{
		Message* pcMsg = new Message( *m->m_pcExpandMsg );
		pcMsg->AddPointer( "node", pcNode );
		pcMsg->AddBool( "expanded", true );
		Invoke( pcMsg );
	}
}

/** Collapse a node
 *      \par    Description:
 *                      Contracts a node so that subnodes are made invisible.
 *  \param      pcNode Pointer to node to collapse.
 *
 *  \sa Expand
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
void ETreeView::Collapse( ETreeViewNode* pcNode )
{
printf( "collapse\n" );
	pcNode->SetExpanded( false );

	RefreshLayout();
        
	if( m->m_pcExpandMsg ) 
	{
		Message* pcMsg = new Message( *m->m_pcExpandMsg );
		pcMsg->AddPointer( "node", pcNode );
		pcMsg->AddBool( "expanded", false );
		Invoke( pcMsg );
	}
}

/** Find out if a node has sub nodes
 *      \par    Description:
 *                      Returns true if a node has at least one child node.
 *  \param      pcNode Pointer to node.
 *
 *  \sa GetChild
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
bool ETreeView::HasChildren( ETreeViewNode* pcNode )
{
        uint                    count = GetRowCount();
        uint                    i;
        bool                    begin = false;
        
        for(i = 0; i < count; i++)
        {
                ETreeViewNode*           lvr = (ETreeViewNode*)GetRow( i );
                
                if( begin ) {
                        if( lvr->GetIndent() > pcNode->GetIndent() ) {
                                return true;
                        } else {
                                return false;
                        }
                }
                
                if( lvr == pcNode ) begin = true;               
        }

        return false;
}

void ETreeView::_ConnectLines()  // Scan through the tree and connect the treetrunk to each node
{
        uint                    count = GetRowCount();
        uint                    i;
        uint32                  bits = 0;
        uint                    depth = 33;

        for( i = count; i; i-- )
        {
                ETreeViewNode*           lvr = (ETreeViewNode*)GetRow( i - 1 );

                depth = lvr->GetIndent();

                if( ( bits >> ( depth - 1 ) ) & 1 ) {
                        lvr->_SetLastSibling( false );
                } else {
                        lvr->_SetLastSibling( true );
                }

                bits &= ~( ~0 << depth );
                if( depth > 1 ) bits |= 1 << ( depth - 1 );

                lvr->_SetLinePositions( bits );
        }
}

/** Set expander image when expanded
 *      \par    Description:
 *                      Used to set the expander image (in expanded state). Normally you
 *                      should not use this function, but instead use the system default
 *                      image.
 *  \param      pcImage Pointer to image to use for expander image.
 *
 *  \sa GetExpandedImage(), SetCollapsedImage()
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
void ETreeView::SetExpandedImage(Image* pcImage)
{
        if( m->m_pcExpandedImage ) delete m->m_pcExpandedImage;
    m->m_pcExpandedImage = pcImage;
        m->CalcExpImgBounds();
}

/** Set expander image when collapsed
 *      \par    Description:
 *                      Used to set the expander image (in collapsed state). Normally you
 *                      should not use this function, but instead use the system default
 *                      image.
 *  \param      pcImage Pointer to image to use for expander image.
 *
 *  \sa GetCollapsedImage(), SetExpandedImage()
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
void ETreeView::SetCollapsedImage(Image* pcImage)
{
        if( m->m_pcCollapsedImage ) delete m->m_pcCollapsedImage;
    m->m_pcCollapsedImage = pcImage;
        m->CalcExpImgBounds();
}

/** Get expander image when expanded
 *      \par    Description:
 *                      Returns the image used for expander icons in expanded state.
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
Image* ETreeView::GetExpandedImage() const
{
    return m->m_pcExpandedImage;
}

/** Get expander image when expanded
 *      \par    Description:
 *                      Returns the image used for expander icons in collapsed state.
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
Image* ETreeView::GetCollapsedImage() const
{
    return m->m_pcCollapsedImage;
}

/** Get indentation width
 *      \par    Description:
 *                      Returns the size of indentations. Default is 10 pixels.
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
uint ETreeView::GetIndentWidth() const
{
    return m->m_nIndentWidth;
}

/** Set indentation width
 *      \par    Description:
 *                      Set the size of indentations. Default is 10 pixels.
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
void ETreeView::SetIndentWidth( uint nIndentWidth )
{
    m->m_nIndentWidth = nIndentWidth;
}

/** Draw box around expander image
 *      \par    Description:
 *                      Returns true if a box is drawn around the expander image.
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
bool ETreeView::GetDrawExpanderBox() const
{
    return m->m_bDrawBox;
}

/** Draw box around expander image
 *      \par    Description:
 *                      Set to true to have a box drawn around the expander image.
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
void ETreeView::SetDrawExpanderBox(const bool bDraw)
{
    m->m_bDrawBox = bDraw;
}

/** Draw tree trunk
 *      \par    Description:
 *                      Returns true if lines are drawn to each node (aka. tree trunk).
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
bool ETreeView::GetDrawTrunk() const
{
        return m->m_bDrawTrunk;
}

/** Draw tree trunk
 *      \par    Description:
 *                      Set to true to have lines drawn to each node (aka. tree trunk).
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
void ETreeView::SetDrawTrunk( bool bDraw )
{
        m->m_bDrawTrunk = bDraw;
}

/** Get parent node
 *      \par    Description:
 *                      Returns a pointer to the parent node of pcNode, or the parent of
 *                      the currently selected node, if pcNode is NULL.
 *  \param      pcNode Pointer to node.
 *
 *  \sa GetChild(), GetNext(), GetPrev(), ListView::GetRow()
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
ETreeViewNode* ETreeView::GetParent(ETreeViewNode* pcNode)
{
        uint                    count = GetRowCount();
        bool                    begin = false;

        if( !pcNode ) {
                int nLastSelected = GetLastSelected();
                if( nLastSelected != -1 ) {
                        pcNode = (ETreeViewNode*)GetRow( nLastSelected );
                        count = nLastSelected;
                        begin = true;
                } else return NULL;
        }

        uint                    i, j = pcNode->GetIndent();

        for( i = count; i; i-- )
        {
                ETreeViewNode*           node = (ETreeViewNode*)GetRow( i - 1 );

                if( begin && node->GetIndent() < j )
                        return node;

                if( node == pcNode )
                        begin = true;
        }

        return NULL;
}

/** Get child node
 *      \par    Description:
 *                      Returns a pointer to the child node (sub node) of pcNode, or the
 *                      child node of the currently selected node, if pcNode is NULL.
 *  \param      pcNode Pointer to node.
 *
 *  \sa GetParent(), GetNext(), GetPrev(), ListView::GetRow()
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
ETreeViewNode* ETreeView::GetChild(ETreeViewNode* pcNode)
{
        uint                    i = 0;

        if( !pcNode ) {
                int nLastSelected = GetLastSelected();
                if( nLastSelected != -1 ) {
                        pcNode = (ETreeViewNode*)GetRow( nLastSelected );
                        i = nLastSelected;
                } else return NULL;
        }

        ETreeViewNode*           node = (ETreeViewNode*)GetRow( i + 1 );

        if( node ) {
                if( node->GetIndent() > pcNode->GetIndent() )
                        return node;
        }
        
        return NULL;
}

/** Get previous sibling
 *      \par    Description:
 *                      Returns a pointer to the previous sibling of pcNode, or the
 *                      sibling of the currently selected node, if pcNode is NULL.
 *  \param      pcNode Pointer to node.
 *
 *  \sa GetChild(), GetNext(), GetParent(), ListView::GetRow()
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
ETreeViewNode* ETreeView::GetPrev(ETreeViewNode* pcNode)
{
        uint                    count = GetRowCount();
        bool                    begin = false;

        if( !pcNode ) {
                int nLastSelected = GetLastSelected();
                if( nLastSelected != -1 ) {
                        pcNode = (ETreeViewNode*)GetRow( nLastSelected );
                        count = nLastSelected;
                        begin = true;
                } else return NULL;
        }

        uint                    i, j = pcNode->GetIndent();

        for( i = count; i; i-- )
        {
                ETreeViewNode*           node = (ETreeViewNode*)GetRow( i - 1 );

                if( begin ) {
                        if( node->GetIndent() < j )
                                return NULL;
                        if( node->GetIndent() == j )
                                return node;
                }

                if( node == pcNode )
                        begin = true;
        }

        return NULL;
}

/** Get next sibling
 *      \par    Description:
 *                      Returns a pointer to the next sibling of pcNode, or the
 *                      sibling of the currently selected node, if pcNode is NULL.
 *  \param      pcNode Pointer to node.
 *
 *  \sa GetChild(), GetPrev(), GetParent(), ListView::GetRow()
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
ETreeViewNode* ETreeView::GetNext(ETreeViewNode* pcNode)
{
        uint                    i = 0;

        if( !pcNode ) {
                int nLastSelected = GetLastSelected();
                if( nLastSelected != -1 ) {
                        pcNode = (ETreeViewNode*)GetRow( nLastSelected );
                        i = nLastSelected;
                } else return NULL;
        }

        uint                    count = GetRowCount();
        uint                    j = pcNode->GetIndent();
        bool                    begin = false;

        for( ; i < count; i++ )
        {
                ETreeViewNode*           node = (ETreeViewNode*)GetRow( i );

                if( begin ) {
                        if( node->GetIndent() < j )
                                return NULL;
                        if( node->GetIndent() == j )
                                return node;
                }

                if( node == pcNode )
                        begin = true;
        }
        
        return NULL;
}

/** Set expand/collapse message
 *      \par    Description:
 *                      Set a message to be sent when a node is expanded or collapsed.
 *                      When the message is sent, a boolean named "expanded" is added,
 *                      containing the new expanded state of the node (true = expanded,
 *                      false = collapsed). A pointer named "node" points to the node that
 *                      has been expanded/collapsed.
 *  \param      pcMsg Pointer to message to send.
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
void ETreeView::SetExpandedMessage( Message *pcMsg )
{
        if( m->m_pcExpandMsg ) delete m->m_pcExpandMsg;
        m->m_pcExpandMsg = pcMsg;
}

/** Set expand/collapse message
 *      \par    Description:
 *                      Get expanded/collapsed message.
 * \sa SetExpandedMessage()
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
Message* ETreeView::GetExpandedMessage() const
{
        return m->m_pcExpandMsg;
}

/** Get subnodes
 *      \par    Description:
 *                      Fills a std::vector with the sub nodes (children) of a supplied
 *                      node that is attached to this ETreeView, or the currently selected
 *                      node if pcNode is NULL.
 *      \param  cvecChildren std::vector to be filled with pointers to nodes.
 *      \param  pcNode Pointer to node to examine or NULL for selected node.
 * \sa GetSiblings()
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
void ETreeView::GetChildren( std::vector<ETreeViewNode*>& cvecChildren, ETreeViewNode* pcNode )
{
        uint                                            i = 0;

        if( !pcNode ) {
                int nLastSelected = GetLastSelected();
                if( nLastSelected != -1 ) {
                        pcNode = (ETreeViewNode*)GetRow( nLastSelected );
                        i = nLastSelected;
                } else return;
        }

        uint                                            count = GetRowCount();
        uint                                            j = pcNode->GetIndent();
        bool                                            begin = false;

        for( ; i < count; i++ )
        {
                ETreeViewNode*           node = (ETreeViewNode*)GetRow( i );

                if( begin ) {
                        if( node->GetIndent() == j+1 ) {
                                cvecChildren.push_back( node );
                        } else if( node->GetIndent() <= j ) {
                                return ;
                        }
                }

                if( node == pcNode )
                        begin = true;
        }
}

/** Get sibling nodes
 *      \par    Description:
 *                      Fills a std::vector with the siblings of a supplied
 *                      node that is attached to this ETreeView, or the currently selected
 *                      node if pcNode is NULL.
 *      \param  cvecSibling std::vector to be filled with pointers to nodes.
 *      \param  pcNode Pointer to node to examine or NULL for selected node.
 * \sa GetChildren()
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
void ETreeView::GetSiblings( std::vector<ETreeViewNode*>& cvecSiblings, ETreeViewNode* pcNode )
{
        ETreeViewNode* pcParent = GetParent( pcNode );
        if( pcParent ) {
                GetChildren( cvecSiblings, pcParent );
        }
}

/** Get max size of expander image
 *      \par    Description:
 *                      Returns a Rect with the maximum dimensions for the expander icon.
 *                      Normally there is no reason to call this method directly, it is
 *                      used by the ETreeViewNodes to calculate what their sizes will be.
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
const Rect& ETreeView::GetExpanderImageBounds( ) const
{
        return m->m_cExpImgBounds;
}

void ETreeView::SortRows( std::vector<ListViewRow*>* pcRows, int nColumn )
{
        std::vector<ListViewRow*>::const_iterator               cSource;
        std::vector<ListViewRow*>                                               cDest;
        std::vector<ListViewRow*>::iterator                             cIterator;
        std::vector<ListViewRow*>::iterator                             cBefore;
        std::vector<int>                                                                cPath;
        int                                                                                             nIndent;
        int                                                                                             nBeforeIndex = 0;

        /* NOTE: This routine can probably be optimised a lot. For starters, cDest could be */
        /* a std::list instead of a vector. */

        cPath.push_back( 0 );
        cBefore = cDest.begin();
        cSource = pcRows->begin();
        nIndent = ((ETreeViewNode*)*cSource)->GetIndent();

        for( ; cSource != pcRows->end() ; cSource++ ) {
                int nNewIndent = ((ETreeViewNode*)*cSource)->GetIndent();

                if( nNewIndent < nIndent ) {
                        cPath.pop_back();
                }

                if( nNewIndent > nIndent ) {
                        cPath.push_back( nBeforeIndex );
                }

                cBefore = cIterator = cPath.back() + cDest.begin();

                for( ; cIterator != cDest.end() ; cIterator++ ) {
                        int cmpindent = ((ETreeViewNode*)*cIterator)->GetIndent();
                        if( cmpindent < nNewIndent ) {
                                cBefore = cIterator + 1;
                                break;
                        }
                        if( cmpindent == nNewIndent ) {
                                if( (*cSource)->IsLessThan( (*cIterator), nColumn ) ) {
                                        break;
                                }
                        }
                        cBefore = cIterator + 1;
                }

                nBeforeIndex = (cBefore-cDest.begin());

                cDest.insert( cBefore, (*cSource) );
        
                nIndent = nNewIndent;
        }

        *pcRows = cDest;

    m->m_bTrunkValid = false;
}

void ETreeView::Paint( const Rect& cUpdateRect )
{
        if( !m->m_bTrunkValid && m->m_bDrawTrunk ) {
                _ConnectLines();
                m->m_bTrunkValid = true;
        }

        ListView::Paint( cUpdateRect );
}

/** Remove a node
 *      \par    Description:
 *                      Does exactly the same thing as RemoveRow().
 * \sa ListView::RemoveRow()
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
ETreeViewNode* ETreeView::RemoveNode( int nIndex, bool bUpdate )
{
        m->m_bTrunkValid = false;
        return (ETreeViewNode*)RemoveRow( nIndex, bUpdate );
}

ListViewRow* ETreeView::RemoveRow( int nIndex, bool bUpdate )
{
        m->m_bTrunkValid = false;
        return ListView::RemoveRow( nIndex, bUpdate );
}

		
/** Remove a node
 *      \par    Description:
 *                      Removes the node from the TreeView
 *      \param  pcNode Pointer to node to remove.
 *      \param  bUpdate Set to true to refresh ETreeView.
 * \sa GetChildren()
 * \author Henrik Isaksson (henrik@isaksson.tk)
 *****************************************************************************/
ETreeViewNode*	ETreeView::RemoveNode( ETreeViewNode* pcNode, bool bUpdate )
{
	m->m_bTrunkValid = false;

	uint count = GetRowCount();

	for( uint i = 0 ; i < count; i++ )
	{
		if( GetRow( i )  == pcNode )		
			return (ETreeViewNode*)RemoveNode( i, bUpdate );
	}


	return NULL;
}

void ETreeView::Clear()
{
        m->m_bTrunkValid = false;
        ListView::Clear();
}

/* Disabled - not valid in a ETreeView */

void ETreeView::InsertRow( int nPos, ListViewRow* pcRow, bool bUpdate )
{
}

void ETreeView::InsertRow( ListViewRow* pcRow, bool bUpdate )
{
}

void ETreeView::__TVW_reserved1__()
{
}

void ETreeView::__TVW_reserved2__()
{
}

void ETreeView::__TVW_reserved3__()
{
}

void ETreeView::__TVW_reserved4__()
{
}

void ETreeView::__TVW_reserved5__()
{
}

void ETreeViewNode::__TVN_reserved1__()
{
}

void ETreeViewNode::__TVN_reserved2__()
{
}

void ETreeViewStringNode::__TVS_reserved1__()
{
}

void ETreeViewStringNode::__TVS_reserved2__()
{
}

