// EFileBrowser (C)opyright 2007 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <list>
#include <algorithm>
#include <storage/file.h>
#include "thumbcreator.h"
#include "thumbsdb.h"
#include "dirkeeper.h"

using namespace std;
using namespace os;
using namespace os_priv;

///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E   -   T H U M B C R E A T O R
//
///////////////////////////////////////////////////////////////////////////////

class ThumbCreator :: _Private
{
public:
	_Private() {};

	void _Scaling( Image* img )
	{
		// Calculate scaling
		#define PREVIEW_SIZE 64.0f 

		Point current_size = img->GetSize();
		Point new_size;

		if( current_size.x > current_size.y )
		{
			new_size.x = PREVIEW_SIZE;
			new_size.y = PREVIEW_SIZE * ( current_size.y / current_size.x );
		}
		else	
		{
			new_size.y = PREVIEW_SIZE;
			new_size.x = PREVIEW_SIZE * ( current_size.x / current_size.y );
		}

	 	img->SetSize( new_size );
	}

	DirKeeper*	m_pcDirKeeper;
	String m_cPath;
	bool m_bKilled;
	list< Thumbnail* > m_ListOfThumbnails;
};

///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S   -   T H U M B C R E A T O R
//
///////////////////////////////////////////////////////////////////////////////
ThumbCreator :: ThumbCreator( DirKeeper* keeper, String path ) : Thread( String( String( "thumb_creator_" ) + path ).c_str() ), Locker( "thumb_creator" )
{
printf( "New thumbcreator %s Thread id: %d\n", path.c_str(),GetThreadId() );
	// Create the private class
	m = new _Private();
	m->m_pcDirKeeper = keeper;
	m->m_cPath = path;
	m->m_bKilled = false;
printf( "New thumbcreator created %s Thread id: %d\n", path.c_str(),GetThreadId() );
}

ThumbCreator :: ~ThumbCreator()
{
printf( "Thumbnail deleted %s (%d)\n", m->m_cPath.c_str(), GetThreadId() );
	delete m;
}

void ThumbCreator :: AddThumbnail( Thumbnail* thumb )
{	
	// If thread is killed
	if( m->m_bKilled )
	{
		printf( "Thread already killed\n" );
		return;
	}
	bool found = false;

	// Make sure this thumb is not already in the list
	for( list< Thumbnail* >::iterator it = m->m_ListOfThumbnails.begin() ;
		 it != m->m_ListOfThumbnails.end() ; it++ )
	{
		if( *it == thumb )
		{
			found = true;
			break;
		}
	}

	if( !found )
	{
		printf( "Adding thumbnail %s\n", thumb->GetPath().c_str() );
		m->m_ListOfThumbnails.push_back( thumb );
	}
	else
		printf( "Already existing in the list %s\n", thumb->GetPath().c_str() );
}

void ThumbCreator :: Start( void )
{
	printf( "starting thread %d\n", GetThreadId() );
	Thread::Start();
	printf( "started thread %d\n", GetThreadId() );
}

int32 ThumbCreator :: Run()
{
	int updated = 0;

	while( 1 )
	{

		// Check if queue is empty
		if( !m->m_ListOfThumbnails.empty() )
		{
			Lock();

			// Get thumbnail
			Thumbnail* thumb = m->m_ListOfThumbnails.front();
			m->m_ListOfThumbnails.pop_front();

			Unlock();

			printf( "Creating thumbnail of %s\n", thumb->GetPath().c_str() );	
			
			File* fd;

			fd = new File( thumb->GetPath() );

			Image* image = new BitmapImage( fd );

			m->_Scaling( image );

			Lock();
			thumb->SetImage( image );
			Unlock();

			updated++;

			if( ( updated > 5 || m->m_ListOfThumbnails.empty() ) )
			{
				printf( "Send layout request!\n" );

				Lock();

				Message cMsg( DirKeeper::M_THUMBNAIL_LAYOUT );			

				if( m->m_pcDirKeeper )
				{
					m->m_pcDirKeeper->Lock();
					m->m_pcDirKeeper->PostMessage( &cMsg, m->m_pcDirKeeper );
					m->m_pcDirKeeper->Unlock();
				}

				Unlock();
				updated = 0;
			}
		}

		Delay( 1000 );
	}
}

