// EFileBrowser	 (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <list>
#include <storage/file.h>
#include <storage/path.h>
#include <storage/directory.h>
#include <storage/volumes.h>
#include "foldertreeview.h"
#include "../common.h"

using namespace os;
using namespace std;

static uint8 g_aCDImage[] = {
#include "../images/cd.h"
};

static uint8 g_aDiskImage[] = {
#include "../images/disk.h"
};

static uint8 g_aFloppyImage[] = {
#include "../images/floppy.h"
};

class FolderTreeView::Private
{
public:
	Message*			m_pcDirChangeMsg;
	Path				m_cPath;	
	String				m_cSearchString;
	bool				m_bLocked;
	Invoker				m_Invoker;
};

FolderTreeView::FolderTreeView( const Rect & cFrame, const String& cPath, uint32 nResizeMask ): ETreeView( cFrame, "_folder_treeview", ETreeView::F_NO_HEADER | ETreeView::F_NO_AUTO_SORT, nResizeMask )
{
	m = new FolderTreeView::Private();
        
	m->m_cPath = cPath;
	m->m_pcDirChangeMsg = NULL;
	m->m_bLocked = false;

	InsertColumn( "folders", 1 );
	SetExpandedMessage( new Message( MSG_EXPAND ) );
	SetSelChangeMsg( new Message( MSG_SELECT ) );
	SetExpandedImage( LoadImage( "expander.png" ) );
	SetCollapsedImage( LoadImage( "collapse.png" )  );
	SetDrawTrunk( false );
	SetDrawExpanderBox( false );
}

FolderTreeView::~FolderTreeView()
{
	delete( m );
}

void FolderTreeView::SetDirChangeMsg( Message * pcMsg )
{
	delete m->m_pcDirChangeMsg;
	m->m_pcDirChangeMsg = pcMsg;
}

void FolderTreeView::AllAttached()
{        
	ETreeView::SetTarget( this );
	ReadRoot();
}

void FolderTreeView::DetachedFromWindow()
{
	ETreeView::DetachedFromWindow();
}

status_t FolderTreeView::SetTarget( const Handler* pcHandler, const Looper* pcLooper )
{
	return m->m_Invoker.SetTarget( pcHandler, pcLooper );
}

status_t FolderTreeView::SetTarget( const Messenger& cMessenger )
{
	return m->m_Invoker.SetTarget( cMessenger );
}

void FolderTreeView::HandleMessage( Message* pcMessage )
{
	switch ( pcMessage->GetCode() )
	{
		case MSG_EXPAND:
			{
				bool expanded;
				ETreeViewNode* node;

				if( pcMessage->FindBool( "expanded", &expanded) != 0 )
					break;

				if( pcMessage->FindPointer( "node", (void**) &node) != 0 )
					break;

				if( expanded  )
					ExpandItem( GetLastSelected(), node );
				else
					CollapseItem( GetLastSelected(), node );
			break;
			}
		case MSG_SELECT:
			{
				ListViewRow* row = GetRow( GetLastSelected() );

				if(row != NULL )
					ChangePath( row->GetCookie().AsString() + "/" );				
			}
	}
}

void FolderTreeView :: ReadRoot()
{
	Volumes cVolumes;
	String cMountPoint;
	BitmapImage* pcItemIcon = NULL;
	vector<String> list_of_mounts;

	for( uint i = 0; i < cVolumes.GetMountPointCount(); i++ )
	{
		if( cVolumes.GetMountPoint( i, &cMountPoint ) == 0 )
		{
			fs_info sInfo;
			
			if( cVolumes.GetFSInfo( i, &sInfo ) == 0 && ( sInfo.fi_flags & FS_IS_BLOCKBASED ) )
			{
				list_of_mounts.push_back( cMountPoint );

				String zVolumeName = sInfo.fi_volume_name;
				if( ( zVolumeName == "" ) || ( zVolumeName == "/" ) )
					zVolumeName = "Unknown";				
			
				// Create nice icon (from IconDirView)
				String zDevice = Path( sInfo.fi_device_path ).GetDir().GetLeaf();                                        
				if( zDevice.Length() > 2 )
				{
					os::StreamableIO* pcSource = NULL;
					if( zDevice[0] == 'c' && zDevice[1] == 'd' ) {
						try {
							pcSource = new os::File( "/system/icons/cdrom.png" );
						} catch( ... ) {
							pcSource = new MemFile( g_aCDImage, sizeof( g_aCDImage ) );
						}
					}
					else if( zDevice[0] == 'f' && zDevice[1] == 'd' ) {
						try {
							pcSource = new os::File( "/system/icons/floppy.png" );
						} catch( ... ) {
							pcSource = new MemFile( g_aFloppyImage, sizeof( g_aFloppyImage ) );
						}
					}
					else {
						try {
							pcSource = new os::File( "/system/icons/harddisk.png" );
						} catch( ... ) {
							pcSource = new MemFile( g_aDiskImage, sizeof( g_aDiskImage ) );
						}
					}

					pcItemIcon = new BitmapImage();
                                                                        
					pcItemIcon->Load( pcSource );
					delete( pcSource );
				}

				if( pcItemIcon == NULL )
				{
					// Default icon
					MemFile* pcSource = new MemFile( g_aDiskImage, sizeof( g_aDiskImage ) );
					pcItemIcon = new BitmapImage();
					pcItemIcon->Load( pcSource );
					delete( pcSource );
				}
        
				if( pcItemIcon && !( pcItemIcon->GetSize() == Point( 24, 24 ) ) )
					pcItemIcon->SetSize( os::Point( 24, 24 ) );
			
				// Add drive to tree	
				ETreeViewStringNode* root_node = new ETreeViewStringNode();
				root_node->AppendString( zVolumeName );
				root_node->SetIndent( 1 );
				root_node->SetCookie( cMountPoint );	
				root_node->SetIcon( pcItemIcon );
				InsertNode( root_node );		

				// Check if there is subdirectories
				if( FindSubDir( cMountPoint.c_str() ) )
				{
					// Insert a empty item
					ETreeViewStringNode* node = new ETreeViewStringNode();								
					node->AppendString( "Please wait..." );
					root_node->InsertChild( node );
					root_node->SetExpanded( false );
				}

			}
		}
	}

	list<String> list_of_dirs;

	try
	{
		String cName;
		Directory dir( "/" );
		dir.Rewind();

		while( dir.GetNextEntry( &cName ) == 1 )
		{
			if( cName != "." && cName != ".." && cName != "dev")
			{		
				bool found = false;
				String t = String( "/" ) + cName;
	
				// Check if this is a folder and that it is not mount point
				for( int i = 0 ; i < (int) list_of_mounts.size() ; i++ )
				{
					if( list_of_mounts[ i ] == t )
					{
						found = true;
						break;
					}
				}

				FSNode fs( t );
				if( fs.IsDir() && !found )
					list_of_dirs.push_back( cName );
			}
		}
	}
	catch( ... ) { }

	// sort the list
	list_of_dirs.sort();

	// reverse the list
	list_of_dirs.reverse();

	for( list<String>::iterator it = list_of_dirs.begin() ; it != list_of_dirs.end() ; it++ )
	{
		// Add drive to tree
		ETreeViewStringNode* root_node = new ETreeViewStringNode();
		root_node->AppendString( *it );
		root_node->SetIndent( 1 );
		root_node->SetCookie( String( "/" ) + (*it) );
		InsertNode( root_node );	

		// Check if there is subdir
		if( FindSubDir( String( "/")  + (*it)) )
		{
			// Insert a empty item
			ETreeViewStringNode* empty = new ETreeViewStringNode();
			empty->AppendString( "Please wait..." );
	 		root_node->InsertChild( empty, false );
		}	

		root_node->SetExpanded( false );
	}

	RefreshLayout();	
}

bool FolderTreeView :: FindSubDir( String path )
{
	// Dont read the dev directory as that will cause EFilebrowser to crash (why???)
	if( path.find( "/dev" ) == 0 )
		return false;

	// Check if there is any subdirectories
	try
	{
		String cName;
		Directory dir( path );
		dir.Rewind();

		while( dir.GetNextEntry( &cName ) == 1 )
		{
			if( cName != "." && cName != ".." )
			{
				// Check if this is a folder

				FSNode fs( path + '/' + cName );				
				if( fs.IsDir() )
					return true;
			}
		}
	}
	catch( ... ) { }

	return false;
}

void FolderTreeView :: DisplayPath( int row, ETreeViewNode* parent, String path )
{
	list<String> list_of_dirs;

	try
	{
		String cName;
		Directory dir( path );
		dir.Rewind();

		while( dir.GetNextEntry( &cName ) == 1 )
		{
			if( cName != "." && cName != ".." )
			{				
				// Check if this is a folder
				FSNode fs( path + '/' + cName );				
				if( fs.IsDir() )
					list_of_dirs.push_back( cName );
			}
		}
	}
	catch( ... ) { }

	// sort the list
	list_of_dirs.sort();

	// reverse the list
	list_of_dirs.reverse();

	for( list<String>::iterator it = list_of_dirs.begin() ; it != list_of_dirs.end() ; it++ )
	{
		// Add drive to tree
		ETreeViewStringNode* node = new ETreeViewStringNode();
		node->AppendString( *it );
		node->SetCookie( path + '/' + *it );
		node->SetExpanded( false );
		parent->InsertChild( node, false );

		// Check if there is subdir
		if( FindSubDir( path + '/' + *(it)) )
		{
			// Insert a empty item
			ETreeViewStringNode* empty = new ETreeViewStringNode();
			empty->AppendString( "Please wait..." );
			node->InsertChild( empty, false );
		}	
	}

	parent->SetExpanded( true );

	RefreshLayout();
}


void FolderTreeView :: ExpandItem( int row, ETreeViewNode* parent )
{
	parent->RemoveChildren( false );

	DisplayPath( row, parent, parent->GetCookie().AsString() );
}

void FolderTreeView :: CollapseItem( int row, ETreeViewNode* parent )
{
	// Remove all items but create an empty
	parent->RemoveChildren( false );

	// Insert a empty item
	ETreeViewStringNode* empty = new ETreeViewStringNode();
	empty->AppendString( "Please wait..." );
	parent->InsertChild( empty );

	// Check if we need to update the path
	if( m->m_cPath.GetPath().find( parent->GetCookie().AsString().c_str() ) == 0 )	
	{
		ChangePath( parent->GetCookie().AsString() + "/" );
	}
}

void FolderTreeView :: SetPath( const String& cPath )
{
	String path = cPath;

	// Fix path
	if( path[ path.size() - 1 ] != '/' && path.size() > 1 )
		path = path + "/";
	
	if( path[0] != '/' )
		path = String( "/" ) + path;

	// Is this path already the current one?
	if( m->m_cPath == path )
		return;

	m->m_cPath = path;

	// Check if this is root
	if( path == "/" )
	{
		ClearSelection();
		return;
	}

	vector<String> res;

	SplitPath( path, res );

	for( int i = 0 ; i < (int) res.size() - 1; i++ )
	{
		// Find item
		for( uint j = 0 ; j < GetRowCount() ; j++ )
		{
			ETreeViewNode* row = dynamic_cast<ETreeViewNode*>( GetRow( j ) );
			
			if( res[i] == row->GetCookie().AsString() && !row->IsExpanded() )
			{
				ExpandItem( j, row );							
				break;
			}
		}
	}
	
	ClearSelection();

	// Select last entry
	for( int j = 0 ; j < (int) GetRowCount() ; j++ )
	{
		ETreeViewNode* row = dynamic_cast<ETreeViewNode*>( GetRow( j ) );
			
		if( res[ res.size() - 1 ] == row->GetCookie().AsString() )
		{
			Select( j );							
			break;
		}
	}
}

void FolderTreeView :: SplitPath( const String& cPath, std::vector<String>& vec )
{
	int a = 1;
	int b = 1;

	String leaf;

	while( 1 )
	{
		b = cPath.find( "/", a );

		if( b == (int) string::npos )
			break;

		leaf = cPath.substr( 0, b  );

		vec.push_back( leaf );
		
		a = b + 1;
	}
}

void FolderTreeView :: ChangePath( String path )
{
	if( path != m->m_cPath && m->m_pcDirChangeMsg != NULL )
	{
		m->m_pcDirChangeMsg->RemoveName( "file/path" );
		m->m_pcDirChangeMsg->AddString( "file/path", path );
		m->m_Invoker.Invoke( m->m_pcDirChangeMsg );
	}
}

