// EFileBrowser (C)opyright 2007 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <util/application.h>
#include <util/settings.h>
#include "gui/splitter.h"
#include <gui/menu.h>
#include <gui/statusbar.h>
#include <gui/toolbar.h>
#include <gui/layoutview.h>
#include "address_field.h"
#include "appwindow.h"
#include "messages.h"
#include "preferences.h"
#include "sidebar_foldertreeview.h"
#include "sidebar_info.h"
#include "common.h"
#include "gui/tabview.h"
#include "tab_browser.h"
#include "about.h"

using namespace os;

///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E
//
///////////////////////////////////////////////////////////////////////////////

class AppWindow :: _Private
{
public:
	_Private()
	{
		m_DiaAbout = NULL;	
	};

	void _CreateMenu()
	{
		m_MenuBar = new Menu( Rect(), "", ITEMS_IN_ROW );				
	
		// Application menu
		m_ApplicationMenu = new Menu( Rect(), "Application", ITEMS_IN_COLUMN );
		m_ApplicationMenu->AddItem( "Quit", new Message( MSG_MENU_APP_QUIT ), "Ctrl + Q" );
		m_ApplicationMenu->AddItem( new MenuSeparator() );
		m_ApplicationMenu->AddItem( "About", new Message( MSG_MENU_APP_ABOUT ) );
		m_MenuBar->AddItem( m_ApplicationMenu );
		
	}

	BrowserTab* _GetCurrentBrowserTab( bool create = true )
	{
		BrowserTab* ret = NULL;

		if( !( ( m_TabView->GetTabCount() < 1 ) || ( (int) m_TabView->GetSelection() < 0 ) ) )
		{
			ret = dynamic_cast< BrowserTab* >( m_TabView->GetTabView( m_TabView->GetSelection() ) );
		}

		if( ret == NULL && create )
		{
			ret = _CreateBrowserTab( getenv( "HOME" ) );
			m_TabView->SetSelection( 0 );
		}

		return ret;
	}

	BrowserTab* _CreateBrowserTab( String path )
	{
		BrowserTab* tab = new BrowserTab( path, 
										  new Message( _Private::MSG_BROWSER_DIR_CHANGE ), 
										  new Message( _Private::MSG_BROWSER_READY ),
 										  new Message( _Private::MSG_BROWSER_SEL_CHANGE ),
 										  new Message( _Private::MSG_BROWSER_OPEN_TAB ) );
		m_TabView->AppendTab( tab->GetName(), tab ); 

		tab->SetView( m_Preferences.GetDefaultView() );

		return tab;
	}

	void _UpdateTabName()
	{
		BrowserTab* tab = _GetCurrentBrowserTab( false );

		if( tab != NULL )
			m_TabView->SetTabTitle( m_TabView->GetSelection(), tab->GetName() );
	}

	void _UpdateWindowTitle( Window* window )
	{
		BrowserTab* tab = _GetCurrentBrowserTab();

		if( tab != NULL )
		{
			String title = tab->GetPath();

			if( title == "/" )
				title = "Mounted Drives";

			title += " - EFileBrowser";
			window->SetTitle( title );
		}
	}

	void _UpdateStatusBar()
	{
		BrowserTab* tab = _GetCurrentBrowserTab();

		if( tab != NULL )
		{
			struct BrowserTab::DirectoryStat stat;

			tab->GetStat( &stat );

			String text;

			if( tab->GetPath() != "/" )
			{
				if( stat.no_folders == 0 && stat.no_files == 0 )
				{
					text = "No objects founds.";
				}
				else if( stat.no_sel_folders == 0 && stat.no_sel_files == 0 )
				{
					String t;				
					t.Format( "Total %d objects (of which %d is folders).", stat.no_folders + stat.no_files, stat.no_folders);

					int size = stat.total_size;
					String f;
					if( size >= 1000000 )
						f.Format( "Total size %i Mb.", ( int )( size / 1000000 + 1 ) );
					else if ( size >= 1000 )
						f.Format( "Total size %i Kb.", ( int )( size / 1000 + 1 ) );
					else
						f.Format( "Total size %i bytes.", size );
	
					text = t + String( " " ) + f;
				}
				else
				{
					String t;
					t.Format( "%d objects (of which %d is folders) selected.", stat.no_sel_folders + stat.no_sel_files, stat.no_sel_folders );
	
					int size = stat.total_sel_size;
					String f;
					if( size >= 1000000 )
						f.Format( "Total size %i Mb selected.", ( int )( size / 1000000 + 1 ) );
					else if ( size >= 1000 )
						f.Format( "Total size %i Kb selected.", ( int )( size / 1000 + 1 ) );
					else
						f.Format( "Total size %i bytes selected.", size );
				
					text = t + String( " " ) + f;
				}
			}
			else
			{
				if( stat.no_files == 0 )
				{
					text = "No objects founds.";
				}
				else if( stat.no_sel_files == 1 )
				{
					text.Format( "One object selected. %d %% free.", stat.total_sel_size );								
				}
				else if( stat.no_sel_files > 0 )
				{
					text.Format( "%d selected objects.",stat.no_sel_files );
				}
				else
					text.Format( "%d objects.",stat.no_files );
			}

			m_StatusBar->SetText( "message", text );
		}
	}

	enum AppWindowMessage { MSG_ADDRESS_FIELD,
							MSG_BROWSER_DIR_CHANGE, MSG_BROWSER_READY, MSG_BROWSER_SEL_CHANGE, MSG_BROWSER_OPEN_TAB,
							MSG_TAB_CHANGE, MSG_TAB_DRAGDROP, MSG_TAB_CLOSE,
							MSG_TOOLBAR_HOME, MSG_TOOLBAR_DESKTOP, MSG_TOOLBAR_BACK, MSG_TOOLBAR_UP, MSG_TOOLBAR_SHOW_SIDEBAR,
							MSG_MENU_APP_QUIT, MSG_MENU_APP_ABOUT,
							MSG_SIDEBAR_HIDE, MSG_SIDEBAR_SET_PATH };
	
	// Widgets
	Menu* m_MenuBar;
	Menu* m_ApplicationMenu;
	ToolBar* m_ToolBar;
	SideBar* m_SideBar;
	TabView* m_TabView;
	Splitter* m_Splitter;
	StatusBar* m_StatusBar;
	AddressField* m_AddressField;

	Preferences m_Preferences;

	// Dialogs;
	About* m_DiaAbout;
};

///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S
//
///////////////////////////////////////////////////////////////////////////////
AppWindow :: AppWindow( String path, const Rect& cFrame ) : Window( cFrame, "main_window", "EFileBrowser" )
{	
	// Create the private class
	m = new _Private();

	// Load settings
	m->m_Preferences.Load();

	// Create widgets
	LayoutNode*	root_node = new VLayoutNode( "pcRootNode" );
	LayoutView*	root_view = new LayoutView( GetBounds(), "pcRootView" );
	root_view->SetRoot( root_node );
	AddChild( root_view );

	// Menu
	m->_CreateMenu();
	m->m_MenuBar->SetTargetForItems( this );

	// Toolbar	
	m->m_ToolBar = new ToolBar( Rect(), "toolbar" );
	m->m_ToolBar->AddButton( "home", "Home", LoadImage( "home.png" ), new Message( _Private::MSG_TOOLBAR_HOME ) );
	m->m_ToolBar->AddButton( "desktop", "Desktop", LoadImage( "desktop.png" ), new Message( _Private::MSG_TOOLBAR_DESKTOP ) );

	m->m_ToolBar->AddSeparator( "" );

	m->m_ToolBar->AddButton( "back", "Back", LoadImage( "back.png" ), new Message( _Private::MSG_TOOLBAR_BACK ) );
	m->m_ToolBar->AddButton( "up", "Up", LoadImage( "up.png" ), new Message( _Private::MSG_TOOLBAR_UP ) );

	m->m_ToolBar->AddSeparator( "" );

	Message* msg = new Message( _Private::MSG_TOOLBAR_SHOW_SIDEBAR );
	SideBar* default_sidebar = new SideBarFolderTreeView( new Message( _Private::MSG_SIDEBAR_HIDE  ), new Message( _Private::MSG_SIDEBAR_SET_PATH  ) );
	msg->AddPointer( "sidebar",  default_sidebar );
	m->m_ToolBar->AddButton( "sidebar_folders", "Folders", LoadImage( "up.png" ), msg );

	msg = new Message( _Private::MSG_TOOLBAR_SHOW_SIDEBAR );
	msg->AddPointer( "sidebar",  new SideBarInfo( new Message( _Private::MSG_SIDEBAR_HIDE  ), new Message( _Private::MSG_SIDEBAR_SET_PATH  ) ) );
	m->m_ToolBar->AddButton( "sidebar_info", "Info", LoadImage( "back.png" ), msg );

	// Address field
	m->m_AddressField = new AddressField( path, new Message( _Private::MSG_ADDRESS_FIELD ) );
	m->m_AddressField->SetTarget( this );
	m->m_AddressField->SetPath( path );

	// Sidebar
	m->m_SideBar = default_sidebar;
	m->m_SideBar->SetPath( path );

	// TabView
	m->m_TabView = new TabView( Rect(), "tab_view" );
	m->m_TabView->SetMessage( new Message( _Private::MSG_TAB_CHANGE ) );
	m->m_TabView->SetDragDropMsg( new Message( _Private::MSG_TAB_DRAGDROP ) );
	m->m_TabView->SetCloseTabMsg( new Message( _Private::MSG_TAB_CLOSE ) );
	m->m_TabView->SetTarget( this );
	m->m_TabView->SetTabType( TabView::SHOW_TABS );

	// Create default tab
	BrowserTab* tab = m->_CreateBrowserTab( path );

	// Create the statusbar
	m->m_StatusBar = new StatusBar( Rect(), "" );
	m->m_StatusBar->AddPanel( "message", "" );

	// For the tab and the status
	LayoutNode*	tabview_node = new VLayoutNode( "pcRootNode" );
	LayoutView*	tabview_view = new LayoutView( GetBounds(), "pcRootView" );
	tabview_view->SetRoot( tabview_node );
	tabview_node->AddChild( m->m_TabView, 1.0f );
	tabview_node->AddChild( m->m_StatusBar, 0.0f );

	// Splitter
	m->m_Splitter = new Splitter( Rect(), "", m->m_SideBar, tabview_view, HORIZONTAL );
	m->m_Splitter->SetSplitRatio( m->m_Preferences.GetSplitRatio() );

	// Add widgets to the layout system
//	root_node->AddChild( m->m_MenuBar, 0.0f );
	root_node->AddChild( m->m_ToolBar, 0.0f );
	root_node->AddChild( m->m_AddressField, 0.0f );
	root_node->AddChild( m->m_Splitter );
	root_view->InvalidateLayout();

	// Setup everything
	tab->ReRead();
	m->_UpdateTabName();
	m->_UpdateWindowTitle( this );
}

AppWindow :: ~AppWindow()
{
	delete m->m_DiaAbout;
	delete m;
}

void AppWindow :: HandleMessage( Message* pcMessage )
{
	switch( pcMessage->GetCode() )	//Get the message code from the message
	{	
		// Menubar messages
		case _Private::MSG_MENU_APP_QUIT:
			OkToQuit();
			break;
		case _Private::MSG_MENU_APP_ABOUT:
			if( m->m_DiaAbout == NULL )
				m->m_DiaAbout = new About();

			m->m_DiaAbout->Raise();
			break;

		// Address bar messages
		case _Private::MSG_ADDRESS_FIELD:
		{
			String p;

			if( pcMessage->FindString( "file/path", &p.str() ) == EOK )
			{
				BrowserTab* tab = m->_GetCurrentBrowserTab();

				if( tab != NULL )
				{
printf( "1\n" );		
					tab->SetPath( p );	
printf( "2\n" );
					m->m_AddressField->SetPath( tab->GetPath() );
printf( "3\n" );
					m->m_SideBar->SetPath( tab->GetPath() );
printf( "4\n" );
					m->_UpdateTabName();
printf( "5\n" );
					m->_UpdateWindowTitle( this );
printf( "6\n" );
				}
			}
			break;
		}		

		// Browser message
		case _Private::MSG_BROWSER_DIR_CHANGE:
		{
			BrowserTab* tab = m->_GetCurrentBrowserTab();
			if( tab != NULL )
			{			
				String path;

				if( pcMessage->FindString( "file/path", &path.str() ) == EOK )
				{
					tab->SetPath( path );
					m->m_AddressField->SetPath( tab->GetPath() );
					m->m_SideBar->SetPath( tab->GetPath() );
					m->_UpdateTabName();
					m->_UpdateWindowTitle( this );
				}
			}
			break;
		}
		case _Private::MSG_BROWSER_READY:
		{
			m->_UpdateStatusBar();
			break;
		}
		case _Private::MSG_BROWSER_SEL_CHANGE:
		{
			m->_UpdateStatusBar();
			break;
		}
		case _Private::MSG_BROWSER_OPEN_TAB:
		{
			String path;
			int i = 0;
				
			while( pcMessage->FindString( "file/path", &path, i ) == EOK )
			{				
				FSNode fn( path );
	
				// only open if it is a directory
				if( fn.IsDir() )
				{
					BrowserTab* tab = m->_CreateBrowserTab( path );
					tab->ReRead();
				}
				i++;
			}

			break;
		}

		// Tab changed
		case _Private::MSG_TAB_CHANGE:
		{
			BrowserTab* tab = m->_GetCurrentBrowserTab();
			if( tab != NULL )
			{			
				m->m_AddressField->SetPath( tab->GetPath() );
				m->m_SideBar->SetPath( tab->GetPath() );
				m->_UpdateTabName();
				m->_UpdateWindowTitle( this );
				m->_UpdateStatusBar();
			}
			break;
		}
		case _Private::MSG_TAB_DRAGDROP:
		{
			Message p;

			if( pcMessage->FindMessage( "msg", &p ) == EOK )
			{
				String path;
				int i = 0;
				
				while( p.FindString( "file/path", &path, i ) == EOK )
				{				
					FSNode fn( path );
	
					// only open if it is a directory
					if( fn.IsDir() )
					{
						BrowserTab* tab = m->_CreateBrowserTab( path );
						tab->ReRead();
						m->m_AddressField->SetPath( tab->GetPath() );
						m->m_SideBar->SetPath( tab->GetPath() );
						m->_UpdateTabName();
						m->_UpdateWindowTitle( this );
						m->m_TabView->SetSelection( m->m_TabView->GetTabCount() - 1 );
						m->_UpdateStatusBar();
					}
					i++;
				}
			}
			break;
		}
		case _Private::MSG_TAB_CLOSE:
		{
			int tab_no;
			
			if( pcMessage->FindInt32( "tab", &tab_no ) == EOK )
			{
				delete m->m_TabView->DeleteTab( tab_no );
				BrowserTab* tab = m->_GetCurrentBrowserTab( false );

				if( tab != NULL )
				{
					m->m_AddressField->SetPath( tab->GetPath() );
					m->m_SideBar->SetPath( tab->GetPath() );
					m->_UpdateTabName();
					m->_UpdateWindowTitle( this );
					m->_UpdateStatusBar();
				}
			}
			break;
		}

		// Toolbar messages
		case _Private::MSG_TOOLBAR_HOME:
		{
			BrowserTab* tab = m->_GetCurrentBrowserTab();
			if( tab != NULL )
			{
				tab->SetPath( getenv( "HOME" ) );
				m->m_AddressField->SetPath( tab->GetPath() );
				m->m_SideBar->SetPath( tab->GetPath() );
				m->_UpdateTabName();
				m->_UpdateWindowTitle( this );
				m->_UpdateStatusBar();
			}
			break;
		}
		case _Private::MSG_TOOLBAR_DESKTOP:
		{
			BrowserTab* tab = m->_GetCurrentBrowserTab();
			if( tab != NULL )
			{
				tab->SetPath( String( getenv( "HOME" ) ) + String( "/Desktop" ) );
				m->m_AddressField->SetPath( tab->GetPath() );
				m->m_SideBar->SetPath( tab->GetPath() );
				m->_UpdateTabName();
				m->_UpdateWindowTitle( this );
				m->_UpdateStatusBar();
			}
			break;
		}
		case _Private::MSG_TOOLBAR_BACK:
		{
			BrowserTab* tab = m->_GetCurrentBrowserTab();
			if( tab != NULL )
			{
				tab->Back();
				m->m_AddressField->SetPath( tab->GetPath() );
				m->m_SideBar->SetPath( tab->GetPath() );
				m->_UpdateTabName();
				m->_UpdateWindowTitle( this );
				m->_UpdateStatusBar();
			}
			break;
		}
		case _Private::MSG_TOOLBAR_UP:
		{
			BrowserTab* tab = m->_GetCurrentBrowserTab();
			if( tab != NULL )
			{
				tab->Up();
				m->m_AddressField->SetPath( tab->GetPath() );
				m->m_SideBar->SetPath( tab->GetPath() );
				m->_UpdateTabName();
				m->_UpdateWindowTitle( this );
				m->_UpdateStatusBar();
			}
			break;
		}
		case _Private::MSG_TOOLBAR_SHOW_SIDEBAR:
		{
			SideBar* sidebar;

			if( pcMessage->FindPointer( "sidebar", (void**) &sidebar ) == EOK )
			{
				if( !m->m_Splitter->IsView1Visible() )
					m->m_Splitter->ShowView1();

				m->m_SideBar = sidebar;

				m->m_Splitter->SetView1( sidebar );
				m->m_Splitter->AdjustLayout();

				BrowserTab* tab = m->_GetCurrentBrowserTab();
				if( tab != NULL )
				{
					m->m_SideBar->SetPath( tab->GetPath() );
				}
			}
			
			break;
		}

		// From sidebar
		case _Private::MSG_SIDEBAR_HIDE:
		{
			m->m_Splitter->HideView1();
			m->m_Splitter->AdjustLayout();
			break;
		}
		case _Private::MSG_SIDEBAR_SET_PATH:
		{	
			String path;

			if( pcMessage->FindString( "file/path", &path ) == EOK )
			{
				BrowserTab* tab = m->_GetCurrentBrowserTab();
				if( tab != NULL )
				{
					tab->SetPath( path );
					m->m_AddressField->SetPath( tab->GetPath() );
					m->_UpdateTabName();
					m->_UpdateWindowTitle( this );
					m->_UpdateStatusBar();
				}
			}
			break;
		}

		default:
			Window::HandleMessage( pcMessage );
			break;
	}
}

bool AppWindow :: OkToQuit( void )
{
	m->m_Preferences.Save();
	Application::GetInstance()->PostMessage( M_QUIT );
	return true;	
}

