// oCADis (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


#include "fparser.h"
#include "toolbox_point.h"

using namespace os;

ToolboxPoint :: ToolboxPoint( const Rect& cFrame, String label, String entry_1, String entry_2, Message* msg, bool allow_rel, uint32 nResizeMask, uint32 nFlags) :
				Control( cFrame, "", "", msg, nResizeMask, nFlags )
{
	m_theMessage = msg;
	m_FrameLabel = label;

	// Calculate font height
	font_height tmp;	
	GetFontHeight( &tmp );
	m_FontHeight = tmp.ascender + tmp.descender;

	// Create the frame view
	m_Frame = new ToolboxFrame( Rect(), "", label );
	AddChild( m_Frame );

	// Create table and attach our widgets
	m_Table = new TableView( cFrame, "", 2 , ( allow_rel ) ? 3 : 2, false, CF_FOLLOW_ALL );
	m_Table->SetRowSpacings( 2 );
	if( allow_rel )
		m_Table->SetRowSpacing( 1, 10 );

	m_Frame->AddChild( m_Table );

	m_Label_1 = new StringView( Rect(), "", entry_1 );
	m_Table->Attach( m_Label_1, 0, 1, 0, 1, 0, 0 );

	m_Entry_1 = new SLTextView( Rect(), "", new Message( MSG_ENTRY1 ) );
	m_Entry_1->SetTarget( this );
	m_Entry_1->SetMinWidth( 40.0f );
	m_Table->Attach( m_Entry_1, 1, 2, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m_Label_2 = new StringView( Rect(), "", entry_2 );
	m_Table->Attach( m_Label_2, 0, 1, 1, 2, 0, 0 );

	m_Entry_2 = new SLTextView( Rect(), "", new Message( MSG_ENTRY2 ) );
	m_Entry_2->SetTarget( this );
	m_Entry_2->SetMinWidth( 40.0f );
	m_Table->Attach( m_Entry_2, 1, 2, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// If relative coord is allowed
	if( allow_rel )
	{
		m_RelativeCoord = new CheckBox( Rect(), "", "Relative coordinates", new Message( MSG_RELATIVE ) );
		m_Table->Attach( m_RelativeCoord, 0, 2, 2, 3, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	}
	else
		m_RelativeCoord = NULL;

	_Layout();
}

ToolboxPoint :: ~ToolboxPoint()
{
	delete m_Frame;
	delete m_RelativeCoord;
	delete m_Label_1;
	delete m_Entry_1;
	delete m_Label_2;
	delete m_Entry_2;
	delete m_Table;

}

void ToolboxPoint :: AllAttached( void )
{
	m_Entry_1->SetTarget( this );
	m_Entry_2->SetTarget( this );
}

void ToolboxPoint :: HandleMessage( Message* msg )
{
	switch( msg->GetCode() )
	{	
		case MSG_ENTRY1:
		{
			if( _CheckEntryForPoint() )
				break;

			m_Entry_2->MakeFocus();
			break;
		}
		case MSG_ENTRY2:
		{
			_InvokePoint( m_Entry_1->GetText(), m_Entry_2->GetText() );
			break;
		}
		case MSG_RELATIVE:
		{
			break;
		}
		
	}
}

void ToolboxPoint :: SetFrame( const Rect& cRect, bool bNotifyServer )
{
	View::SetFrame( cRect, bNotifyServer );
	_Layout();
}

Point ToolboxPoint :: GetPreferredSize( bool bLargest ) const
{
	if( bLargest )
	return Point( COORD_MAX, COORD_MAX );

	Point table = m_Table->GetPreferredSize( false );

	return Point( table.x + 3, table.y + m_FontHeight + 11);
}

void ToolboxPoint :: SetValue( Point point )
{
	// If any of the entries has focus we do not allow the new value to be valid
	if( m_Entry_1->HasFocus() || m_Entry_2->HasFocus() )
		return;

	// Check if to use relative coord
	if( m_RelativeCoord != NULL && m_RelativeCoord->GetValue().AsBool() )
		point -= m_Origin;

	String t;

	t.Format( "%f", point.x );
	m_Entry_1->SetText( t );
	t.Format( "%f", point.y );
	m_Entry_2->SetText( t );

}

void ToolboxPoint :: ClearValue()
{
	// If any of the entries has focus we do not allow the new value to be valid
	if( m_Entry_1->HasFocus() || m_Entry_2->HasFocus() )
		return;

	m_Entry_1->Clear();
	m_Entry_2->Clear();
}

/////////////////////////////////////////////////////////////////////
//
// P R I V A T E   M E T H O D S
//
/////////////////////////////////////////////////////////////////////
bool ToolboxPoint :: _CheckEntryForPoint()
{
	String value = m_Entry_1->GetText();

	// check if we can find separate
	uint i = value.find( ";" );

	if( i != std::string::npos )
	{
		String v1( value, 0, i );
		String v2( value, i + 1, value.Length() - i );

		_InvokePoint( v1, v2 );

		return true;
	}

	return false;
}

void ToolboxPoint :: _InvokePoint( String value1, String value2 )
{
	FunctionParser fparser;
	const char* eval_errors[] = { "Divison by zero", "Sqrt error", "Log error", "Trigonometric error", "Maximum recursion" };
	Point coord;

	value1.Strip();
	value2.Strip();

	// Set up parser for value1
	int ret = fparser.Parse( value1, "" );
	if( ret >=0 )
	{
		printf( "Error parsing!: %s\n", fparser.ErrorMsg() );
		return;
	}

	// Do the parsing for value1
	coord.x = fparser.Eval( NULL );
	ret = fparser.EvalError();
	if( ret > 0 )
	{
		printf( "Error val!: %s\n", eval_errors[ ret - 1 ] );
		return;
	}

	// Set up parser for value2
	ret = fparser.Parse( value2, "" );
	if( ret >=0 )
	{
		printf( "Error parsing!: %s\n", fparser.ErrorMsg() );
		return;
	}

	// Do the parsing for value2
	coord.y = fparser.Eval( NULL );
	ret = fparser.EvalError();
	if( ret > 0 )
	{
		printf( "Error val!: %s\n", eval_errors[ ret - 1 ] );
		return;
	}

	// Check if to use relative coord
	if( m_RelativeCoord != NULL && m_RelativeCoord->GetValue().AsBool() )
		coord  += m_Origin;

	m_theMessage->RemoveName("value" );
	m_theMessage->AddPoint( "value", coord );

	Invoke( m_theMessage );
}

void ToolboxPoint :: _Layout()
{
	m_Frame->SetFrame( GetBounds() );
	Rect cFrame = ConvertFromParent( GetFrame() );	
	cFrame.Resize( 5, m_FontHeight + 5 , -5, -2 );
	m_Table->SetFrame( cFrame );
}
