// oCADis (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <math.h>

#include "tool_circle_state_2.h"
#include "common.h"
#include "object_circle.h"

using namespace os;

///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E
//
///////////////////////////////////////////////////////////////////////////////

class CircleState2 :: _Private
{
public:
	_Private(){};

	void _Layout( Rect cFrame )
	{
		m_Table->SetFrame( Rect( 0, 0, cFrame.Width(), cFrame.Height() ) );
	}

	void _CreateObject( Tool* tool, Canvas* canvas, Document* doc, Point center, float diameter )
	{
		canvas->Undraw();

		// Create object
		ObjectCircle* obj = new ObjectCircle( center, diameter );
		doc->AddObject( obj );
		obj->Draw( canvas );

		// Create Undo object
		UndoAddObject* undo = new UndoAddObject( "Create Circle", obj );
		doc->AddUndoAction( undo );
		doc->SetModification();

		tool->SetState( 0 );
	}

	float _CalculateDiameter( Point p2, Point p1 )
	{
		Point dist = p1 - p2;
		float d = dist.x * dist.x + dist.y * dist.y;
		return sqrt( d );
	}

	enum CircleState2Message { MSG_CLOSE, MSG_POINT, MSG_RADIUS, MSG_CANCEL };

	ToolboxValue* m_DiameterValue;
	ToolboxAction* m_Action;

	os::TableView* m_Table;
};

///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S
//
///////////////////////////////////////////////////////////////////////////////
CircleState2 :: CircleState2( ToolCircle* theTool, ToolboxSideBar* toolbox ) : ToolState( theTool, toolbox )
{
	// Create the private class
	m = new _Private();

	// Create the table
	m->m_Table = new TableView( Rect(), "", 1, 2, false );
	m->m_Table->SetRowSpacings( 4 );
	AddChild( m->m_Table );

	// Create buttons and attach them to table
	m->m_DiameterValue = new ToolboxValue( Rect(), "Diameter", "", new Message( _Private::MSG_POINT ) );
	m->m_Table->Attach( m->m_DiameterValue, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m->m_Action = new ToolboxAction( Rect(), "" );
	m->m_Table->Attach( m->m_Action, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), ( TABLE_EXPAND | TABLE_FILL ) );

	// Add action items
	m->m_Action->AddButton( "Cancel", (Image*) LoadImage( "cancel.png" ), new Message( _Private::MSG_CANCEL ) , "Cancel the tool and return to main toolbox " );
	m->m_Action->AddButton( "Radius", (Image*) LoadImage( "circle.png" ), new Message( _Private::MSG_RADIUS ), "Describe a circle using radius " );

	State::SetToolbox( this );
}

CircleState2 :: ~CircleState2()
{
	delete m;
}

void CircleState2 :: GotFocus( Canvas* canvas, Document* doc, Point coord )
{
	coord = doc->MakeSnap( coord );
	m->m_DiameterValue->SetValue( m->_CalculateDiameter( GetTool()->GetPropertyAsPoint( "center_point" ), coord ) * 2 );
}

void CircleState2 :: LostFocus( Canvas* canvas, Document* doc, Point coord )
{
	m->m_DiameterValue->ClearValue();
}
 
void CircleState2 :: MouseMove( Canvas* canvas, Document* doc, Point coord )
{
	coord = doc->MakeSnap( coord );

	canvas->SetStyle( doc->GetCurrentStyle() );

	// Calculate radius
	float r = m->_CalculateDiameter( GetTool()->GetPropertyAsPoint( "center_point" ), coord );
	m->m_DiameterValue->SetValue( r * 2 );
	canvas->DrawCircle( GetTool()->GetPropertyAsPoint( "center_point" ), r, true );
}

String CircleState2 :: GetStatusText()
{
	return String( "Select diameter of the circle" );
}

void CircleState2 :: PointSelected( Canvas* canvas, Document* doc, Point coord )
{
	coord = doc->MakeSnap( coord );
	m->_CreateObject( GetTool(), canvas, doc, GetTool()->GetPropertyAsPoint( "center_point" ), m->_CalculateDiameter( GetTool()->GetPropertyAsPoint( "center_point" ), coord ) );
}

void CircleState2 :: SetFrame( const Rect& cRect, bool bNotifyServer )
{
	Toolbox::SetFrame( cRect, bNotifyServer );
	m->_Layout( cRect );
}

void CircleState2 :: AllAttached( void )
{
	Toolbox::AllAttached();

	m->m_Action->SetTarget( this );
	m->m_DiameterValue->SetTarget( this );
}

void CircleState2 :: HandleMessage( Canvas* canvas, Document* doc, Message* pcMessage )
{
	switch( pcMessage->GetCode() )
	{	
		case _Private::MSG_CANCEL:		
			SetMainToolbox();
			break;

		case _Private::MSG_RADIUS:
		{
			GetTool()->SetState( 1 );
			break;
		}

		case _Private::MSG_POINT:
		{
			float p;

			if( pcMessage->FindFloat( "value", &p ) == EOK )
				m->_CreateObject( GetTool(), canvas, doc, GetTool()->GetPropertyAsPoint( "center_point" ), p / 2 );
			break;
		}
	}
}
