// oCADis (C)opyright 2007 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#ifndef __DOCUMENT_H_
#define __DOCUMENT_H_

#include <fstream>
#include <vector>

#include <util/locker.h>

#include "layer.h"
#include "object.h"
#include "style.h"
#include "undo.h"
#include "transform.h"
#include "pattern.h"

class Document : public os::Locker, public Undo
{
public:
	enum OrthoMode { OrthoNone = 0, OrthoBoth = 1, OrthoHoriz, OrthoVert };

	#define OSNAP_NONE		0x00
	#define OSNAP_ENDPOINT	0x01
	#define OSNAP_MIDPOINT	0x02
	#define OSNAP_CENPOINT	0x04
	#define OSNAP_NEAPOINT	0x08
	#define OSNAP_QUAPOINT	0x10

	Document();
	~Document();

	void Check();

	void SetCurrentLayer( Layer* layer );
	Layer* GetCurrentLayer() { return m_CurrentLayer; };

	void AddLayer( Layer* layer );
	void AddObject( Object* obj );
	void AddObject( Layer* layer, Object* obj );

	void Redraw( Canvas* canvas );

	int GetLayerCount()  { return m_ListOfLayers.size(); };
	Layer* GetLayer( int no ) { return m_ListOfLayers.at( no ); };
	void DeleteLayer( int no );
	void SwapLayers( int x, int y );
	bool LayerExists( Layer* layer );

	void AddPattern( Pattern* pattern ) { m_ListOfPatterns.push_back( pattern ); };
	int GetPatternCount() { return m_ListOfPatterns.size(); };
	Pattern* GetPattern( os::String name );
	Pattern* GetPattern( int no ) { return m_ListOfPatterns.at( no ); };

	void SetModification( bool save = true );
	bool IsModified() { return m_NeedSave; };

	os::String GetTitle();
	void SetTitle( os::String title ) { m_Path = title; };

	Style& GetCurrentStyle() { m_CurrentStyle.SetLayer( m_CurrentLayer ); return m_CurrentStyle; };
	void SetCurrentStyle( Style& style ) { m_CurrentStyle = style; m_CurrentStyle.SetLayer( m_CurrentLayer ); };

	Object* HitCheck( os::Point coord, double sens );
	void HitCheck( os::Rect box, std::vector< Object* >& hits );

	void RemoveObject( Object* obj );
	void ReplaceObject( Object* old_obj, Object* new_obj );

	void SelectObject( Object* obj );
	void DeselectObject( Object* obj );
	void SelectAllObjects();
	void DeselectAllObjects( Canvas* canvas = NULL );

	void TransformAllObjects( Transform& trans );	
	os::Rect GetBoundingBox();

	void SetContinuePoint( os::Point p, double tangent );
	void SetContinuePoint( os::Point center, os::Point endpoint );
	bool GetContinuePoint( os::Point& p, double& tangent );

	os::Point MakeSnap( os::Point p1 );
	os::Point MakeSnap( os::Point p1, os::Point p2 );

	void SetSnapActive( bool active ) { m_SnapActive = active; };
	bool GetSnapActive() { return m_SnapActive; };

	void SetSnapOrigo( os::Point origo ) { m_SnapOrigo = origo; };
	os::Point GetSnapOrigo() { return m_SnapOrigo; };
	void SetSnapDistance( os::Point dist ) { m_SnapDistance = dist; };
	os::Point GetSnapDistance() { return m_SnapDistance; };
	void SetSnapAngle( double angle ) { m_SnapAngle = angle; };
	double GetSnapAngle() { return m_SnapAngle; };

	void SetGridActive( bool active ) { m_GridActive = active; };
	bool GetGridActive() { return m_GridActive; };
	void SetGridSkip( os::Point skip ) { m_GridSkip = skip; };
	os::Point GetGridSkip() { return m_GridSkip; };

	void SetOrthoMode( enum OrthoMode mode ) { m_OrthoMode = mode; };
	enum OrthoMode GetOrthoMode() { return m_OrthoMode; };

	void SetOSnapMode( int mode ) { m_FoundOSnap = false; m_OSnap = mode; };
	int GetOSnapMode() { return m_OSnap; };

	bool IsValidOSnap() { return m_FoundOSnap; };
	os::Point ValidOSnapPoint() { return m_OsnapPoint; };

	int GetNoSelectedObjects() { return (int) m_ListOfSelectedObjects.size(); };
	Object* GetSelectedObject( int no ) { return m_ListOfSelectedObjects[ no ]; };
	os::Rect GetSelectedBoundingBox();

	void SetPan( os::Point p ) { m_Pan = p; };
	os::Point GetPan() { return m_Pan; };
	void SetZoom( double factor ) { m_ZoomFactor = factor; };
	double GetZoom() { return m_ZoomFactor; };

private:
	os::Point _OSnap( os::Point p );

	os::String m_Path;  // Where the document is saved

	bool m_NeedSave;  // Set to true if there as been modifications to the drawing

	std::vector< Layer* > m_ListOfLayers;

	std::vector< Object* > m_ListOfSelectedObjects;

	std::vector< Pattern* > m_ListOfPatterns;

	// Data for snap, grid and ortho
	bool m_SnapActive;
	os::Point m_SnapOrigo;
	os::Point m_SnapDistance;
	double m_SnapAngle;

	bool m_GridActive;
	os::Point m_GridSkip;

	enum OrthoMode m_OrthoMode;
	
	// Object snap mode
	int m_OSnap;
	// Save the found OSnap item for drawing
	os::Point m_OsnapPoint;
	bool m_FoundOSnap;

	// Save point for arc and line
	bool m_ContinuePointAvailable;	
	os::Point m_ContinuePoint;
	double m_ContinueTangent;
	
	// Zooming
	os::Point m_Pan;
	float m_ZoomFactor;

	// Current layer
	Layer* m_CurrentLayer;

	// Current style
	Style m_CurrentStyle;
};

#endif
