// oCADis (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "canvasbitmap.h"

#include <agg_path_storage.h>
#include <agg_conv_stroke.h>
#include <agg_conv_curve.h>
#include <agg_conv_dash.h>
#include <agg_conv_transform.h>
#include <agg_conv_clip_polygon.h>
#include <agg_ellipse.h>
#include <agg_gsv_text.h>

using namespace os;

CanvasBitmap :: CanvasBitmap( int width, int height ) : Bitmap( width, height, CS_RGBA32, Bitmap::SHARE_FRAMEBUFFER )
{
	// Create AGG rendering buffer
	m_RenderBuffer = new agg::rendering_buffer();
    m_RenderBuffer->attach( LockRaster(),
							(uint32) width,
							(uint32) height,
							GetBytesPerRow() );

	// Create pixelformat
	m_PixelFormat = new agg::pixfmt_rgba32( *m_RenderBuffer );

	// Create base renderer
	m_RendererBase = new renderer_base( *m_PixelFormat );

	m_Renderer = new renderer_type( *m_RendererBase );

	m_Rasterizer = new rasterizer_type();

	m_Scanline = new scanline_type;

	m_UnpackedScanline = new scanline_unpacked_type();
	m_PackedScanline = new scanline_packed_type();

    // clear out the bitmap with some background color
	m_RendererBase->clear( agg::rgba( 1.0f, 1.0f, 1.0f ) );
}

CanvasBitmap :: ~CanvasBitmap()
{
	UnlockRaster();

	delete m_RenderBuffer;
	delete m_PixelFormat;
	delete m_RendererBase;
	delete m_Renderer;
	delete m_Rasterizer;
	delete m_Scanline;
	delete m_UnpackedScanline;
	delete m_PackedScanline;
}

void CanvasBitmap :: SetWidth( double width )
{
	m_Width = width;
}

double CanvasBitmap :: GetWidth()
{
	return m_Width;
}

void CanvasBitmap :: SetColour( double r, double g, double b )
{
	m_Colour = agg::rgba( b, g, r, 1.0f );		// FIXME: Why are r and b backwards?
}

void CanvasBitmap :: GetColour( double& r, double& g, double& b)
{
	r = g = b = 0.0f;  // FIXME!
}

void CanvasBitmap :: SetPattern( double* dashes, int num_dashes, double offset )
{
	m_Pattern.clear();

	for( int i = 0 ; i < num_dashes ; i++ )
		m_Pattern.push_back( dashes[ i ] );

	m_Offset = offset;
}

void CanvasBitmap :: SetAntiAlias( bool use )
{
}

void CanvasBitmap :: DrawLine( Point p1, Point p2 )
{
	// in case it's the same pixel, we offset differently to have something drawn
	if( p1 == p2 )
	{
		p1.x += 0.25;
		p1.y += 0.25;
		p2.x += 0.75;
		p2.y += 0.75;
	} 
	else 
	{
		// do the pixel center offset here
		p1.x += 0.5;
		p1.y += 0.5;
		p2.x += 0.5;
		p2.y += 0.5;
	}

	agg::path_storage path;
	path.move_to( p1.x, p1.y );
	path.line_to( p2.x, p2.y );

	_StrokePath( path );
}

void CanvasBitmap :: DrawArc( Point center, double radius, double start_angle, double end_angle )
{
	double span;

	// Make sure the angles are within 0 <= angle <= 2 * M_PI
	while( start_angle > 2.0f * M_PI )  start_angle -= 2.0f * M_PI;
	while( end_angle > 2.0f * M_PI )  end_angle -= 2.0f * M_PI;

	while( start_angle < 0.0f )  start_angle += 2.0f * M_PI;
	while( end_angle < 0.0f )  end_angle += 2.0f * M_PI;

	// Calculate size of the arc
	if( start_angle < end_angle )
		span = end_angle - start_angle;
	else
		span = end_angle + ( M_PI * 2.0f - start_angle );

	agg::bezier_arc arc(center.x, center.y, radius, radius, start_angle, span);
	agg::conv_curve< agg::bezier_arc > path(arc);
//	path.approximation_scale(2.0);

	_StrokePath( path );
}

void CanvasBitmap :: DrawCircle( Point center, double radius )
{
	int divisions = (int) ( ( radius * 2 + 2 * m_Width ) * M_PI / 2 );

	if( divisions < 12 )
		divisions = 12;
	if( divisions > 4096 )
		divisions = 4096;

	agg::ellipse path( center.x, center.y, radius, radius, divisions );

	_StrokePath( path );
}

void CanvasBitmap :: DrawRectangle( Point offset, Point size )
{
}

void CanvasBitmap :: Fill()
{
	m_RendererBase->clear( m_Colour );
}

void CanvasBitmap :: DrawBitmap( CanvasBitmap* bsrc, Rect size )
{
	// Copy the background to the bitmap
	int w = (int) size.Width();
	int h = (int) size.Height();
	int bw = (int) GetBounds().Width() + 1;

	int offset = (int) ( bw * (int) size.top + (int) size.left );
	uint32* src = (uint32*) ( bsrc->LockRaster() )  + offset;
	uint32* dst = (uint32*) ( LockRaster() ) + offset;

	bw = bw - w;

	for( int i = 0 ; i < h ; i++ )
	{
		for( int x = 0 ; x < w ; x++)
			*dst++ = *src++;

		src += bw;
		dst += bw;
	}
}

void CanvasBitmap :: DrawPixel( Point p, double r, double g, double b, double alpha )
{
	// Clipping
	if( p.x >= 0 && p.x < GetBounds().Width() && p.y >= 0 && p.y < GetBounds().Height() )
	{
		int bw = (int) GetBounds().Width() + 1;
		int offset = (int) ( bw * (int) ( p.y + 0.5f ) + (int) ( p.x + 0.5f ) );
		uint32* dst = (uint32*) ( LockRaster() ) + offset; 

		*dst = (uint32) ( ( (uint32) r * 255 ) << 24 ) | ( ( (uint32) g * 255 ) << 16 ) | ( ( (uint32) b * 255 ) << 8 ) | ( (uint32) alpha * 255 );
	}
}

///////////////////////////////////////////////////////////////////////////////////////////
//
//  P R I V A T E
//
///////////////////////////////////////////////////////////////////////////////////////////

template<class VertexSource>
void CanvasBitmap :: _StrokePath( VertexSource& path )
{
	agg::conv_dash< VertexSource > dash( path );

	// Set up line pattern
	if( m_Pattern.size() > 0 )
	{
		for( int i = 0 ; i < (int) m_Pattern.size() ; i += 2 )
		{
			if( i < (int) m_Pattern.size() - 1 )
				dash.add_dash( m_Pattern[ i ], m_Pattern[ i + 1 ] );
			else
				dash.add_dash( m_Pattern[ i ], 0.0f );
		}
		dash.dash_start( m_Offset );
	}
	else
		dash.add_dash( 1.0f, 0.0f );	// Solid line

	// Set colour
	m_Renderer->color( m_Colour );

	m_Rasterizer->reset();

	// Stroke the path
	agg::conv_stroke< agg::conv_dash< VertexSource > > stroke(dash);

	// Set the size of the pen
	stroke.width( m_Width );

	// special case line width = 1 with square caps
//	if( m_Width == 1.0f && m_LineCapMode == agg::butt_cap )
//		stroke.line_cap( agg::square_cap );
//	else
//		stroke.line_cap(agg::butt_cap );

//	stroke.line_join(agg::miter_join );

	m_Rasterizer->add_path( stroke );

	agg::render_scanlines( *m_Rasterizer, *m_Scanline, *m_Renderer );

}

