// oCADis (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <cassert>

#include "canvas.h"
#include "appwindow.h"
#include "main.h"
#include "cadmath.h"

using namespace os;
using namespace std;

Canvas :: Canvas( AppWindow* parent, const Rect& cFrame, uint32 nResizeMask ) : View( cFrame, "canvas", nResizeMask, WID_WILL_DRAW )
{
	m_CurrentBitmapView = NULL;
	m_DocumentBitmapView = NULL;

	m_Parent = parent;

	m_SnapDrawn = false;
	m_SnapMouse = false;

	m_Pan = Point( 0, 0 );
	m_ZoomFactor = 1.0f;

	m_WantsToPan = false;
	m_MousePressed = false;
	m_MouseSelecting = false;
}

Canvas :: ~Canvas()
{
	delete m_CurrentBitmapView;
	delete m_DocumentBitmapView;
}

Point Canvas :: GetPreferredSize( bool bLargest ) const
{
	if( bLargest )
		return Point( 10000, 10000 );

	return Point( 1.0f, 1.0f );
}

void Canvas :: FrameSized( const Point& delta )
{
	View::FrameSized( delta );

	_CreateBitmap();

	if( oCADisApp::GetAppWindow() != NULL )
		Refresh();
}

void Canvas :: AttachedToWindow()
{
	View::AttachedToWindow();
	_CreateBitmap();
}

void Canvas :: Paint( const Rect& cUpdate )
{		
	// Draw bitmap
	SetDrawingMode( DM_COPY );
	Rect u = cUpdate;
	//u.bottom -= 1.0f;
	//u.right -= 1.0f;
	DrawBitmap( m_CurrentBitmapView, u, u );

	// Render selection rectangle */
	if( m_MouseSelecting )
	{
		Color32_s sColor = get_default_color( COL_ICON_SELECTED );
		sColor.alpha = 128;
			
		Rect cSelect( m_SelectStart, m_LastSelectPosition );

		float vTemp;
		if( cSelect.left > cSelect.right )
		{
			vTemp = cSelect.right;
			cSelect.right = cSelect.left;
			cSelect.left = vTemp;
		}
		if( cSelect.top > cSelect.bottom )
		{
			vTemp = cSelect.bottom;
			cSelect.bottom = cSelect.top;
			cSelect.top = vTemp;
		}

		SetDrawingMode( os::DM_COPY );
		SetFgColor( sColor );
		View::DrawLine( os::Point( cSelect.left, cSelect.top ), os::Point( cSelect.right, cSelect.top ) );
		View::DrawLine( os::Point( cSelect.right, cSelect.top ), os::Point( cSelect.right, cSelect.bottom ) );
		View::DrawLine( os::Point( cSelect.left, cSelect.bottom ), os::Point( cSelect.right, cSelect.bottom ) );
		View::DrawLine( os::Point( cSelect.left, cSelect.top ), os::Point( cSelect.left, cSelect.bottom ) );

		SetDrawingMode( os::DM_BLEND );
		cSelect.Resize( 1, 1, -1, -1 );
		FillRect( cSelect ); 
	}

	// Draw OSnap is needed and mouse pointer is within the area
	Document* doc = GET_DOCUMENT();
	if( m_SnapMouse && doc != NULL && doc->GetOSnapMode() != OSNAP_NONE && doc->IsValidOSnap())
	{
		m_SnapDrawn = true;

		Point p = WorldToScreen( doc->ValidOSnapPoint() );
		Rect r( p.x - 5, p.y - 5, p.x + 5, p.y + 5 );

		m_SnapPosition = r;

		Color32_s sColor( 0xBA, 0xE3, 0xC7, 128);
			
		SetDrawingMode( os::DM_COPY );
		SetFgColor( sColor );
		View::DrawLine( os::Point( r.left, r.top ), os::Point( r.right, r.top ) );
		View::DrawLine( os::Point( r.right, r.top ), os::Point( r.right, r.bottom ) );
		View::DrawLine( os::Point( r.left, r.bottom ), os::Point( r.right, r.bottom ) );
		View::DrawLine( os::Point( r.left, r.top ), os::Point( r.left, r.bottom ) );

		SetDrawingMode( os::DM_BLEND );
		r.Resize( 1, 1, -1, -1 );
		FillRect( r );		
	}
}

void Canvas :: WheelMoved( const Point& delta )
{
	Point orig_mouse_pos;
	Point mouse_pos;
	uint32 button;

	if( delta.y == 0.0f )
		return;

	Undraw();
	UndrawOSnap();

	// Calucate new pan origo and zoom factor
	GetMouse( &orig_mouse_pos, &button );
	mouse_pos = ScreenToWorld( orig_mouse_pos );
	orig_mouse_pos.y = GetBounds().Height() - orig_mouse_pos.y ;

	if( delta.y < 0 )
		m_ZoomFactor *= 0.9;		
	else
		m_ZoomFactor *= 1.1;

	m_Pan = - mouse_pos;
	m_Pan += Point( orig_mouse_pos.x / m_ZoomFactor, orig_mouse_pos.y / m_ZoomFactor );

	Refresh();
}

void Canvas :: MouseDown( const Point& cPosition, uint32 nButtons )
{
	m_MousePressed = true;
	m_MouseSelecting = false;
	m_WantsMouseBox = false;
	m_LastSelectPosition = m_SelectStart = cPosition;
	m_WantsToPan = false;
	State* state = m_Parent->GetCurrentState();

	// Check with current tool if it wants a mouse box drawn
	if( nButtons == 0x01 && state != NULL && state->AllowMouseBox( this, GET_DOCUMENT(), ScreenToWorld( cPosition ) ) )
	{
		m_WantsMouseBox = true;
	}
		
	if( nButtons == 0x03 )
	{
		m_WantsToPan = true;
		m_PanStart = ScreenToWorld( cPosition );
	}
}

void Canvas :: MouseMove( const Point& cPos, int nCode, uint32 nButtons, Message* pcData )
{
	State* state = m_Parent->GetCurrentState();

	Point cPosition = ScreenToWorld( cPos );

	m_SnapMouse = true;

	if( nCode == MOUSE_ENTERED )
	{
		if( state != NULL )
			state->GotFocus( this, GET_DOCUMENT(), cPosition );

		m_Parent->SetCoord( cPosition );
	
	}
	else if( nCode == MOUSE_EXITED )
	{
		Undraw();

		if( state != NULL )
			state->LostFocus( this, GET_DOCUMENT(), cPosition );

		m_Parent->ClearCoord();

		m_MousePressed = false;
		m_SnapMouse = false;
		UndrawOSnap();

		// Erase the selection box
		if( m_MouseSelecting )
		{
			m_MouseSelecting = false;
			
			Invalidate();
			Flush();

		}

		return;
	}

	if( state != NULL && !m_WantsToPan )
	{
		Undraw();
		state->MouseMove( this, GET_DOCUMENT(), cPosition );
	}

	if( m_MousePressed && m_WantsMouseBox && nButtons == 0x01 )
	{
		m_MouseSelecting = true;
			
		// Draw new frame
		Invalidate( CadMath::SortRect( Point ( m_LastSelectPosition.x, m_SelectStart.y ), Point( cPos.x, m_LastSelectPosition.y ) ) );
		Invalidate( CadMath::SortRect( Point ( m_SelectStart.x, m_LastSelectPosition.y ), Point( cPos.x, cPos.y ) ) );
		Flush();

		m_LastSelectPosition = cPos;
	}

	if( m_WantsToPan )
	{
		Undraw();

		m_Pan -= ( m_PanStart - cPosition 	);
		Refresh();
	}

	m_Parent->SetCoord( cPosition );
}

void Canvas :: MouseUp( const Point& cPosition, uint32 nButtons, Message* pcData )
{
	// Make sure that the mouse also was pressed within the view
	if( !m_MousePressed )
		return;

	State* state = m_Parent->GetCurrentState();
	if( state != NULL && nButtons == 0x01 && !m_WantsMouseBox ) 	
		state->PointSelected( this, GET_DOCUMENT(), ScreenToWorld( cPosition ) );

	if( state != NULL && nButtons == 0x01 && m_WantsMouseBox ) 	
	{
		state->MouseBox( this, GET_DOCUMENT(), CadMath::SortRect( ScreenToWorld( m_SelectStart ), ScreenToWorld( cPosition ) ) );

		// Remove mouse box
		Invalidate();
		Flush();		
	}

	// Ends the panning
	if( nButtons == 0x03 )
	{
		m_WantsToPan = false;
		PrepareForXORMode();
	}

	m_WantsMouseBox = false;
	m_MousePressed = false;
	m_MouseSelecting = false;
}

Point Canvas :: ScreenToWorld( Point p )
{
	Point ret( p.x / m_ZoomFactor, ( GetBounds().Height() - p.y ) / m_ZoomFactor );
	ret -= m_Pan;

	return ret;
}

Point Canvas :: WorldToScreen( Point p )
{
	Point ret = p + m_Pan;
	ret.x *= m_ZoomFactor;
	ret.y *= m_ZoomFactor;

	ret.y = GetBounds().Height() - ret.y;
	return ret;
}

float Canvas :: WorldToScreen( float p )
{
	return p * m_ZoomFactor;
}

float Canvas :: ScreenToWorld( float p )
{
	return p / m_ZoomFactor;
}

void Canvas :: DrawSelectedObjects( Transform& transform )
{
	for( int i = 0 ; i < GET_DOCUMENT()->GetNoSelectedObjects() ; i++ )
	{
		Object* obj = GET_DOCUMENT()->GetSelectedObject( i );
		obj->Draw( this, transform, true, true);
	}
}

void Canvas :: DrawLine( Point p1, Point p2, bool xor_mode, bool update_server )
{
	Point sp1 = WorldToScreen( p1 );
	Point sp2 = WorldToScreen( p2 );

	Rect invalid;
	invalid = CadMath::SortRect( sp1, sp2 );
	invalid.Resize( -m_CurrentBitmapView->GetWidth() * 2.0f, -m_CurrentBitmapView->GetWidth() * 2.0f, m_CurrentBitmapView->GetWidth() * 2.0f, m_CurrentBitmapView->GetWidth() * 2.0f );
	invalid.Resize( -3, -3, 3, 3 );

	if( xor_mode )
		_SaveBackground( invalid );

	m_CurrentBitmapView->DrawLine( sp1, sp2 );

	if( update_server )
		Invalidate( invalid );
}

void Canvas :: DrawRectangle( Rect r, bool xor_mode, bool update_server )
{
	Point sp1 = Point( r.left, r.top );
	Point sp2 = Point( r.right, r.top );
	Point sp3 = Point( r.right, r.bottom );
	Point sp4 = Point( r.left, r.bottom );

	DrawLine( sp1, sp2, xor_mode, update_server );
	DrawLine( sp2, sp3, xor_mode, update_server );
	DrawLine( sp3, sp4, xor_mode, update_server );
	DrawLine( sp4, sp1, xor_mode, update_server );
}

void Canvas :: DrawCircle( Point center, float radius, bool xor_mode, bool update_server )
{
	Point scenter = WorldToScreen( center );
	float sradius = WorldToScreen( radius );

	Rect invalid( Point( scenter.x - sradius , scenter.y - sradius ), Point( scenter.x + sradius, scenter.y + sradius ) );
	invalid.Resize( -m_CurrentBitmapView->GetWidth() * 2.0f, -m_CurrentBitmapView->GetWidth() * 2.0f, m_CurrentBitmapView->GetWidth() * 4.0f, m_CurrentBitmapView->GetWidth() * 4.0f );
	invalid.Resize( -3, -3, 3, 3 );

	if( xor_mode )
		_SaveBackground( invalid );

	m_CurrentBitmapView->DrawCircle( scenter, sradius );

	if( update_server )
		Invalidate( invalid );
}

void Canvas :: DrawArc( Point center, float radius, float start_angle, float end_angle, bool xor_mode, bool update_server )
{
	Point scenter = WorldToScreen( center );
	float sradius = WorldToScreen( radius );

	double t = 2*M_PI - end_angle;
	end_angle = 2*M_PI - start_angle;
	start_angle = t;

	Rect invalid( Point( scenter.x - sradius , scenter.y - sradius ), Point( scenter.x + sradius, scenter.y + sradius ) );
	invalid.Resize( -m_CurrentBitmapView->GetWidth() * 2.0f, -m_CurrentBitmapView->GetWidth() * 2.0f, m_CurrentBitmapView->GetWidth() * 3.0f, m_CurrentBitmapView->GetWidth() * 3.0f );	
	invalid.Resize( -3, -3, 3, 3 );

	if( xor_mode )
		_SaveBackground( invalid );

	m_CurrentBitmapView->DrawArc( scenter, sradius, start_angle, end_angle );

	if( update_server )
		Invalidate( invalid );
}

void Canvas :: DrawGrid( Document* doc )
{
	Point grid_distance;
	Point v1, v2;
	Rect bound_box;
	Point p;
	double rot;
	
	rot = doc->GetSnapAngle();

	// Distance between each grid point
	grid_distance = doc->GetSnapDistance() * doc->GetGridSkip();

	// Find world coord for the view window
	v1 = ScreenToWorld( Point( 0.0f, 0.0f ) );
	v2 = ScreenToWorld( Point( GetBounds().Width(), GetBounds().Height() ) );

	// Rotate world coord to snap angle
	v1 = CadMath::RotatePoint( rot, Point( 0.0f, 0.0f ) , v1 - doc->GetSnapOrigo() );
	v2 = CadMath::RotatePoint( rot, Point( 0.0f, 0.0f ), v2 - doc->GetSnapOrigo() );

	// Calculate bounding box
	bound_box.left = CadMath::min( v1.x, v2.x );
	bound_box.top = CadMath::min( v1.y, v2.y );
	bound_box.right = CadMath::max( v1.x, v2.x );
	bound_box.bottom = CadMath::max( v1.y, v2.y );

	// Calculate first grid point
	p = Point( 0.0f, 0.0f );

	while( p.x < bound_box.left )
		p.x += grid_distance.x;

	while( p.x > bound_box.left )
		p.x -= grid_distance.x;

	while( p.y < bound_box.top )
		p.y += grid_distance.y;

	while( p.y > bound_box.top )
		p.y -= grid_distance.y;

	// Draw all grid points
	for( ; p.y < bound_box.bottom ; p.y += grid_distance.y )
	{
		for( double x = p.x; x < bound_box.right ; x += grid_distance.x )
		{
			// Rotate point back
			Point t = CadMath::RotatePoint( -rot, doc->GetSnapOrigo(), Point( x, p.y ) );
			t = WorldToScreen( t );

			if( GET_DOCUMENT()->GetGridActive() )
				m_CurrentBitmapView->DrawPixel( t, 0.0f, 0.0f, 0.0f );
			else
				m_CurrentBitmapView->DrawPixel( t, 1.0f, 1.0f, 1.0f );
		}
	}
}

void Canvas :: SetColour( double r, double g, double b )
{
	m_CurrentBitmapView->SetColour( r, g, b );
}

void Canvas :: SetLineWidth( double width )
{
	// If width = 0.0f use smallest possible with
	if( fabs( width ) < FloatEllipson )
		m_CurrentBitmapView->SetWidth( 1.0f );
	else
		// Convert to screen
		m_CurrentBitmapView->SetWidth( (double) WorldToScreen( width ) );
}

void Canvas :: SetAntiAlias( bool use )
{
	m_CurrentBitmapView->SetAntiAlias( use );	
}

void Canvas :: SetStyle( Style& style )
{
	// Set linewidth
	SetLineWidth( style.GetLineWidth() );

	// Set colour
	double r, b, g;
	style.GetColour( r, g, b );
	SetColour( r, b, g );

	// Set pattern
	Pattern* pattern = style.GetPattern();
	assert( pattern != NULL );
	if( pattern->IsSolid() )
		m_CurrentBitmapView->SetPattern( NULL, 0, 0 );
	else
	{
		double* l = new double[ pattern->GetDashesCount() ];
		if( pattern->IsZoomingAllowed() )
		{
			for( int i = 0 ; i < pattern->GetDashesCount() ; i++ )
			{
				l[ i ] = pattern->GetDash( i ) * m_ZoomFactor;
				if( l[ i ] < 1.0f ) l[ i ] = 1.0f;
			}
		
			m_CurrentBitmapView->SetPattern( l, pattern->GetDashesCount(), pattern->GetOffset() * m_ZoomFactor );
		}
		else
		{
			for( int i = 0 ; i < pattern->GetDashesCount() ; i++ )
				l[ i ] = pattern->GetDash( i );
		
			m_CurrentBitmapView->SetPattern( l, pattern->GetDashesCount(), pattern->GetOffset() );
		}

		delete l;
	}
}

void Canvas :: SetDefaultStyle()
{
	m_CurrentBitmapView->SetWidth( 1.0f );
	m_CurrentBitmapView->SetColour( 0.0f, 0.0f, 0.0f );
	m_CurrentBitmapView->SetPattern( NULL, 0, 0 );
}

void Canvas :: Refresh()
{
	if( m_CurrentBitmapView == NULL )
		return;

	// Clear background
	m_CurrentBitmapView->SetColour( 1.0f, 1.0f, 1.0f );
	m_CurrentBitmapView->Fill();
 
	// Make sure that we have a document
	if( GET_DOCUMENT() != NULL )
	{
		Rect bound_box = CadMath::SortRect( ScreenToWorld( Point( 0.0f, 0.0f ) ), 
								   			ScreenToWorld( Point( GetBounds().Width(), GetBounds().Height() ) ) );

		// For each layer
		for( int i = 0 ; i < GET_DOCUMENT()->GetLayerCount() ; i ++ )
		{
			Layer* layer = GET_DOCUMENT()->GetLayer( i );

			// Draw layer only if it is visible
			if( layer->IsVisible() )
			{
				// Draw all objects in the layer
				for( int j = 0 ; j < layer->GetObjectCount();  j++ )
				{
					Object* object = layer->GetObject( j ) ;

					// Draw object only if it is visible
					if( bound_box.DoIntersect( object->GetBounds() )  )
					{
						object->Draw( this, false, false );

						// Uncomment next line to draw the bounding boxes of the objects
						// DrawRectangle( object->GetBounds(), false, false );
					}
				}	
			}
		}
	}

	// Check if a grid shall be drawn
	if( GET_DOCUMENT()->GetGridActive() )
		DrawGrid( GET_DOCUMENT() );

	// Update background image
	PrepareForXORMode();

	// Complete redraw
	Invalidate();
	Sync();
}

// Returns the size of the canvas in world coordinates
Rect Canvas :: GetViewPortSize()
{
	Rect r = CadMath::SortRect( ScreenToWorld( Point( 0, 0 ) ), ScreenToWorld( Point( GetBounds().Width(), GetBounds().Height() ) ) );
	
	return r;
}

String Canvas :: Save()
{
	String v;
	v.Format( "V, %f, %f, %f\n", m_Pan.x, m_Pan.y, m_ZoomFactor );

	return v;
}


void Canvas :: ZoomBox( Rect bounds )
{
	// Set pan

	// Calculate bounding box
	Point p1 = ScreenToWorld( Point( 0.0f, 0.0f ) );
	Point p2 = ScreenToWorld( Point( GetBounds().Width(), GetBounds().Height() ) );
	Rect current_view = CadMath::SortRect( p1, p2 );

	double zoom_x = 480.0f;
	double zoom_y = 640.0f;

	// Get zoom in X and zoom in Y:
    if( bounds.Width() > 1.0e-6 )
	{
        zoom_x = current_view.Width() / bounds.Width();
    }

    if( bounds.Height() > 1.0e-6 )
	{
        zoom_y = current_view.Height() / bounds.Height();
    }

	// Keep ascept ratio
	if( zoom_x < zoom_y )
		zoom_x = zoom_y = ( current_view.Width() - 2.0f * ScreenToWorld( 5.0f ) ) / ( current_view.Width() * zoom_x );
	else
		zoom_x = zoom_y = ( current_view.Height() - 2.0f * ScreenToWorld( 5.0f ) ) / ( current_view.Height() * zoom_y );

	if( zoom_x < 0.0f )
		zoom_x *= -1.0f; 
	if( zoom_y < 0.0f )
		zoom_y *= -1.0f; 

//	m_ZoomFactor /= zoom_x;

	m_Pan.x = bounds.left + bounds.Width() / 2;
	m_Pan.y = -bounds.bottom + bounds.Height() / 2;

/*

	Point p1 = ScreenToWorld( Point( GetBounds().Width(), GetBounds().Height() ) );
	Point p2 = ScreenToWorld( Point( 0, 0 ) );
	p1 = p1 - p2;
	p1.x = p1.x / bounds.Width();
	p1.y = p1.y / bounds.Height();
	double scale;

	if( p1.x < p1.y )
		scale = p1.x * 0.95f;
	else
		scale = p1.y * 0.95f;

	m_ZoomFactor *= scale; */
}

void Canvas :: PrepareForXORMode()
{
	int size = (int) m_CurrentBitmapView->GetBounds().Width() * (int) m_CurrentBitmapView->GetBounds().Height();
	uint32* src = (uint32*) m_CurrentBitmapView->LockRaster();
	uint32* dst = (uint32*) m_DocumentBitmapView->LockRaster();

	// Copy current view to the document bitmap
	for( int i = 0 ; i < size ; i++ )
		*dst++ = *src++;
}

void Canvas :: UndrawOSnap()
{
	m_SnapMouse = false;
	if( m_SnapDrawn )
	{
		Invalidate( m_SnapPosition );
		Flush();
		m_SnapDrawn = false;
	}
}

///////////////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E   M E T H O D S
//
///////////////////////////////////////////////////////////////////////////////////////
void Canvas :: _CreateBitmap()
{
	delete m_CurrentBitmapView;
	delete m_DocumentBitmapView;

	int w = (int) GetFrame().Width() + 1;
	int h = (int) GetFrame().Height() + 1;

	// Make sure the size is realistic
	if( w > 0 && h > 0 && w < 5000 && h < 5000 )
	{		
	 	m_CurrentBitmapView = new CanvasBitmap( w, h );
		m_DocumentBitmapView = new CanvasBitmap( w, h );
	}
	else
		m_CurrentBitmapView = m_DocumentBitmapView = NULL;
}

void Canvas :: _SaveBackground( Rect size )
{
	// Make some clipping, ensuring that we are always inside the bitmap
	if( size.left < 0 )
		size.left = 0;
	if( size.top < 0 )
		size.top = 0;
	if( size.right > m_CurrentBitmapView->GetBounds().Width() )
		size.right = m_CurrentBitmapView->GetBounds().Width();
	if( size.bottom > m_CurrentBitmapView->GetBounds().Height() )
		size.bottom = m_CurrentBitmapView->GetBounds().Height();

	if( size.Width() != 0 && size.Height() != 0 )
	{
		m_ListofDirtyRects.push_back( size );
	}
}

void Canvas :: _RestoreBackground()
{
	for( int i = m_ListofDirtyRects.size() - 1 ; i >= 0 ; i-- )
	{
		Rect tmp = m_ListofDirtyRects[ i ];

		m_CurrentBitmapView->DrawBitmap( m_DocumentBitmapView, tmp );

		Invalidate( );
	}

	m_ListofDirtyRects.clear();
}

void Canvas :: _DrawGridNegative( Document* doc )
{
}

void Canvas :: _DrawGridPositive( Document* doc )
{

}

