// oCADis (C)opyright 2007 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <gui/button.h>
#include <gui/checkbox.h>
#include <gui/textview.h>
#include <gui/stringview.h>
#include <gui/dropdownmenu.h>

#include "windialogs/windia_modifylayer.h"
#include "windialogs/windia_alert.h"
#include "postoffice.h"
#include "messages.h"
#include "main.h"
#include "cadmath.h"

using namespace os;

///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E
//
///////////////////////////////////////////////////////////////////////////////

class WinDiaModifyLayer :: Private1
{
public:
	Private1(){};

	void _Layout( Rect cFrame )
	{
		cFrame.Resize( 2, 2, -2, -2 );

		// Set button table
		Point tsize = m_ButtonTable->GetPreferredSize( false );
		Rect tframe( cFrame.right - tsize.x, cFrame.bottom - tsize.y, cFrame.right, cFrame.bottom );
		m_ButtonTable->SetFrame( tframe );

		// Set table
		Rect t1frame( cFrame.left, cFrame.top, cFrame.right, tframe.top - 2.0f );
		m_theTable->SetFrame( t1frame );
	}

	bool _Apply()
	{
		if( m_CurrentDocument == NULL )
		{
			WinDiaAlert::ShowErrorAlert( "Edit layer", "No valid document." );
			return false;
		}

		if( m_CurrentLayer == NULL )
		{
			WinDiaAlert::ShowErrorAlert( "Edit layer", "No valid layer selected." );
			return false;
		}

		// Make sure layer is not already deleted
		if( !m_CurrentDocument->LayerExists( m_CurrentLayer ) )
		{
			WinDiaAlert::ShowErrorAlert( "Edit layer", "The layer has already been deleted." );
			return false;
		}

		// Check name
		String name;
		name = m_Name->GetValue().AsString().Strip();

		// Check for empty string
		if( name == "" )
		{
			WinDiaAlert::ShowErrorAlert( "Edit layer", "A layer must have a name." );
			return false;
		}

		// Check for name duplicates
		for( int i = 0 ; i < m_CurrentDocument->GetLayerCount() ; i++ )
		{
			Layer* layer = m_CurrentDocument->GetLayer( i );

			if( layer->GetName() == name && m_CurrentLayer != layer )
			{
				WinDiaAlert::ShowErrorAlert( "Edit layer", "A layer with this name already exists." );
				return false;
			}
		}

		// Extract the other variables
		double linewidth = CadMath :: Strtod( m_Linewidth->GetValue().AsString() );
		int sel = m_Pattern->GetSelection();
		if( sel < 0 )
		{
			WinDiaAlert::ShowErrorAlert( "Edit layer", "No pattern selected." );
			return false;
		}
		Pattern* pattern = m_CurrentDocument->GetPattern( sel );

		// Erase the old layer
		m_CurrentLayer->Erase( GET_CANVAS() );

		// Lets modify our layer
		m_CurrentLayer->SetName( name );
		m_CurrentLayer->SetVisible( m_Visible->GetValue().AsBool() );
		m_CurrentLayer->SetEditable( m_Editable->GetValue().AsBool() );

		Style style = m_CurrentLayer->GetStyle();

		style.SetLineWidth( linewidth );
		style.SetPattern( pattern->GetName() );
		Color32_s col = m_Colour->GetColour();
		style.SetColour( col.red / 255.0f, col.blue / 255.0f, col.green * 255.0f );

		m_CurrentLayer->SetStyle( style );

		Message* msg = new Message( MSG_BROADCAST_MODIFY_LAYER  );
		msg->AddPointer( "document", (void*) m_CurrentDocument );
		msg->AddPointer( "layer", (void*) m_CurrentLayer );
	
		Mail( "AddLayer", msg );

		// Draw layer
		m_CurrentLayer->Redraw( GET_CANVAS() );
		GET_CANVAS()->PrepareForXORMode();

		return true;
	}

	void _UpdatePattern()
	{
		m_Pattern->Clear();
	
		if( m_CurrentDocument == NULL )
			return;

		for( int i = 0 ; i < m_CurrentDocument->GetPatternCount() ; i++ )
		{
			Pattern* pattern = m_CurrentDocument->GetPattern( i );
			m_Pattern->AppendItem( pattern->GetName() );
		}

		m_Pattern->SetSelection(0, false );
	}

	void _SetLayer( Layer* layer )
	{

		m_CurrentLayer = layer;

		if( layer == NULL )
		{
			m_Name->SetValue( "" );
			m_Visible->SetValue( false );
			m_Editable->SetValue( false );
			m_Linewidth->SetValue( "" );
			m_Pattern->SetSelection( 0 );
			m_Colour->SetColour( Color32_s( 0, 0, 0 ) );
			m_Apply->SetEnable( false );
			m_OK->SetEnable( false );
		}
		else
		{
			m_Apply->SetEnable( true );
			m_OK->SetEnable( true );

			String t;

			m_Name->SetValue( layer->GetName() );
			m_Visible->SetValue( layer->IsVisible() );
			m_Editable->SetValue( layer->IsEditable() );

			Style style = layer->GetStyle();
		
			t.Format( "%f", style.GetLineWidth() );
			m_Linewidth->SetValue( t );

			double r, g, b;
			style.GetColour( r, g, b );
			m_Colour->SetColour( Color32_s( (int) ( r * 255.0f ), (int) ( g * 255.0f ), (int) ( b * 255.0f ) ) );

			Pattern* pattern = style.GetPattern();

			for( int i = 0 ; i < m_Pattern->GetItemCount() ; i++ )
			{
				if( m_Pattern->GetItem( i ) == pattern->GetName() )
				{
					m_Pattern->SetSelection( i );
					break;
				}
			}
		}
	}

	enum messages { MSG_OK, MSG_APPLY, MSG_CANCEL, MSG_COLOUR, MSG_PATTERN };

	Document* m_CurrentDocument;
	Layer* m_CurrentLayer;

	os::TableView* m_theTable;

	// Attributes
	ToolboxFrame* m_AttrFrame;
	os::TableView* m_AttrTable;

	os::StringView* m_LabelName;
	os::TextView* m_Name;

	os::CheckBox* m_Visible;
	os::CheckBox* m_Editable;

	// Style
	ToolboxFrame* m_StyleFrame;
	os::TableView* m_StyleTable;

	os::StringView* m_LabelColour;
	ColourButton* m_Colour;

	os::StringView* m_LabelLinewidth;
	os::TextView* m_Linewidth;
	
	os::StringView* m_LabelPattern;
	os::DropdownMenu* m_Pattern;

	// Button bar
	os::TableView* m_ButtonTable;
	os::Button* m_Apply;
	os::Button* m_OK;
	os::Button* m_Cancel;
};

///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S
//
///////////////////////////////////////////////////////////////////////////////
WinDiaModifyLayer :: WinDiaModifyLayer( const Rect& cFrame, Document* current ) : Window( cFrame, "modify_layer", "Edit Layer - oCADis" )
{
	// Create the private class
	m1 = new Private1();

	// Create the mailbox
	AddMailbox( "Document" );

	// Create the table
	m1->m_theTable = new TableView( Rect(), "", 1, 2, false );
	m1->m_theTable->SetRowSpacings( 4 );
	AddChild( m1->m_theTable );
	
	//////////////////////////////////////////////
	// Create the attribute frame
	m1->m_AttrFrame = new ToolboxFrame( Rect(0,0,100,100), "Attributes", "Attributes" );
	m1->m_theTable->Attach( m1->m_AttrFrame, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create attribute table
	m1->m_AttrTable = new TableView( Rect(), "", 2, 3, false );
	m1->m_AttrTable->SetRowSpacings( 4 );
	m1->m_AttrFrame->AddTableView( m1->m_AttrTable );

	m1->m_LabelName = new StringView( Rect(), "", "Name: " );
	m1->m_AttrTable->Attach( m1->m_LabelName, 0, 1, 0, 1, 0, 0 );
	m1->m_Name = new TextView( Rect(), "", "" );
	m1->m_AttrTable->Attach( m1->m_Name, 1, 2, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m1->m_Visible = new CheckBox( Rect(), "", "Layer is visible", NULL );
	m1->m_Visible->SetValue( true );
	m1->m_AttrTable->Attach( m1->m_Visible, 0, 2, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m1->m_Editable = new CheckBox( Rect(), "", "Layer is editable", NULL );
	m1->m_Editable->SetValue( true );
	m1->m_AttrTable->Attach( m1->m_Editable, 0, 2, 2, 3, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	//////////////////////////////////////////////
	// Create the style frame
	m1->m_StyleFrame = new ToolboxFrame( Rect(0,0,100,100), "Style", "Style" );
	m1->m_theTable->Attach( m1->m_StyleFrame, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create style table
	m1->m_StyleTable = new TableView( Rect(), "", 2, 4, false );
	m1->m_StyleTable->SetRowSpacings( 4 );
	m1->m_StyleFrame->AddTableView( m1->m_StyleTable );

	// Create colour
	m1->m_LabelColour = new StringView( Rect(), "", "Colour: " );
	m1->m_StyleTable->Attach( m1->m_LabelColour, 0, 1, 0, 1, 0, 0 );
	m1->m_Colour = new ColourButton( Rect(), new Message( Private1::MSG_COLOUR ) );
	m1->m_StyleTable->Attach( m1->m_Colour, 1, 2, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create linewidth
	m1->m_LabelLinewidth = new StringView( Rect(), "", "Linewidth: " );
	m1->m_StyleTable->Attach( m1->m_LabelLinewidth, 0, 1, 1, 2, 0, 0 );
	m1->m_Linewidth = new TextView( Rect(), "", "0" );
	m1->m_StyleTable->Attach( m1->m_Linewidth, 1, 2, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create pattern
	m1->m_LabelPattern = new StringView( Rect(), "", "Pattern: " );
	m1->m_StyleTable->Attach( m1->m_LabelPattern, 0, 1, 2, 3, 0, 0 );
	m1->m_Pattern = new DropdownMenu( Rect(), "" );
	m1->m_Pattern->SetSelectionMessage( new Message( Private1::MSG_PATTERN ) );
	m1->m_Pattern->SetReadOnly();
	m1->m_StyleTable->Attach( m1->m_Pattern, 1, 2, 2, 3, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	//////////////////////////////////////////////
	// Create the button table
	m1->m_ButtonTable = new TableView( Rect(), "", 3, 1, true );
	m1->m_ButtonTable->SetColumnSpacings( 5 );	
	AddChild( m1->m_ButtonTable );

	// Create the buttons
	m1->m_Apply = new Button( Rect(), "", "Apply", new Message( Private1::MSG_APPLY ) );
	m1->m_ButtonTable->Attach( m1->m_Apply, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );	

	m1->m_OK = new Button( Rect(), "", "OK", new Message( Private1::MSG_OK ) );
	m1->m_ButtonTable->Attach( m1->m_OK, 1, 2, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );	

	m1->m_Cancel = new Button( Rect(), "", "Cancel", new Message( Private1::MSG_CANCEL ) );
	m1->m_ButtonTable->Attach( m1->m_Cancel, 2, 3, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );	
	
	// Fix layout
	m1->_Layout( GetBounds() );

	// Close dialog with escape
	ShortcutKey s( VK_ESCAPE ); 
	AddShortcut( s, new Message( Private1::MSG_CANCEL ) );

	// Init pointers
	m1->m_CurrentDocument = current;
	m1->_UpdatePattern();
}

WinDiaModifyLayer :: ~WinDiaModifyLayer()
{
	// Remove the mailbox
	RemoveMailbox( "Document" );

	delete m1;
}

void WinDiaModifyLayer :: HandleMessage( os::Message* pcMessage )
{
	switch( pcMessage->GetCode() )	//Get the message code from the message
	{		
		case Private1::MSG_APPLY:
			m1->_Apply();
			break;
		case Private1::MSG_OK:
			if( m1->_Apply() )
				Hide();
			break;
		case Private1::MSG_CANCEL:
			Hide();
			break;

		// Broadcast messages
		case MSG_BROADCAST_NEW_DOCUMENT:
			if( pcMessage->FindPointer( "document", (void**) &m1->m_CurrentDocument ) != EOK )
				m1->m_CurrentDocument = NULL;
			m1->_UpdatePattern();
			m1->_SetLayer( NULL );
			break;
		default:
		{
			Window::HandleMessage( pcMessage );
			break;
		}
	}
}

bool WinDiaModifyLayer ::OkToQuit( void )
{
	Hide();
	return false;
}

void WinDiaModifyLayer :: FrameSized( const os::Point& delta )
{
	Window::FrameSized( delta );
	m1->_Layout( GetBounds() );
}
	
void WinDiaModifyLayer :: Raise( Layer* layer )
{
	if( IsVisible() )
		Hide();

	Show();
	MakeFocus();

	m1->_SetLayer( layer );
}

