// oCADis (C)opyright 2007 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <assert.h>
#include "undo.h"
#include "document.h"
#include "postoffice.h"
#include "messages.h"

using namespace std;
using namespace os;

////////////////////////////////////////////////////////////
//
// A D D   O B J E C T   U N D O   A C T I O N
//
////////////////////////////////////////////////////////////

UndoAddObject :: UndoAddObject( String name, Object* object ) : UndoAction( name )
{
	AddObject( object );
}

UndoAddObject :: UndoAddObject( String name ) : UndoAction( name )
{
}

UndoAddObject :: ~UndoAddObject()
{
}

void UndoAddObject :: Undo( Canvas* canvas, Document* doc )
{
	// As objects have been added we want to erase then if we do a undo
	canvas->Undraw();

	for( uint i = 0 ; i < m_ListOfObjects.size() ; i++ )
	{
		Object* obj = m_ListOfObjects[ i ];		

		obj->Erase( canvas );		
		doc->RemoveObject( obj );		
	}

	_InvalidateCanvas( canvas );
}

void UndoAddObject :: Redo( Canvas* canvas,Document* doc )
{
	// With undo we remove the objects and hence with Redo we add them again
	canvas->Undraw();

	for( uint i = 0 ; i < m_ListOfObjects.size() ; i++ )
	{
		Object* obj = m_ListOfObjects[ i ];

		// Make sure layer is still valid
		Layer* layer = obj->GetLayer();

		if( doc->LayerExists( layer ) )
		{
			layer->AddObject( obj );
			obj->Draw( canvas );
		}
		else
			printf( "Layer does not exists!\n" );
	}

	_InvalidateCanvas( canvas );
}

////////////////////////////////////////////////////////////
//
// E R A S E   O B J E C T S   U N D O   A C T I O N
//
////////////////////////////////////////////////////////////

UndoEraseObjects :: UndoEraseObjects( String name ) : UndoAction( name )
{
}

UndoEraseObjects :: ~UndoEraseObjects()
{
}

void UndoEraseObjects :: Undo( Canvas* canvas,Document* doc )
{
	// If we have erased the objects we add them again with undo
	canvas->Undraw();

	for( uint i = 0 ; i < m_ListOfObjects.size() ; i++ )
	{
		Object* obj = m_ListOfObjects[ i ];

		// Make sure layer is still valid
		Layer* layer = obj->GetLayer();

		if( doc->LayerExists( layer ) )
		{
			layer->AddObject( obj );
			obj->Draw( canvas );
		}
		else
			printf( "Layer does not exists!\n" );
	}

	_InvalidateCanvas( canvas );
}

void UndoEraseObjects :: Redo( Canvas* canvas,Document* doc )
{
	// With undo we add objects and with redo we erase them again
	canvas->Undraw();

	for( uint i = 0 ; i < m_ListOfObjects.size() ; i++ )
	{
		Object* obj = m_ListOfObjects[ i ];		

		obj->Erase( canvas );		
		doc->RemoveObject( obj );		
	}

	_InvalidateCanvas( canvas );
}

////////////////////////////////////////////////////////////
//
// M O D I F Y   O B J E C T S   U N D O   A C T I O N
//
////////////////////////////////////////////////////////////

UndoModifyObjects :: UndoModifyObjects( String name ) : UndoAction( name )
{
}

UndoModifyObjects :: ~UndoModifyObjects()
{
}

void UndoModifyObjects :: AddOriginalObject( Object* object )
{
	AddObject( object );
}

void UndoModifyObjects :: AddTransformedObject( Object* object )
{
	m_ListOfTransformedObjects.push_back( object );
}

void UndoModifyObjects :: Undo( Canvas* canvas, Document* doc )
{
	canvas->Undraw();

	for( uint i = 0 ; i < m_ListOfObjects.size() ; i++ )
	{
		Object* old_obj = m_ListOfTransformedObjects[ i ];
		old_obj->Erase( canvas );
	
		Object* new_obj = m_ListOfObjects[ i ];
		new_obj->Draw( canvas );

		doc->ReplaceObject( old_obj, new_obj );
	}

	_InvalidateCanvas( canvas );
}

void UndoModifyObjects :: Redo( Canvas* canvas, Document* doc )
{
	canvas->Undraw();

	for( uint i = 0 ; i < m_ListOfObjects.size() ; i++ )
	{
		Object* old_obj = m_ListOfObjects[ i ];
		old_obj->Erase( canvas );
	
		Object* new_obj = m_ListOfTransformedObjects[ i ];
		new_obj->Draw( canvas );

		doc->ReplaceObject( old_obj, new_obj );
	}

	_InvalidateCanvas( canvas );
}

////////////////////////////////////////////////////////////
//
// U N D O   A C T I O N
//
////////////////////////////////////////////////////////////

UndoAction :: UndoAction( os::String name )
{
	m_Name = name;
}

UndoAction :: ~UndoAction()
{
	for( uint i = 0 ; i < m_ListOfObjects.size() ; i++ )
	{
		delete m_ListOfObjects[ i ];
	}

	m_ListOfObjects.clear();
}

String UndoAction :: GetName()
{
	return m_Name;
}

void UndoAction :: AddObject( Object* object )
{
	m_ListOfObjects.push_back( object );
}

void UndoAction :: _InvalidateCanvas( Canvas* canvas )
{
	bool first = true;
	Rect ret;

	for( uint i = 0 ; i < m_ListOfObjects.size() ; i++ )
	{
		Object* obj = m_ListOfObjects[ i ];

		// Get the screen area of the objects so that we know later what to redraw on screen
		Rect bound = obj->GetBounds();
		if( first )
		{
			ret = bound;
			first = false;
		}
		else
			ret |= bound;
	}

	canvas->Invalidate( Rect( canvas->WorldToScreen( ret.LeftTop() ), canvas->WorldToScreen( ret.RightBottom() ) ) );
	canvas->Sync();
	canvas->PrepareForXORMode();
}

////////////////////////////////////////////////////////////
//
// U N D O
//
////////////////////////////////////////////////////////////

Undo :: Undo()
{
}

Undo :: ~Undo()
{
	for( uint i = 0 ; i < m_ListOfUndoActions.size() ; i++ )
	{
		delete m_ListOfUndoActions[ i ];
		delete m_ListOfRedoActions[ i ];
	}

	m_ListOfUndoActions.clear();
	m_ListOfRedoActions.clear();
}

void Undo :: AddUndoAction( UndoAction* action )
{
	m_ListOfUndoActions.push_back( action );

	// Send signal to the appwindow to update the menu
	Message* msg = new Message( MSG_BROADCAST_UPDATE_UNDO );
	msg->AddPointer( "undo", (void**) &action );

	Mail( "Undo", msg );
}

String Undo :: GetUndoName() 
{ 
	if( HasUndoObjects() ) 
	{
		UndoAction* action = m_ListOfUndoActions[ m_ListOfUndoActions.size() -1 ];
		return action->GetName();
	}

	return String( "Not available" );
} 

String Undo :: GetRedoName() 
{ 
	if( HasRedoObjects() ) 
	{
		UndoAction* action = m_ListOfRedoActions[ m_ListOfRedoActions.size() -1 ];
		return action->GetName();
	}

	return String( "Not available" );
}

bool Undo :: DoUndo( Canvas* canvas, Document* doc )
{
	assert( canvas != NULL );
	assert( doc != NULL );

	if( !HasUndoObjects() ) 
		return false;

	// Move last undo to redo instead
	UndoAction* action = m_ListOfUndoActions[ m_ListOfUndoActions.size() -1 ];
	m_ListOfRedoActions.push_back( action );

	// and erase it from the undo list
	m_ListOfUndoActions.pop_back();

	// Make it happen!
	action->Undo( canvas, doc );

	return true;
}

bool Undo :: DoRedo( Canvas* canvas, Document* doc )
{
	assert( canvas != NULL );
	assert( doc != NULL );

	if( !HasRedoObjects() ) 
		return false;

	// Move last undo to redo instead
	UndoAction* action = m_ListOfRedoActions[ m_ListOfRedoActions.size() -1 ];
	m_ListOfUndoActions.push_back( action );

	// and erase it from the undo list
	m_ListOfRedoActions.pop_back();

	// Make it happen!
	action->Redo( canvas, doc );

	return true;
}
