// oCADis (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <gui/checkbox.h>
#include "tool_rotate_state_2.h"
#include "common.h"
#include "tool_rotate.h"
#include "toolbox_value.h"
#include "toolbox_action.h"
#include "tableview.h"
#include "cadmath.h"

using namespace os;

///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E
//
///////////////////////////////////////////////////////////////////////////////

class RotateState2 :: Private
{
public:
	Private(){};

	void _Layout( Rect cFrame )
	{
		m_Table->SetFrame( Rect( 0, 0, cFrame.Width(), cFrame.Height() ) );
	}

	void _RotateObjects( Canvas* canvas, Document* doc, double angle )
	{
		Transform transform;
		transform.Translation( -m_Origin );
		transform.Rotation( angle );
		transform.Translation( m_Origin );

		// Create Undo object
		UndoAddObject* undo = new UndoAddObject( "Rotate" );
		doc->AddUndoAction( undo );

		for( int i = 0 ; i < doc->GetNoSelectedObjects() ; i++ )
		{
			Object* new_obj = doc->GetSelectedObject( i )->Duplicate();
			undo->AddObject( new_obj );
			new_obj->SetSelected( false );
			new_obj->DoTransform( transform );
			new_obj->Draw( canvas );
			doc->AddObject( doc->GetSelectedObject( i )->GetLayer(), new_obj ); 
		}

		doc->SetModification();
	}

	void _DeleteOrginalObjectsIfNeeded( Canvas* canvas, Document* doc )
	{
		if( m_KeepOrginal->GetValue().AsBool() )
			return;

		UndoEraseObjects* undo = new UndoEraseObjects( "Erase orginal" );

		// Erase the orginal selected object
		for( int i = 0 ; i < doc->GetNoSelectedObjects() ; i++ )
		{
			Object* obj = doc->GetSelectedObject( i );
			obj->SetSelected( false );
			obj->Erase( canvas );
			undo->AddObject( obj );
			doc->RemoveObject( obj );
		}

		doc->AddUndoAction( undo );
	}

	enum RotateState2Message { MSG_ANGLE, MSG_CANCEL, MSG_MULTICOPIES, MSG_KEEP };
	
	CheckBox* m_MultipleCopies;
	CheckBox* m_KeepOrginal;

	ToolboxValue* m_Angle;
	ToolboxFrame* m_Settings;
	ToolboxAction* m_Action;

	TableView* m_SettingsTable;
	TableView* m_Table;

	Point m_Origin;
};

///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S
//
///////////////////////////////////////////////////////////////////////////////
RotateState2 :: RotateState2( Tool* theTool, ToolboxSideBar* toolbox ) : ToolState( theTool, toolbox )
{
	// Create the private class
	m = new Private();

	// Create the table
	m->m_Table = new TableView( Rect(), "", 1, 2, false );
	m->m_Table->SetRowSpacings( 4 );
	AddChild( m->m_Table );

	// Create the setting table
	m->m_SettingsTable = new TableView( Rect(), "", 1, 2, false );

	// Create settings frame content
	m->m_MultipleCopies = new CheckBox( Rect(), "", "Multiple copies", new Message( Private::MSG_MULTICOPIES ) );
	m->m_SettingsTable->Attach( m->m_MultipleCopies, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );
	m->m_KeepOrginal = new CheckBox( Rect(), "", "Keep original objects", new Message( Private::MSG_KEEP ) );
	m->m_SettingsTable->Attach( m->m_KeepOrginal, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );


	// Create buttons and attach them to table
	m->m_Angle = new ToolboxValue( Rect(), "Angle", "", new Message( Private::MSG_ANGLE ) );
	m->m_Table->Attach( m->m_Angle, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	// Create the setting frame and add it to the table
	m->m_Settings = new ToolboxFrame( Rect( 0, 0, 100, 100 ), "", "Settings" );
	m->m_Settings->AddTableView( m->m_SettingsTable );
	m->m_Table->Attach( m->m_Settings, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), 0 );


	m->m_Action = new ToolboxAction( Rect(), "" );
	m->m_Table->Attach( m->m_Action, 0, 1, 2, 3, ( TABLE_EXPAND | TABLE_FILL ), ( TABLE_EXPAND | TABLE_FILL ) );

	// Add action items
	m->m_Action->AddButton( "Cancel", (Image*) LoadImage( "cancel.png" ), new Message( Private::MSG_CANCEL ) , "Cancel the tool and return to main toolbox " );


	State::SetToolbox( this );
}

RotateState2 :: ~RotateState2()
{
	delete m;
}

void RotateState2 :: Init( Canvas* canvas, Document* doc )
{
	Variant t;
	GetTool()->GetProperty( "origin", t );
	m->m_Origin = t.AsPoint();
}

void RotateState2 :: GotFocus( Canvas* canvas, Document* doc, Point coord )
{
	double angle = CadMath :: GetAngle( m->m_Origin, coord );
	angle = CadMath :: ConvertRadianToUserAngle( angle );

	m->m_Angle->SetValue( angle );
}

void RotateState2 :: LostFocus( Canvas* canvas, Document* doc, Point coord )
{
	canvas->Undraw();
	m->m_Angle->ClearValue();
}

void RotateState2 :: MouseMove( Canvas* canvas, Document* doc, Point coord )
{
	canvas->Undraw();

	double angle = CadMath :: GetAngle( m->m_Origin, coord );
	angle = CadMath :: ConvertRadianToUserAngle( angle );
	m->m_Angle->SetValue( angle );

	Draw( canvas, doc, coord );
}

void RotateState2 :: Draw( Canvas* canvas, Document* doc, os::Point coord )
{	
	canvas->SetDefaultStyle();
	canvas->DrawLine( m->m_Origin, coord, true );

	Transform transform;
	transform.Translation( -m->m_Origin );
	transform.Rotation( CadMath :: GetAngle( m->m_Origin, coord ) );
	transform.Translation( m->m_Origin );

	// Draw objects
	canvas->DrawSelectedObjects( transform );
}

String RotateState2 :: GetStatusText()
{
	return String( "Select rotation angle" );
}

void RotateState2 :: PointSelected( Canvas* canvas, Document* doc, Point coord )
{
	canvas->Undraw();

	m->_RotateObjects( canvas, doc, CadMath :: GetAngle( m->m_Origin, coord ) );

	if( m->m_MultipleCopies->GetValue().AsBool() )
	{
		canvas->PrepareForXORMode();
		GetTool()->SetState( 2 );	
	}
	else
	{
		m->_DeleteOrginalObjectsIfNeeded( canvas, doc );
		SetMainToolbox();
	}
}


void RotateState2 :: SetFrame( const Rect& cRect, bool bNotifyServer )
{
	Toolbox::SetFrame( cRect, bNotifyServer );
	m->_Layout( GetFrame() );
}

void RotateState2 :: AllAttached( void )
{
	Toolbox::AllAttached();

	m->m_Angle->SetTarget( this );
	m->m_Action->SetTarget( this );
}

void RotateState2 :: HandleMessage( Canvas* canvas, Document* doc, Message* pcMessage )
{
	switch( pcMessage->GetCode() )
	{	
	case Private::MSG_CANCEL:	
		doc->DeselectAllObjects( canvas );
		SetMainToolbox();
		break;
	case Private::MSG_ANGLE:
		{
			float p;

			if( pcMessage->FindFloat( "value", &p ) == EOK )
			{
				canvas->Undraw();
				m->_RotateObjects( canvas, doc, CadMath :: ConvertUserAngleToRadian( p ) );

				if( m->m_MultipleCopies->GetValue().AsBool() )
				{
					canvas->PrepareForXORMode();
					GetTool()->SetState( 2 );	
				}
				else
				{
					m->_DeleteOrginalObjectsIfNeeded( canvas, doc );
					SetMainToolbox();
				}
			}
			break;
		}
	case Private::MSG_MULTICOPIES:
		break;
	case Private::MSG_KEEP:
		break;
	}
}


