// oCADis (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "tool_arc_state_7.h"
#include "common.h"
#include "cadmath.h"
#include "object_arc.h"
#include "toolbox_value.h"

using namespace os;

///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E
//
///////////////////////////////////////////////////////////////////////////////

class ArcState7 :: _Private
{
public:
	_Private(){};

	void _Layout( Rect cFrame )
	{
		m_Table->SetFrame( Rect( 0, 0, cFrame.Width(), cFrame.Height() ) );
	}

	enum ArcState7Message { MSG_POINT, MSG_CANCEL };
	
	ToolboxValue* m_Length;
	ToolboxAction* m_Action;

	TableView* m_Table;
};

///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S
//
///////////////////////////////////////////////////////////////////////////////
ArcState7 :: ArcState7( ToolArc* theTool, ToolboxSideBar* toolbox ) : ToolState( theTool, toolbox )
{
	// Create the private class
	m = new _Private();

	// Create the table
	m->m_Table = new TableView( Rect(), "", 1, 2, false );
	m->m_Table->SetRowSpacings( 4 );
	AddChild( m->m_Table );

	// Create buttons and attach them to table
	m->m_Length = new ToolboxValue( Rect(), "Length of coord", "", new Message( _Private::MSG_POINT ) );
	m->m_Table->Attach( m->m_Length, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m->m_Action = new ToolboxAction( Rect(), "" );
	m->m_Table->Attach( m->m_Action, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), ( TABLE_EXPAND | TABLE_FILL ) );

	// Add action items
	m->m_Action->AddButton( "Cancel", (Image*) LoadImage( "cancel.png" ), new Message( _Private::MSG_CANCEL ) , "Cancel the tool and return to main toolbox " );

	State::SetToolbox( this );
}

ArcState7 :: ~ArcState7()
{	
	delete m;
}

void ArcState7 :: Init( Canvas* canvas, Document* doc )
{
	canvas->PrepareForXORMode();
}

void ArcState7 :: GotFocus( Canvas* canvas, Document* doc, Point coord )
{
	coord = doc->MakeSnap( coord );
	m->m_Length->SetValue( CadMath::Distance( coord, GetTool()->GetPropertyAsPoint( "start_point" ) ) );
}

void ArcState7 :: LostFocus( Canvas* canvas, Document* doc, Point coord )
{
	canvas->Undraw();
	m->m_Length->ClearValue();
}

void ArcState7 :: MouseMove( Canvas* canvas, Document* doc, Point coord )
{
	double length = CadMath::Distance( coord, GetTool()->GetPropertyAsPoint( "start_point" ) );

	coord = doc->MakeSnap( coord );

	m->m_Length->SetValue( length );

	// Draw help line
	canvas->SetDefaultStyle();
	canvas->DrawLine( GetTool()->GetPropertyAsPoint( "start_point" ), coord, true );

	// Calculate
	Point center;
	double radius, start_angle, end_angle;

	canvas->SetStyle( doc->GetCurrentStyle() );
	if( CadMath::CalcArcCSL( GetTool()->GetPropertyAsPoint( "center_point" ), GetTool()->GetPropertyAsPoint( "start_point" ), length, center, radius, start_angle, end_angle ) )
		canvas->DrawArc( center, radius, start_angle, end_angle, true );
}

String ArcState7 :: GetStatusText()
{
	return String( "Select length of the coord for the arc" );
}

void ArcState7 :: PointSelected( Canvas* canvas, Document* doc, Point coord )
{
	canvas->Undraw();

	coord = doc->MakeSnap( coord );
	double length = CadMath::Distance( coord, GetTool()->GetPropertyAsPoint( "start_point" ) );

	// Calculate
	Point center;
	double radius, start_angle, end_angle;

	if( CadMath::CalcArcCSL( GetTool()->GetPropertyAsPoint( "center_point" ), GetTool()->GetPropertyAsPoint( "start_point" ), length, center, radius, start_angle, end_angle ) )
	{
		// Create object
		ObjectArc* obj = new ObjectArc( center, radius, start_angle, end_angle );
		doc->AddObject( obj );
		obj->Draw( canvas );

		// Create Undo object
		UndoAddObject* undo = new UndoAddObject( "Arc", obj );
		doc->AddUndoAction( undo );

		// Calc continue arc
		doc->SetContinuePoint( center, coord );

		GetTool()->SetState( 3 );
	}
}

void ArcState7 :: SetFrame( const Rect& cRect, bool bNotifyServer )
{
	Toolbox::SetFrame( cRect, bNotifyServer );
	m->_Layout( cRect );
}

void ArcState7 :: AllAttached( void )
{
	Toolbox::AllAttached();

	m->m_Length->SetTarget( this );
	m->m_Action->SetTarget( this );
}

void ArcState7 :: HandleMessage( Canvas* canvas, Document* doc, Message* pcMessage )
{
	switch( pcMessage->GetCode() )
	{
		case _Private::MSG_CANCEL:
			SetMainToolbox();
			break;
		case _Private::MSG_POINT:
			{
				canvas->Undraw();

				float p;

				if( pcMessage->FindFloat( "value", &p ) == EOK )
				{
					// Calculate
					Point center;
					double radius, start_angle, end_angle;

					if( CadMath::CalcArcCSL( GetTool()->GetPropertyAsPoint( "center_point" ), GetTool()->GetPropertyAsPoint( "start_point" ), p, center, radius, start_angle, end_angle ) )
					{
						// Create object
						ObjectArc* obj = new ObjectArc( center, radius, start_angle, end_angle );
						doc->AddObject( obj );
						obj->Draw( canvas );

						// Create Undo object
						UndoAddObject* undo = new UndoAddObject( "Arc", obj );
						doc->AddUndoAction( undo );

						// Calc continue arc
						Point end( radius * cos( end_angle ) + center.x, radius * sin( end_angle ) - center.y);
						doc->SetContinuePoint( center, end );

						GetTool()->SetState( 3 );
					}
				}
				break;
			}
	}
}

