// oCADis (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "tool_arc_state_15.h"
#include "common.h"
#include "cadmath.h"
#include "object_arc.h"

using namespace os;

///////////////////////////////////////////////////////////////////////////////
//
// P R I V A T E
//
///////////////////////////////////////////////////////////////////////////////

class ArcState15 :: _Private
{
public:
	_Private(){};

	void _Layout( Rect cFrame )
	{
		m_Table->SetFrame( Rect( 0, 0, cFrame.Width(), cFrame.Height() ) );
	}

	enum ArcState15Message { MSG_POINT, MSG_CANCEL };	

	ToolboxPoint* m_EndPoint;
	ToolboxAction* m_Action;

	os::TableView* m_Table;
};

///////////////////////////////////////////////////////////////////////////////
//
// T H E   C L A S S
//
///////////////////////////////////////////////////////////////////////////////

ArcState15 :: ArcState15( ToolArc* theTool, ToolboxSideBar* toolbox ) : State( theTool ), Toolbox( toolbox )
{
	// Create the private class
	m = new _Private();

	// Create the table
	m->m_Table = new TableView( Rect(), "", 1, 2, false );
	m->m_Table->SetRowSpacings( 4 );
	AddChild( m->m_Table );

	// Create buttons and attach them to table
	m->m_EndPoint = new ToolboxPoint( Rect(), "Third point", "X: ", "Y: ", new Message( _Private::MSG_POINT ) );
	m->m_Table->Attach( m->m_EndPoint, 0, 1, 0, 1, ( TABLE_EXPAND | TABLE_FILL ), 0 );

	m->m_Action = new ToolboxAction( Rect(), "" );
	m->m_Table->Attach( m->m_Action, 0, 1, 1, 2, ( TABLE_EXPAND | TABLE_FILL ), ( TABLE_EXPAND | TABLE_FILL ) );

	// Add action items
	m->m_Action->AddButton( "Cancel", (Image*) LoadImage( "circle.png" ), new Message( _Private::MSG_CANCEL ) , "Cancel the tool and return to main toolbox " );

	State::SetToolbox( this );
}

ArcState15 :: ~ArcState15()
{	
	delete m;
}

void ArcState15 :: Init( Canvas* canvas, Document* doc )
{
	canvas->PrepareForXORMode();
}

void ArcState15 :: GotFocus( Canvas* canvas, Document* doc, Point coord )
{
	m->m_EndPoint->SetValue( coord );
}

void ArcState15 :: LostFocus( Canvas* canvas, Document* doc, Point coord )
{
	m->m_EndPoint->ClearValue();
}

void ArcState15 :: MouseMove( Canvas* canvas, Document* doc, Point coord )
{
	m->m_EndPoint->SetValue( coord );

	canvas->SetStyle( doc->GetCurrentStyle() );

	Point p;
	double t;

	if( doc->GetContinuePoint( p, t ) )
	{
		Point p2;
		Point center;
		double radius, start_angle, end_angle;

		CadMath::CalcArc2P( t, p, p2, coord, center, radius, start_angle, end_angle );
		canvas->DrawArc( center, radius, start_angle, end_angle, true );
	}
}

String ArcState15 :: GetStatusText()
{
	return String( "Select third point of the arc" );
}

void ArcState15 :: PointSelected( Canvas* canvas, Document* doc, Point coord )
{
	canvas->Undraw();
	Point p;
	double t;

	// Calculate
	if( doc->GetContinuePoint( p, t ) )
	{
		Point center, p2;
		double radius, start_angle, end_angle;

		CadMath::CalcArc2P( t, p, p2, coord, center, radius, start_angle, end_angle );

		// Create object
		ObjectArc* obj = new ObjectArc( center, radius, start_angle, end_angle );
		doc->AddObject( obj );
		obj->Draw( canvas );

		// Create Undo object
		UndoAddObject* undo = new UndoAddObject( "Create Arc", obj );
		doc->AddUndoAction( undo );

		// Calc continue arc
		doc->SetContinuePoint( center, coord );
		GetTool()->SetState( 3 );	

		doc->SetModification();
	}	
}

void ArcState15 :: SetFrame( const Rect& cRect, bool bNotifyServer )
{
	Toolbox::SetFrame( cRect, bNotifyServer );
	m->_Layout( cRect );
}

void ArcState15 :: AllAttached( void )
{
	Toolbox::AllAttached();

	m->m_EndPoint->SetTarget( this );
	m->m_Action->SetTarget( this );
}

void ArcState15 :: HandleMessage( Canvas* canvas, Document* doc, Message* pcMessage )
{
	switch( pcMessage->GetCode() )
	{
		case _Private::MSG_CANCEL:
			SetMainToolbox();
			break;
		case _Private::MSG_POINT:
			{
				Point p;

				if( pcMessage->FindPoint( "value", &p ) == EOK )
					PointSelected( canvas, doc, p );
		
				break;
			}
	}
}
