// oCADis (C)opyright 2006 Jonas Jarvoll
//
// This is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <cmath>
#include <cstdio>
#include <sstream>
#include "cadmath.h"
#include "main.h"

using namespace os;

void CadMath :: CalcAngleForArc3P( double& sa, double ma, double& ea)
{
	double ta;

	if( ea > sa )
	{
		if( ma < sa )
		{
			ta = ea;
			ea = sa;
			sa = ta;
		}
		else if( ma > ea )
		{
			ta = ea;
			ea = sa;
			sa = ta;
		}
	}
	else if( ea < sa )
	{
		if( ma < sa && ma > ea )
		{
			ta = ea;
			ea = sa;
			sa = ta;
		}
	}
}

bool CadMath :: CalcArc3P( 	Point p1, Point p2, Point p3, Point& center, double& radius, double& start_angle, double& end_angle )
{
	double A, B, C, D, E, F, G;
 
 	A = p2.x - p1.x;
  	B = p2.y - p1.y;
	C = p3.x - p1.x;
	D = p3.y - p1.y;

	E = A * ( p1.x + p2.x ) + B * ( p1.y + p2.y );
	F = C * ( p1.x + p3.x ) + D * ( p1.y + p3.y );
  
	G = 2.0f *  ( A * ( p3.y - p2.y) - B * ( p3.x - p2.x));

	if ( G != 0.0f )
	{
		center.x = ( D * E - B * F ) / G;
		center.y = ( A * F - C * E ) / G;
    	radius = CadMath::Distance( p1, center );

		start_angle = CadMath::GetAngle( center, p1 );
		end_angle = CadMath::GetAngle( center, p3);

		CalcAngleForArc3P( start_angle, CadMath::GetAngle( center, p2 ), end_angle);

		return true;
	}

	return false;
}

void CadMath :: CalcArc2P( double ta, Point p1, Point& p2, Point p3, Point& center, double& radius, double& start_angle, double& end_angle)
{
	double tx,ty;
	double r;
	double d;
	double alfa, sangle, eangle;

	tx = 100.0f * cos( ta + M_PI / 2.0f ) + p1.x;
	ty = -100.0f * sin( ta + M_PI / 2.0f ) + p1.y;   

	d = Distance( Point( tx, ty), p1);

	if( ( ( ( ty - p3.y ) * ( p1.x - tx ) - ( tx - p3.x ) * ( p1.y - ty ) ) / ( d * d ) ) > 0 )
	{
		tx = 100.0f * cos( ta ) + p1.x;
		ty = -100.0f * sin( ta ) + p1.y;   

		alfa = acos( ( ( tx - p1.x ) * ( p3.x - p1.x ) + ( ty - p1.y ) * ( p3.y - p1.y ) ) /
				     ( Distance( Point( tx, ty ), p1 ) * Distance( p3, p1 ) ) );
      
		radius = r = ( 2.0f * Distance( p3, p1 ) * sin( alfa ) ) / sin( M_PI - 2.0f * alfa );
      
		tx = r * cos( ta ) + p1.x;
		ty = -r * sin( ta ) + p1.y; 
      
		center.x = ( p1.x + tx ) / 2.0f;
		center.y = ( p1.y + ty ) / 2.0f;
      
		eangle = end_angle = GetAngle( center, p1);
		sangle = start_angle = GetAngle( center, p3);
	}
	else
	{
		tx = 100.0f * cos( ta + M_PI ) + p1.x;
		ty = -100.0f * sin( ta + M_PI ) + p1.y;   

		alfa = acos( ( ( tx - p1.x ) * (p3.x - p1.x ) + ( ty - p1.y ) * ( p3.y -p1.y) )/
				  ( Distance( Point( tx, ty ), p1) * Distance( p3, p1 ) ) );
      
		radius = r = ( 2.0f * Distance( p3, p1 ) * sin( alfa ) ) / sin( M_PI - 2.0f * alfa );
      
		tx = r * cos( ta + M_PI ) + p1.x;
		ty = -r * sin( ta + M_PI ) + p1.y; 

      	center.x = ( p1.x + tx ) / 2.0f;
		center.y = ( p1.y + ty ) / 2.0f;
      
		sangle = start_angle = GetAngle( center, p1 );
		eangle = end_angle = GetAngle( center, p3 );
	}

	if( sangle < eangle )
		sangle = ( eangle - sangle ) / 2.0f + sangle;
  	else
	{
		sangle = ( 2.0f * M_PI - sangle + eangle ) / 2.0f + sangle;
		if( sangle > 2.0f * M_PI )
			sangle -= 2.0f * M_PI;
	}

	radius /= 2.0f;

	p2.x = center.x + radius * cos( sangle );
	p2.y = center.y - radius * sin( sangle ); 
}

bool CadMath :: CalcArcCSE( Point center1, Point start_point, Point end_point, Point& center, double& radius, double& start_angle, double& end_angle)
{
   	radius = Distance( start_point, center1 );
  
	start_angle = GetAngle( center1, start_point );
	end_angle = GetAngle( center1, end_point );

	center = center1;

	return true;
}

bool CadMath :: CalcArcCSA( Point center1, Point start_point, double angle, os::Point& center, double& radius, double& start_angle, double& end_angle)
{
   	radius = Distance( start_point, center1 );
  
	start_angle = GetAngle( center1, start_point );
	end_angle = start_angle + angle;
	FixAngle( end_angle );

	center = center1;

	return true;
}

bool CadMath :: CalcArcCSL( Point center1, Point start_point, double length, os::Point& center, double& radius, double& start_angle, double& end_angle)
{
	double sangle, eangle;

	radius = Distance( start_point, center1 );
  
	if( length > 2.0f * radius )
		length = 2 * radius;
	if( length < -2.0f * radius)
 		length = -2.0f * radius;

	sangle = GetAngle( center1, start_point );
	eangle = acos( ( 2.0f * radius * radius - length * length ) / ( 2 * radius * radius ) );
	center = center1;
	
  	if( length >= 0.0f )
    {
		start_angle = sangle;
		end_angle = sangle + eangle;
    }
	else
	{
		start_angle = sangle;
		end_angle = sangle-eangle;
	}

	FixAngle( end_angle );

	return true;
}
double CadMath :: Distance( Point p1, Point p2 )
{
	return sqrt( ( p2.x - p1.x ) * ( p2.x - p1.x ) + ( p2.y - p1.y ) * ( p2.y - p1.y ));
}

double CadMath :: GetAngle( Point p1, Point p2 )
{
	double k;

 	k = atan2( p1.y - p2.y, p2.x - p1.x );

	if( k < 0.0f )
    	k = -k;
	else
    	k = 2.0f * M_PI - k;

	if( fabs( k - 2.0f * M_PI ) < 0.0001f )
		k = 0.0f;

	return k;
}

bool CadMath :: BetweenAngles( double sa, double ea, double angle )
{
	if( ea > sa )
		return( ( angle > sa ) && ( angle < ea ) );
	else
		return( ( angle > sa ) || ( angle < ea ) );
}

double CadMath :: max( double v1, double v2 )
{
	return ( ( v1 > v2 ) ? v1 : v2 );
}

double CadMath :: min( double v1, double v2 )
{
	return ( ( v1 < v2 ) ? v1 : v2 );
}

Rect CadMath :: SortRect( Point p1, Point p2 )
{
	Rect ret;

	ret.left = CadMath::min( p1.x, p2.x );
	ret.top = CadMath::min( p1.y, p2.y );
	ret.right = CadMath::max( p1.x, p2.x );
	ret.bottom = CadMath::max( p1.y, p2.y );

	return ret;
}

double CadMath :: Strtod( os::String string )
{
	std::istringstream in( string.c_str() );
	double value;

	in >> value;

	return value;
}

void CadMath :: FixAngle( double& angle )
{
	while( angle < 0.0f ) angle += 2.0f * M_PI;
	while( angle > 2.0f * M_PI ) angle -= 2.0f * M_PI;
}

double CadMath :: ConvertUserAngleToRadian( double value )
{
	double conv_factor = 1.0f;

	switch( GET_PREFERENCES().GetAngleUnit() )
	{
		case 0:		// Radians
			// No conversion needed
			break;
		case 1:		// Degrees ( 360 deegress = 2*PI radians )
			conv_factor = M_PI / 180.0f;
			break;
		case 2:		// Grad ( 400 deegress = 2*PI radians )
			conv_factor = M_PI / 200.0f;
			break;

		default:
			printf( "Error: Unknown angle unit\n" );
			break;
	}

	return value * conv_factor;
}

double CadMath :: ConvertRadianToUserAngle( double value )
{
	double conv_factor = 1.0f;

	switch( GET_PREFERENCES().GetAngleUnit() )
	{
		case 0:		// Radians
			// No conversion needed
			break;
		case 1:		// Degrees ( 360 deegress = 2*PI radians )
			conv_factor = 180.0f / M_PI;
			break;
		case 2:		// Grad ( 400 deegress = 2*PI radians )
			conv_factor = 200.0f / M_PI;
			break;

		default:
			printf( "Error: Unknown angle unit\n" );
			break;
	}

	return value * conv_factor;
}

float CadMath :: ConvertDeg2Rad( float value ) 
{ 
	float ret = value * M_PI / 180.f;

	return  ret;
}

float CadMath :: ConvertRad2Deg( float value ) 
{ 
	float ret = value * 180.0f / M_PI;

	return  ret;
}

Point CadMath :: RotatePoint( double angle, Point origo, Point point )
{
	double tx;

	point = point - origo;

	tx = point.x;

	point.x = tx * cos( angle ) - point.y * sin( angle );
	point.y = tx * sin( angle ) - point.y * cos( angle );

	point = point + origo;

	return point;
}

